// Swarm library. Copyright (C) 1996 Santa Fe Institute.
// This library is distributed without any warranty; without even the
// implied warranty of merchantability or fitness for a particular purpose.
// See file LICENSE for details and terms of copying.

// interface to BLT Graphs. Legends, etc need to be handled more
// gracefully. Bindings to mouse (see BLT demos), autoscrolling, etc
// should be added.

#include <string.h>
#include <stdio.h>
#include <stdlib.h>

#import <swarmgraph/global.h>
#ifdef _TK_
#import <tclObjc.h>
#import <Tk.h>
#endif

#import <swarmgraph/BLTGraph.h>
#import <collections.h>

@implementation BLTGraph

-createEnd {
  [super createEnd];

#ifdef _TK_
  [globalTkInterp eval: "graph %s;", widgetName];
  // lots of features!
  [globalTkInterp eval: "Blt_ZoomStack %s; Blt_Crosshairs %s; Blt_ActiveLegend %s; Blt_ClosestPoint %s",
		  widgetName, widgetName, widgetName, widgetName];
  [self setWidth: 400 Height: 247];		  // golden ratio
#else
  jGraph = [JavaGraph newW: 400 H: 247];
#endif
  elementList = [List create: [self getZone]];
  
  return self;
}

-setRangesXMin: (double) minx Max: (double) maxx
          YMin: (double) miny Max: (double) maxy 
{
#ifdef _TK_
   [globalTkInterp eval: "%s xaxis configure -min %f -max %f; %s yaxis configure -min %f -max %f", widgetName, minx, maxx, widgetName, miny, maxy];
#else
   [jGraph setRangesXMin: minx Max: maxx YMin: miny Max: maxy];
#endif
    return self;
}


-setScaleModeX: (int) xs Y: (int) ys {
#ifdef _TK_
  [globalTkInterp eval: "%s xaxis configure -loose %d; %s yaxis configure -loose %d", widgetName, xs, widgetName, ys];
#else
  fprintf(stderr,"set scale mode xs=%d ys=%d\n", xs, ys);
  [jGraph setScaleModeX: xs Y: ys];
#endif
  return self;
}

-(GraphElement *) createElement {
  GraphElement * newElement;

  newElement = [GraphElement createOwnerGraph: self];
  [elementList addLast: newElement];
  
  return newElement;
}

-destroyElement: (GraphElement *) g {
  [elementList remove: g];
  [g drop];
  return self;
}

-title: (char *) t {
#ifdef _TK_
  [globalTkInterp eval: "%s configure -title \"%s\";", widgetName, t];
#else
  fprintf(stderr,"set title %s\n", t);
  [jGraph title: t];
#endif
  [self setWindowTitle: t];
  return self;
}

-axisLabelsX: (char *) xl Y: (char *) yl {
#ifdef _TK_
  [globalTkInterp eval: "%s xaxis configure -title \"%s\"; %s yaxis configure -title \"%s\";",
		  widgetName, xl, widgetName, yl];
#else
  [jGraph axisLabelsX: xl Y: yl];
#endif
  return self;
}

-pack {
#ifdef _TK_
  [globalTkInterp eval: "pack %s -fill both -expand true;", widgetName];
#else
#endif
  return self;
}

// first destroy all the elements, then ourselves.
-(void) drop {
  while ([elementList getCount] > 0)
    [self destroyElement: [elementList getFirst]];

#ifdef _TK_
  [globalTkInterp eval: "destroy %s", [parent getWidgetName]]; 
#else
#endif
  [super drop];
}

#ifndef _TK_
-update {
  [jGraph update];
  return self;
}

-addElem: (char *) name X: (double)x Y: (double)y {
  [jGraph addElem: name X: x Y: y];
  return self;
}

-setElem: (char *)name Len: (int)n {
  [jGraph setElem: name Len: n];
  return self;
}

-dropElem: (char *)name {
  [jGraph dropElem: name];
  return self;
}

#endif

@end


@implementation GraphElement

-setOwnerGraph: (BLTGraph *) og {
  ownerGraph = og;
  return self;
}

static int vid = 0;

// create ourselves and two vectors.
-createEnd {
  char v[20];
  if (ownerGraph == nil)
    [InvalidCombination raiseEvent: "This element has no owner graph!\n"];
#ifdef _TK_
  name = strdup(tclObjc_objectToName(self));
#else
  sprintf(v, "V%d", vid);
  ++vid;
  name = strdup(v);
#endif
  xData = [BLTVector create: [self getZone]];
  yData = [BLTVector create: [self getZone]];

#ifdef _TK_
  [globalTkInterp eval: "%s element create %s -xdata %s -ydata %s",
		  [ownerGraph getWidgetName], [self getName],
		  [xData getName], [yData getName]];
#else
#endif

  return self;
}

+createOwnerGraph: (BLTGraph *) og {
  return [[[self createBegin: [og getZone]] setOwnerGraph: og] createEnd];
}

-(void) drop {
#ifdef _TK_
  [globalTkInterp eval: "%s element delete %s",
		  [ownerGraph getWidgetName], [self getName]];
#else
  [ownerGraph dropElem: [self getName]];
#endif
  [xData drop];
  [yData drop];
  [super drop];
}

-(char *)getName {
  return name;
}

-(BLTVector *) getXData {
  return xData;
}

-(BLTVector *) getYData {
  return yData;
}

-addX: (double) x Y: (double) y {
  [xData append: x];
  [yData append: y];
#ifndef _TK_
  [ownerGraph addElem: [self getName] X: x Y: y];
  [ownerGraph update];
#endif
  return self;
}

-resetData {
  [xData resetData];
  [yData resetData];
#ifndef _TK_
  [ownerGraph setElem: [self getName] Len: 0];
  [ownerGraph update];
#endif
  return self;
}

-setLabel: (char *) label {
#ifdef _TK_
  [globalTkInterp eval: "%s element configure %s -label \"%s\"",
		  [ownerGraph getWidgetName], name, label];
#else
fprintf(stderr,"elem: set label %s\n", label);
#endif
  return self;
}

-setColor: (char *) color {
#ifdef _TK_
  [globalTkInterp eval: "%s element configure %s -color \"%s\"",
		  [ownerGraph getWidgetName], name, color];
#else
fprintf(stderr,"elem: set color %s\n", color);
#endif
  return self;
}

-setWidth: (unsigned) w {
#ifdef _TK_
  [globalTkInterp eval: "%s element configure %s -linewidth %u",
		  [ownerGraph getWidgetName], name, w];
#else
fprintf(stderr,"elem: set width %d\n", w);
#endif
  return self;
}

-setSymbol: (char *) s {
#ifdef _TK_
  [globalTkInterp eval: "%s element configure %s -symbol %s",
		  [ownerGraph getWidgetName], name, s];
#else
fprintf(stderr,"elem: set symbol %s\n", s);
#endif
  return self;
}

// set the dash style - 0 means solid.
-setDashes: (int) d {
#ifdef _TK_
  [globalTkInterp eval: "%s element configure %s -dashes %d",
		  [ownerGraph getWidgetName], name, d];
#else
fprintf(stderr,"elem: set dashes %d\n", d);
#endif
  return self;
}

@end


// replace this stuff with the BLT C API - should be much faster.
// also optimize append so it doesn't regrow the vector for every single
// value - regrow in chunks.
@implementation BLTVector
-createEnd {
#ifdef _TK_
  name = strdup(tclObjc_objectToName(self));
  [globalTkInterp eval: "vector %s", name];
#else
  name = strdup("V");	// xxx
  val = (double *)malloc(BLTV_CHUNK * sizeof(double));
  max_val = BLTV_CHUNK;
  cur_val = 0;
#endif
  return self;
}

-(char *)getName {
  return name;
}

-(unsigned) getLength {
#ifdef _TK_
  [globalTkInterp eval: "%s length", name];
  return atoi([globalTkInterp result]);
#else
  return cur_val;
#endif
}

-setLength: (unsigned) n {
#ifdef _TK_
  [globalTkInterp eval: "%s length %u", name, n];
#else
  unsigned i;

  fprintf(stderr,"setLength: %d\n", n);

  if (n > max_val) {
    // extend our array 
    unsigned new_size = max_val;
    fprintf(stderr,"setLength: growing from %d\n", max_val);

    while (new_size < n)
      new_size += new_size;

    val = (double *)realloc(val, new_size * sizeof(double));
    max_val = new_size;
  }

  // zero out any newly extended data
  for (i=cur_val; i<n; i++)
    val[i] = 0.0;

  cur_val = n;
#endif
  return self;
}

-append: (double) v {
#ifdef _TK_
  [globalTkInterp eval: "%s append %g", name, v];
#else
  int idx = cur_val;

  fprintf(stderr,"append: %f\n", (float)v);

  // make sure we have room for 1 more
  [self setLength: cur_val+1];

  // set it
  val[idx] = v;
#endif
  return self;
}

// vector ranges - ":" is like the range "5:7", but it assumes you
// mean from min to max.
-resetData {
#ifdef _TK_
  [globalTkInterp eval: "%s delete :", name];
#else
  [self setLength: 0];
#endif
  return self;
}

-delete: (int) n {
#ifdef _TK_
  [globalTkInterp eval: "%s delete %d", name, n];
#else

  fprintf(stderr,"delete: %d\n", n);

  if (n >= 0 && n < (int)cur_val) {
    int i;

    // we have one fewer datum
    cur_val--;

    // shift everyone down
    for (i=n; i < (int)cur_val; i++)
      val[i] = val[i+1];
  }
#endif
  return self;
}

-(void) drop {
#ifdef _TK_
  [globalTkInterp eval: "unset %s", name];
#else
  free(val);
#endif
}

@end
