/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.pipeline;

import fr.gouv.culture.sdx.documentbase.DefaultIDGenerator;
import fr.gouv.culture.sdx.utils.Utilities;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.xml.AbstractXMLPipe;

import java.util.Hashtable;

/**
 * An abstract transformation for basic operations.
 *
 * <p>
 * This abstract class properly handle the logger and the parameters.
 */
public abstract class AbstractTransformation extends AbstractXMLPipe implements Transformation {

    protected static final String PARAMETER_NAME_DISABLE = "disable";
    protected static final String PARAMETER_NAME_ENABLE = "enable";

    /** The parameters for this transformation provided by the pipeline via the setParameters() method.
     *
     *@see this.setParameters()
     */
    protected Parameters parameters;

    /** The parameters for this transformation provided by the Parameteriable interface's
     * parameterize method.
     *
     *@see this.parameterize();
     */
    protected Parameters transParameters;

    /** Avalon logger to write information. */
    protected org.apache.avalon.framework.logger.Logger logger;

    /**The id for this Transformation*/
    protected String id = "";

    /**The Transformation's properties. */
    protected Hashtable props;

    /** The component manager. */
    protected ComponentManager manager;

    /**Indicates wheter the result of this transformation step should be kept*/
    private boolean keepResult = false; //defaulted

    /**Sets Properties for the subclasses.
     *
     * @param props A Properties object containing path information for an instance of an application.
     */
    public void setProperties(Hashtable props) {
        this.props = props;
    }

    /** Set's the logger for the transformation.
     *
     * @param logger    The logger to use for this transformation.
     */
    public void enableLogging(org.apache.avalon.framework.logger.Logger logger) {
        this.logger = logger;
    }

    /**
     * Sets the pipeline's ComponentManager.
     *
     * @param manager  The ComponentManager to use.
     * @throws org.apache.avalon.framework.component.ComponentException
     */
    public void compose(ComponentManager manager) throws ComponentException {
        this.manager = manager;
    }

    public Parameters getParameters() {
        return parameters;
    }

    /** Sets some parameters from a pipeline.
     * @param p The parameters
     */
    public void setParameters(Parameters p) {
        this.parameters = p;

    }

    /**Recycles the contentHandler, lexicalHandler, and xmlConsumer for the transformation*/
    public void recycle() {
        super.recycle();
    }

    /**Loads base configuration common to all transformations.
     *
     * Currently loads the id from the configuraion object.
     * It will eventually load the optional "keep" parameter which
     * is used to specify that the results of the transformation
     * should be retained.
     *
     *@param configuration   The configuration object
     *@throws ConfigurationException
     */
    public void configure(Configuration configuration) throws ConfigurationException {
        this.id = configuration.getAttribute(ATTRIBUTE_ID, new DefaultIDGenerator().generate());
        //getting the "keep" attribute,
        this.keepResult = configuration.getAttributeAsBoolean(Transformation.ATTRIBUTE_KEEP, false);
        if (configuration != null) {
            this.transParameters = Parameters.fromConfiguration(configuration);
            verifyConfigurationParameters(this.transParameters);
        }

    }

    protected void verifyConfigurationParameters(Parameters params) {
        if (params != null) {
            String[] paramNames = params.getNames();
            for (int i = 0; i < paramNames.length; i++) {
                String paramName = paramNames[i];
                if (Utilities.checkString(paramName)) {
                    String paramValue = params.getParameter(paramName, null);
                    if (paramValue == null || paramValue.equals("null"))
                        params.removeParameter(paramName);
                }


            }
        }

    }

    public String getId() {
        return id;
    }

    /**Indicates wheter the results of this transfomation should be retained*/
    public boolean shouldKeepResult() {
        return keepResult;
    }

    /**Returns a boolean indicating whether
     * this transformation is enabled
     * first check the tranformations parameters
     * secondly checking the parameters passed
     * from the containing pipeline
     *
     */
    protected boolean isStepEnabled() {
        boolean enabled = true;
        //verifying the 'disable' parameter from this TRANSFORMATION's parameters
        if (transParameters != null) {
            String enable = transParameters.getParameter(PARAMETER_NAME_ENABLE, "true");
            enabled = Boolean.getBoolean(enable);
            if (enabled) {
                String disable = transParameters.getParameter(PARAMETER_NAME_DISABLE, "false");
                enabled = !Boolean.getBoolean(disable);
            }
        }

        /*verifying the 'disable' parameter from the GOVERNING PIPELINE parameters
        *currently any pipeline parameters passed will
        *be overriding in nature with regards to this
        *parameter and method
        */
        if (parameters != null) {
            if (!enabled) {
                String enableTransId = parameters.getParameter(PARAMETER_NAME_ENABLE, null);
                if (Utilities.checkString(enableTransId)) {
                    if (enableTransId.equals(this.id))
                        enabled = true;
                }
            } else {
                /*if the step is enabled we check for a 'disable' parameter
                */
                String disableTransId = parameters.getParameter(PARAMETER_NAME_DISABLE, null);
                if (Utilities.checkString(disableTransId)) {
                    if (disableTransId.equals(this.id))
                        enabled = false;
                }
            }
        }

        return enabled;
    }
}
