/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.documentbase;

import fr.gouv.culture.oai.OAIHarvester;
import fr.gouv.culture.oai.OAIRepository;
import fr.gouv.culture.sdx.document.BinaryDocument;
import fr.gouv.culture.sdx.document.Document;
import fr.gouv.culture.sdx.document.IndexableDocument;
import fr.gouv.culture.sdx.document.ParsableDocument;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.framework.Framework;
import fr.gouv.culture.sdx.pipeline.GenericPipeline;
import fr.gouv.culture.sdx.pipeline.Pipeline;
import fr.gouv.culture.sdx.repository.Repository;
import fr.gouv.culture.sdx.repository.RepositoryConnection;
import fr.gouv.culture.sdx.utils.Utilities;
import fr.gouv.culture.sdx.utils.constants.Node;
import fr.gouv.culture.sdx.utils.database.DatabaseBacked;
import fr.gouv.culture.sdx.utils.database.DatabaseEntity;
import fr.gouv.culture.sdx.utils.lucene.LuceneDataStore;
import fr.gouv.culture.sdx.application.Application;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.DefaultContext;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.xml.XMLConsumer;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

import java.io.InputStream;
import java.io.OutputStream;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Locale;

/**
 * an abstract document base  class handling most common behaviors
 */
public abstract class AbstractDocumentBase extends DatabaseBacked implements DocumentBase {

    /************
     Class members related to interface DocumentBase
     ************/
    /** The framework's context. */
    protected DefaultContext context = null;
    /** True if this document base is the default one in the application. */
    protected boolean isDefault = false;
    /**The default indexation pipeline*/
    protected Pipeline indexationPipeline = null;
    /** The repositories that are owned by this document base. */
    protected Hashtable repositories = null;
    /** The default repository for this document base. */
    protected Repository defaultRepository = null;
    /**The oai repository wrapper for this document base*/
    protected OAIRepository oaiRepo = null;
    /**The oai harvester wrapper for this document base*/
    protected OAIHarvester oaiHarv = null;


    /************
     Class members NOT related to interface DocumentBase
     ************/

    /** The locale for this document base (found either in the configuration file or by using <code>Locale.getDefault()</code>). */
    protected Locale locale = null;
    /** A pool of connections to the repositories. */
    protected Hashtable repoConnectionPool = null;
    //TODO: use inner class when javac compiler bug is fixed-rbp
    /**Database for document metadata and relations*/
    //protected AbstractDocumentBase.LookupDatabase lookupDatabase = this.new LookupDatabase();
    /** The id generator for documents. */
    IDGenerator idGen = null;


    /**********************************************************************************************
     Internal fields for relationships between documents (kind of metadata required/proposed by SDX)
     //TODO : move them to a framework/application/relationship class ? -pb
     ***********************************************************************************************

     /** Internal value for this field : an attached document relationship. */
    protected final String PROPERTY_NAME_ATTACHED = "attached"; //TODO : use value defined in a DB/Document/Relationship  class  ? -pb
    /** Internal value for this field : an original document relationship. */
    protected final String PROPERTY_NAME_ORIGINAL = "original"; //TODO : use value defined in a DB/Document/Relationship  class  ? -pb
    /** Internal value for this field : a sub-document document relationship. */
    protected final String PROPERTY_NAME_SUB = "sub"; //TODO : use value defined in a DB/Document/Relationship  class  ? -pb
    protected final String PROPERTY_NAME_PARENT = "parent";

    /**********************************************************************
     Attribute names for the configuration element in the configuration file
     **********************************************************************/

    /** The implied attribute stating whether the document base is to be used by default or not. */
    protected final String DBELEM_ATTRIBUTE_DEFAULT = "default"; //TODO : make static ? -pb
    /** The implied attribute stating whether original documents should be stored or not. */
    protected final String DBELEM_ATTRIBUTE_KEEP_ORIGINAL = "keepOriginalDocuments"; //TODO : make static ? -pb



    /*************************************************************************
     Child element names of the configuration element in the configuration file
     *************************************************************************/

    /** The element used to define the indexation pipelines. */
    protected final String ELEMENT_NAME_INDEX = "index"; //TODO : make static ? -pb
    /** The element used to define an indexation pipeline. */
    protected final String ELEMENT_NAME_PIPELINE = "pipeline"; //TODO : make static ? -pb

    /** The element used to define an oai repository. */
    public static final String ELEMENT_NAME_OAI_REPOSITORY = "oai-repository"; //TODO : make static ? -pb
    /** The element used to define an oai harvester. */
    public static final String ELEMENT_NAME_OAI_HARVESTER = "oai-harvester"; //TODO : make static ? -pb


    /************************************************************************
     Property names for document metadata in SDX
     ************************************************************************/

    /** A constant for the repository property in the database entities. */
    protected final static String PROPERTY_NAME_REPO = "repo";
    /** A constant for the path property for attached documents. */
    protected final static String PROPERTY_NAME_MIMETYPE = "mimetype";
    protected final static String PROPERTY_NAME_CONTENT_LENGTH = "content-length";
    /** A constant for the path property for the type of document, Binary, HTML, or XML. */
    protected final String PROPERTY_NAME_DOCTYPE = "sdxDocType";
    /** Internal field name for document ids */
    public static final String INTERNAL_FIELD_NAME_SDXDOCID = LuceneDataStore.ID_FIELD; //TODO : use value defined in a DB/Document class  ? -pb
    /** Internal field name for a *fixed* value field. This field can serve several purposes :
     1) permit a search that is able to retrieve all the documents (no 'SELECT *' in Lucene),
     2) act as a hack that permits to allow creation of binary queries when user wants only to provide an unary one
     */
    public static final String INTERNAL_FIELD_NAME_SDXALL = LuceneDataStore.ALL_FIELD; //Definitely Lucene dependant...
    /** Internal value for this field. */
    public static final String INTERNAL_SDXALL_FIELD_VALUE = LuceneDataStore.ALL_VALUE; //... and so is this one
    /** Internal field name for application id. */
    public static final String INTERNAL_FIELD_NAME_SDXAPPID = "sdxappid"; //TODO : use value defined in a DB/Document class  ? -pb
    /** Internal field name for document base id. */
    public static final String INTERNAL_FIELD_NAME_SDXDBID = "sdxdbid"; //TODO : use value defined in a DB/Document class  ? -pb
    /** Internal field name for repository id. */
    public static final String INTERNAL_FIELD_NAME_SDXREPOID = "sdxrepoid"; //TODO : use value defined in a DB/Document class  ? -pb
    /** Internal field name for document type. */
    public static final String INTERNAL_FIELD_NAME_SDXDOCTYPE = "sdxdoctype"; //TODO : use value defined in a DB/Document class  ? -pb
    /** Internal field name for modification date. */
    public static final String INTERNAL_FIELD_NAME_SDXMODDATE = "sdxmoddate"; //TODO : use value defined in a DB/Document class  ? -pb
    /** Internal field name for document length in bytes. */
    public static final String INTERNAL_FIELD_NAME_SDXCONTENTLENGTH = "sdxcontentlength"; //TODO : use value defined in a DB/Document class  ? -pb
    public static final String INTERNAL_FIELD_NAME_SDX_OAI_DELETED_RECORD = "sdxOaiDeletedRecord";

    /**
     * Contextualize this class.
     *
     * @param context   The context provided by Cocoon.
     * @throws org.apache.avalon.framework.context.ContextException
     */
    public void contextualize(Context context) throws ContextException {
        if (context == null) throw new ContextException("Context provided was null", null);

        this.context = (DefaultContext) context;
    }


    /**
     * Sets the document base id.
     *
     * @param   id      The id (shoud not be <code>null</code>).
     */
    public void setId(String id) throws SDXException {
        if (!Utilities.checkString(id)) {
            String[] args = new String[1];
            args[0] = this.toString();
            throw new SDXException(logger, SDXExceptionCode.ERROR_INVALID_ID, args, null);
        }
        super.setId(id);
    }

    /**
     * Returns the document base id.
     */
    public String getId() {
        return super.getId();
    }

    /**
     * Returns <code>true</code> if this document base is the default in the application.
     */
    public boolean isDefault() {
        return this.isDefault;
    }

    /**Returns the default indexation pipeline used for indexation in this document base. */
    public Pipeline getIndexationPipeline() {
        try {
            return this.indexationPipeline.newInstance();
        } catch (SDXException e) {
            Utilities.logException(logger, e);
            return null;
        }
    }

    //TODO : I can't understand a default result down here. Either the mimetype is actually retrieved
    //either it's not -pb
    /**Supplies the mimeType for the the document if it exists
     * otherwise the binary document default mimetype is returned
     * @param doc	The document.
     */
    public String getMimeType(Document doc) throws SDXException {
        DatabaseEntity ent = null;
        Utilities.checkDocument(this.logger, doc);
        try {
            ent = database.getEntity(doc.getId());
        } catch (SDXException e) {
            return BinaryDocument.DEFAULT_MIMETYPE;
        }
        if (ent == null)
            return BinaryDocument.DEFAULT_MIMETYPE;

        ent.enableLogging(this.logger);
        String mimeType = ent.getProperty(PROPERTY_NAME_MIMETYPE);
        if (!Utilities.checkString(mimeType))
            return BinaryDocument.DEFAULT_MIMETYPE;
        else
            return mimeType;
    }


    protected void configurePipeline(Configuration configuration) throws ConfigurationException {
        //at this point, we should have a <sdx:pipeline> element
        //getting the pipeline configuration
        //TODO : what if configuration.getChild(ELEMENT_NAME_INDEX) returns null ? -pb*
        //A:it wont't based upon the avalon architecture, please see the avalon javadocs:)-rbp
        //My question was about code design : a code profiler doesn't know of Avalon docs
        //but it does know about potential null pointers -pb
        Configuration pipeConf = configuration.getChild(ELEMENT_NAME_INDEX, true).getChild(ELEMENT_NAME_PIPELINE, false);
        //testing if we have something
        /*null possibility is now handled in AbstractPipeline configuration*/
        if (pipeConf == null) {
            //no <sdx:index> or <sdx:pipeline> element
            String[] args = new String[1];
            args[0] = configuration.getLocation();
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_NO_PIPELINE_CONFIG, args, null);
            //we log a warning if no pipeline decl. but continue as piplines can be provided externally
            Utilities.logWarn(logger, sdxE.getMessage(), sdxE);
        }
        //creating the pipeline and assigning the class field
        this.indexationPipeline = new GenericPipeline();
        //setting the logger for the pipeline object
        this.indexationPipeline.enableLogging(this.logger);
        try {
            //passing component manager (actually Cocoon's one) to the pipeline object
            this.indexationPipeline.compose(super._manager);
            //passing the properties object to the repo
            this.indexationPipeline.setProperties(props);
            //configuring the pipeline object from 'application.xconf'
            this.indexationPipeline.configure(pipeConf);
        } catch (ComponentException e) {
            throw new ConfigurationException(e.getMessage(), e.fillInStackTrace());
        }


    }

    /**Does param checks for subclasses
     *
     * @param doc
     * @param consumer
     * @throws SDXException
     */
    public void getDocument(ParsableDocument doc, XMLConsumer consumer) throws SDXException {
        //checking the document
        Utilities.checkDocument(logger, doc);
        //verifying the consumer
        Utilities.checkXmlConsumer(logger, consumer);
        //sub-classes will do the stuff...
    }

    /**Does param checks for subclasses
     *
     * @param doc
     * @param os
     * @throws SDXException
     */
    public void getDocument(Document doc, OutputStream os) throws SDXException {
        //checking the document
        Utilities.checkDocument(logger, doc);
        //verifying the stream
        Utilities.checkOutputStream(logger, os);
        //sub-classes will do the stuff...
    }

    /**Does param checks for subclasses
     *
     * @param doc
     * @return
     * @throws SDXException
     */
    public InputStream getDocument(Document doc) throws SDXException {
        //checking the document
        Utilities.checkDocument(logger, doc);
        return null; //TODO : wow ! -pb
    }

    /** Returns the default repository for this document base.
     * @return The default repository object.
     */
    public Repository getDefaultRepository() {
        return defaultRepository;
    }

    /** Gets a repository in this document base.
     *
     * @param id The repository's id, if <code>null</code> for the default repository is returned
     * @return The repository object
     */
    public Repository getRepository(String id) throws SDXException {
        if (id == null)
            return this.defaultRepository;
        else {
            Repository repo = (Repository) repositories.get(id);
            if (repo == null) {
                String[] args = new String[2];
                args[0] = id;
                args[1] = this.id;
                throw new SDXException(logger, SDXExceptionCode.ERROR_UNKNOWN_REPOSITORY, args, null);
            } else
                return repo;
        }
    }


    public OAIRepository getOAIRepository() {
        return this.oaiRepo;
    }

    public OAIHarvester getOAIHarvester() {
        return this.oaiHarv;
    }


    /**
     * Feeds a SAX content handler with information about this document base.
     *
     * NOT IMPLEMENTED YET!
     *	@param	hdl		A SAX content handler to feed with events.
     */
    public void toSAX(ContentHandler hdl) throws SAXException, ProcessingException {
        // TODO: what will be the XML structure?
        //Export private variables ? Call their toSAX() method when applicable ? -pb
    }

    protected void addOaiDeletedRecord(IndexableDocument doc) throws SDXException {
        if (this.oaiRepo != null)
            this.oaiRepo.addDeletedRecord(doc.getId());
    }

    protected void removeOaiDeletedRecord(IndexableDocument doc) throws SDXException {
        if (this.oaiRepo != null)
            this.oaiRepo.removeDeletedRecord(doc.getId());
    }

    protected synchronized void optimizeDatabase() throws SDXException {
        if (database != null)
            database.optimize();
    }

    /**This method should be called before
     * releasePooledRepositoryConnections();
     * so we have something to optimize.
     */
    protected synchronized void optimizeRepositories() throws SDXException {
        if (this.repoConnectionPool != null && !this.repoConnectionPool.isEmpty()) {
            Enumeration repos = this.repositories.elements();
            if (repos != null) {
                while (repos.hasMoreElements()) {
                    //getting the next repo
                    Repository repo = (Repository) repos.nextElement();
                    //optimizing it
                    /*NOTE:it is the responsibility of the repository
                    *to ensure that unnecessary optimizations are not
                    *executed with the Repository.optimize() method is called
                    */
                    if (repo != null)
                        repo.optimize();
                }
            }
        }
    }

    protected synchronized void releasePooledRepositoryConnections() throws SDXException {

        Enumeration openConnections = null;
        if (this.repoConnectionPool != null)
            openConnections = this.repoConnectionPool.keys();
        if (openConnections != null) {
            while (openConnections.hasMoreElements()) {
                String repoId = (String) openConnections.nextElement();
                RepositoryConnection conn = (RepositoryConnection) this.repoConnectionPool.get(repoId);
                //getting the repository for the connection
                Repository repo = getRepository(repoId);
                //releasing each connection
                if (repo != null && conn != null) {
                    conn.commit();
                    repo.releaseConnection(conn);
                }
                //removing the connection from the pool
                this.repoConnectionPool.remove(repoId);
            }
        }
    }

    //TODOD this method should handle both an repo id and an actual repository object as it may be provided externally, not within this class
    protected synchronized RepositoryConnection getPooledRepositoryConnection(String repoId) throws SDXException {
        if (this.repoConnectionPool == null) this.repoConnectionPool = new Hashtable();

        RepositoryConnection conn = null;

        if (Utilities.checkString(repoId)) {
            if (repoConnectionPool.containsKey(repoId)) {
                conn = (RepositoryConnection) this.repoConnectionPool.get(repoId);
            } else {
                //getting the repository
                Repository repo = (Repository) this.repositories.get(repoId);
                if (repo != null) {
                    //retrieving a connection
                    conn = repo.getConnection();
                    //putting the connection in the pool
                    if (conn != null)
                        this.repoConnectionPool.put(repoId, conn);
                }
            }
        }

        return conn;

    }

    //TODO : possibly move this method to a LuceneLookupIndex class or to LuceneDatase -pb
    protected DatabaseEntity createEntityForDocMetaData(Document doc, Repository repository, String[] parentDocId) throws SDXException {
        //creating a document entry in repository look-up index
        DatabaseEntity docLookupEntry = new DatabaseEntity(doc.getId());
        //TODO : check that no entry already exists for this id ? -pb
        //TODOA: i think that is the responsibility of the calling method.
        docLookupEntry.enableLogging(logger);

        //saving the repository
        if (repository != null)
            docLookupEntry.addProperty(PROPERTY_NAME_REPO, repository.getId());
        //saving the document type
        docLookupEntry.addProperty(PROPERTY_NAME_DOCTYPE, doc.getDocType());
        //saving the mime type
        if (Utilities.checkString(doc.getMimeType()))
            docLookupEntry.addProperty(PROPERTY_NAME_MIMETYPE, doc.getMimeType());

        String length = Integer.toString(doc.getLength());
        if (Utilities.checkString(length))
            docLookupEntry.addProperty(PROPERTY_NAME_CONTENT_LENGTH, length);

        docLookupEntry.addProperties(PROPERTY_NAME_PARENT, parentDocId);

        return docLookupEntry;

    }

    /**
     * Deletes a document and any attached document(s) if not used by any other document(s).
     *
     *@param	doc The document to delete.
     *      * @param   handler A content handler to feed with information.

     * @throws fr.gouv.culture.sdx.exception.SDXException
     */
    public synchronized void delete(Document doc, ContentHandler handler) throws SDXException, SAXException, ProcessingException {
        Utilities.checkDocument(logger, doc);
        Document[] docs = new Document[1];
        docs[0] = doc;
        delete(docs, handler);
    }


    /**This method does  a special delete. Only the document and it's lookup information are
     * deleted any relations will remain
     * @param doc
     * @param handler
     * @throws SDXException
     * @throws SAXException
     * @throws ProcessingException
     */
    protected void deletePhysicalDocument(Document doc, Repository repo, ContentHandler handler) throws SDXException, SAXException, ProcessingException {
        //ensuring we have valid objects
        Utilities.checkDocument(logger, doc);
        RepositoryConnection conn = null;
        DatabaseEntity ent = null;
        if (repo != null) {
            conn = getPooledRepositoryConnection(repo.getId());

            /*Getting the database entity corresponding to this document*/
            ent = super.database.getEntity(doc.getId());
            /*
            * STEP 1 : delete the document's entry from the look-up index
            */
            //TODO: use inner class when javac compiler bug is fixed-rbp
//          super.lookupDatabase.deleteDocMetaDataFromRelationsIndex(doc);
            if (ent != null)
                super.database.delete(ent);

            /*
            * STEP 2 : deleteing the document itself
            */
            if (conn != null)
                repo.delete(doc, conn);
        }

        //sending sax events
        if (handler != null) {
            AttributesImpl atts = new AttributesImpl();
            atts.addAttribute("", Node.Name.ID, Node.Name.ID, Node.Type.CDATA, doc.getId());
            if (repo != null)
                atts.addAttribute("", Node.Name.REPO, Node.Name.REPO, Node.Type.CDATA, repo.getId());
            atts.addAttribute("", Node.Name.BASE, Node.Name.BASE, Node.Type.CDATA, this.getId());
            String appId = Utilities.getStringFromHashtable(Application.APPLICATION_ID, props);
            if (Utilities.checkString(appId))
                atts.addAttribute("", Node.Name.APP, Node.Name.APP, Node.Type.CDATA, appId);
            if (ent != null) {
                String mimeType = ent.getProperty(PROPERTY_NAME_MIMETYPE);
                if (Utilities.checkString(mimeType))
                    atts.addAttribute("", Node.Name.MIMETYPE, Node.Name.MIMETYPE, Node.Type.CDATA, mimeType);
                String length = ent.getProperty(PROPERTY_NAME_CONTENT_LENGTH);
                if (Utilities.checkString(length))
                    atts.addAttribute("", Node.Name.BYTE_LENGTH, Node.Name.BYTE_LENGTH, Node.Type.CDATA, length);
            }
            handler.startElement(Framework.SDXNamespaceURI, Node.Name.DOCUMENT, Framework.SDXNamespacePrefix + ":" + Node.Name.DOCUMENT, atts);
            handler.endElement(Framework.SDXNamespaceURI, Node.Name.DOCUMENT, Framework.SDXNamespacePrefix + ":" + Node.Name.DOCUMENT);
        }

    }


}
