/*
SDX: Documentary System in XML.
Copyright (C) 2000, 2001, 2002  Ministere de la culture et de la communication (France), AJLSM

Ministere de la culture et de la communication,
Mission de la recherche et de la technologie
3 rue de Valois, 75042 Paris Cedex 01 (France)
mrt@culture.fr, michel.bottin@culture.fr

AJLSM, 17, rue Vital Carles, 33000 Bordeaux (France)
sevigny@ajlsm.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the
Free Software Foundation, Inc.
59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
or connect to:
http://www.fsf.org/copyleft/gpl.html
*/
package fr.gouv.culture.sdx.documentbase;

import fr.gouv.culture.sdx.application.Application;
import fr.gouv.culture.sdx.document.*;
import fr.gouv.culture.sdx.exception.SDXException;
import fr.gouv.culture.sdx.exception.SDXExceptionCode;
import fr.gouv.culture.sdx.framework.Framework;
import fr.gouv.culture.sdx.repository.Repository;
import fr.gouv.culture.sdx.repository.RepositoryConnection;
import fr.gouv.culture.sdx.repository.URLRepository;
import fr.gouv.culture.sdx.utils.ConfigurationUtilities;
import fr.gouv.culture.sdx.utils.Utilities;
import fr.gouv.culture.sdx.utils.constants.Node;
import fr.gouv.culture.sdx.utils.database.DatabaseEntity;
import org.apache.avalon.excalibur.xml.Parser;
import org.apache.avalon.framework.component.ComponentException;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.Logger;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.xml.XMLConsumer;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

import java.io.File;
import java.io.InputStream;
import java.io.OutputStream;
import java.text.DateFormat;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;

/**
 * Created by IntelliJ IDEA.
 * User: rpandey
 * Date: 30 dc. 2003
 * Time: 19:30:10
 * To change this template use Options | File Templates.
 */

/**
 * an abstract document base  class handling SDX based configurations
 */
public abstract class SDXDocumentBase extends AbstractDocumentBase {

    /** String representation for a key in the Properties object : document base's directory location. */
    public static final String DOCUMENTBASE_DIR_PATH = "dbDirPath";
    /** Whether the original documents should be stored or not. */
    protected boolean keepOriginalDocuments = true;
    /* String representation for our default pipeline parameter. */
    protected final String DOC_URL = "docUrl";
    /* String representation for our pipeline parameter. */
    protected final String SDX_USER = "sdxUser";
    /* String representation for our pipeline parameter. */
    protected final String SDX_DATE = "sdxDate";
    /* String representation for our pipeline parameter. */
    protected final String SDX_ISO8601_DATE = "sdxISO8601Date";
    /* String representation for a pipeline parameter. */
    protected final String SDX_DATE_MILLISECONDS = "sdxDateMilliseconds";
	protected static final String[] _documentAdditionStatus = {"failure", "ignored", "added", "refreshed", "replaced"};
	protected static final int DOC_ADD_STATUS_FAILURE = 0;
    protected static final int DOC_ADD_STATUS_IGNORED = 1;
	protected static final int DOC_ADD_STATUS_ADDED = 2;
	protected static final int DOC_ADD_STATUS_REFRESHED = 3;
	protected static final int DOC_ADD_STATUS_REPLACED = 4;
	protected static final String SDX_DATABASE_FORMAT = "sdx_database_format";
	 protected static final String SDX_DATABASE_VERSION = "sdx_database_version";
    protected static final String SDX_DATABASE_VERSION_2_3 = "sdx_2.3";
    /*Begin Configurations*/


    public void configure(Configuration configuration) throws ConfigurationException {
        //verifying the object
        Utilities.checkConfiguration(configuration);
        super.configure(configuration);
        loadBaseConfiguration(configuration); //TODO : naming harmonization ? (see below) -pb
        getChildLogger(super.id);
        this.configurePipeline(configuration);
        this.configureIdGenerator(configuration);
        //adding the repositories to this documentbase
        this.configureRepositories(configuration);
        //doing any documentbase specific configuration in subclasses
        configureDocumentBase(configuration);
        //configuring oai components from subclasses
        configureOAIComponents(configuration);
    }


    protected void loadBaseConfiguration(Configuration configuration) throws ConfigurationException {
        String dbDirPath = "";
        //at this point, we have a <sdx:documentBase> configuration object...
        if (configuration.getName().equals(DocumentBase.ELEMENT_NAME_DOCUMENT_BASE)) {
            try {
                //setting the id from the configuration file
                setId(configuration.getAttribute(DocumentBase.ATTRIBUTE_ID, null)); //TODO : discard null default and catch ConfigurationException -pb
            } catch (SDXException sdxE) {
                throw new ConfigurationException(sdxE.getMessage(), sdxE);
            }
            if (!Utilities.checkString(getId())) {
                String[] args = new String[1];
                args[0] = configuration.getLocation();
                SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_INVALID_ID_VALUE, args, null);
                throw new ConfigurationException(sdxE.getMessage(), sdxE);
            }
            //getting the path for the location where this document base will reside
            dbDirPath = Utilities.getStringFromHashtable(Application.DOCUMENTBASES_DIR_PATH, props) + getId() + File.separator;
        }
        //... otherwise we have a <sdx:userDocumentBase> element
        else if (configuration.getName() == Application.ELEMENT_NAME_USER_DOCUMENT_BASE) {
            try {
                //setting the *fixed* id
                setId(Application.USER_DOCUMENT_BASE_ID);
            } catch (SDXException sdxE) {
                throw new ConfigurationException(sdxE.getMessage(), sdxE);
            }
            //getting the path for the location where this document base will reside
            dbDirPath = Utilities.getStringFromHashtable(Application.USERS_DOCUMENTBASE_DIR_PATH, props);
            //... exit if we have nothing
        } else {
            String[] args = new String[1];
            args[0] = configuration.getName();
            //not passing a logger should be logged later up the stack
            SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_UNRECOGNIZED_DOCUMENTBASE_CONFIG, args, null);
            throw new ConfigurationException(sdxE.getMessage(), sdxE);
        }

        //setting property : the path where this document base resides
        props.put(DOCUMENTBASE_DIR_PATH, dbDirPath);

        //setting the default flag, false if not specified in configuration file
        this.isDefault = configuration.getAttributeAsBoolean(DBELEM_ATTRIBUTE_DEFAULT, false);

        this.locale = Utilities.buildLocale(configuration, null);
    }


    protected Configuration[] getRepositoryConfigurationList(Configuration configuration) throws ConfigurationException {
        //at this point, we should have a <sdx:repositories> element containing a list of repositories
        //intializing the array
        //TODO : what if configuration.getChild(Repository.ELEMENT_NAME_REPOSITORIES) returns null ? -pb
        Configuration[] repoConfList = new Configuration[configuration.getChild(Repository.ELEMENT_NAME_REPOSITORIES).getChildren(Repository.ELEMENT_NAME_REPOSITORY).length];
        //getting an array of configuration objects from each of the <sdx:repository> subElements of <sdx:repositories> element
        //TODO : what if configuration.getChild(Repository.ELEMENT_NAME_REPOSITORIES) returns null ? -pb
        //A:it wont't based upon the avalon architecture, please see the avalon javadocs:)-rbp
        //Same answer -pb
        repoConfList = configuration.getChild(Repository.ELEMENT_NAME_REPOSITORIES).getChildren(Repository.ELEMENT_NAME_REPOSITORY);
        //testing if we have something
        if (repoConfList == null || repoConfList.length == 0) {
            String[] args = new String[1];
            //getting the location of the configuration file
            args[0] = configuration.getLocation();
            SDXException sdxE = new SDXException(logger, SDXExceptionCode.ERROR_NO_REPOSITORIES_IN_CONFIG, args, null);
            throw new ConfigurationException(sdxE.getMessage(), sdxE);
        }

        return repoConfList;
    }


    protected void configureRepositories(Configuration configuration) throws ConfigurationException {
        Configuration[] repoConfList = getRepositoryConfigurationList(configuration);
        //ensuring we have a hashtable to populate
        //TODO : create the hastable when object is instanciated ? -pb
        if (super.repositories == null) super.repositories = new Hashtable();
        //a pointer to the first repository
        Repository firstRepository = null;
        //iterating over the list and creating the repositories
        for (int i = 0; i < repoConfList.length; i++) {
            Repository repo = null;
            try {
                /*check for the ref attribute, if it exists, get the repository object and add it to the local hashtable
                *if the attribute doesn't exist create the repo like below, we also need to handle DEFAULTS with refs*/
                String ref = repoConfList[i].getAttribute(Repository.ATTRIBUTE_REF, null);
                if (Utilities.checkString(ref)) {
                    Hashtable appRepos = (Hashtable) props.get(Application.APPLICATION_REPOSITORIES);
                    if (appRepos != null)
                        repo = (Repository) appRepos.get(ref);
                    if (repo == null) {
                        String[] args = new String[1];
                        args[0] = ref;
                        throw new SDXException(logger, SDXExceptionCode.ERROR_LOAD_REFERENCED_REPO, args, null);
                    }
                    //setting the default flag for the referenced repository
                    repo.setIsDefault(repoConfList[i].getAttributeAsBoolean(Repository.ATTRIBUTE_DEFAULT, false));
                } else
                //creating the repository
                    repo = Utilities.createRepository(repoConfList[i], super._manager, props, logger);

                //populating the hashtable
                repositories.put(repo.getId(), repo);
                //setting the default repository if applicable
                //TODO : not compliant with SDX 1 behaviour ; first default one out of many was assumed -pb
                //TODO:A:it is still the default behavior, if the user does not specify a default repositories-rbp
                if (repo.isDefault()) defaultRepository = repo;
                //retaining first repository
                if (i == 0) firstRepository = repo;
            } catch (SDXException e) {
                //we don't want all repositories configurations to fail so we won't throw this farther out
                //the creation of the SDXException should log this message
            } catch (ConfigurationException e) {
                //we don't want all repository configurations to fail so we won't throw this farther out
                Utilities.logException(logger, e);
            }
        }

        //if no default repository was provided, we set the default to the first repository
        if (defaultRepository == null) defaultRepository = firstRepository;

    }


    public void configureOAIComponents(Configuration configuration) throws ConfigurationException {
        configureOAIRepository(configuration);
        configureOAIHarvester(configuration);
    }

    protected void configureIdGenerator(Configuration configuration) throws ConfigurationException {
        this.idGen = ConfigurationUtilities.configureIDGenerator(this.logger, configuration);
        this.idGen.setDatabase(super.database);
    }

    /*End Configurations*/


    /* BEGIN abstract method declarations for specific implementation overides*/
    protected abstract void configureDocumentBase(Configuration configuration) throws ConfigurationException;

    protected abstract void deleteFromSearchIndex(String id) throws SDXException;

    protected abstract void configureOAIRepository(Configuration configuration) throws ConfigurationException;

    protected abstract void configureOAIHarvester(Configuration configuration) throws ConfigurationException;

    protected abstract Object getIndexationDocument(IndexableDocument doc, String storeDocId, String repoId, IndexParameters params) throws SDXException;

    /*TODORemove, possibly remove the "batchIndex" argument below as all indexations are now done in batches
    * this would have a corresponding affect on the LuceneIndex class
    */
    /**Add a document to the underlying search index
     *
     * @param indexationDoc The document object for the specific search implementation (lucene, or other)
     * @param batchIndex    parameter to indicate wheter a batch indexation pass is taking place so that optimization is done at the end of the batch
     * @throws SDXException
     */
    protected abstract void addToSearchIndex(Object indexationDoc, boolean batchIndex) throws SDXException;

    protected abstract void compactSearchIndex() throws SDXException;

    /* END abstract method declarations for specific implementation overides*/


    /** Gets a SDX document as SAX events.
     *
     * @param doc A ParsableDocument, ie XMLDocument or HTMLDocument.
     * @param consumer A SAX content handler to feed with events.
     * <p>
     * The wrapped contentHandler for including events within an XSP page contentHandler should be created using
     * <CODE>IncludeXMLConsumer stripper = new IncludeXMLConsumer(xspContentHandler);</CODE>
     * @throws SDXException
     */
    public void getDocument(ParsableDocument doc, XMLConsumer consumer) throws SDXException {
        //ensuring we have valid objects
        super.getDocument(doc, consumer);
        Repository repo = null;
        RepositoryConnection conn = null;

        try {
            //getting the repository in which the document resides
            repo = getRepositoryForDocument(doc);
            //getting a connection for the repository
            if (repo != null)
                conn = repo.getConnection();
            //then call toSAX from the repository
            if (repo != null && conn != null)
                repo.toSAX(doc, consumer, conn);
        } finally {
            //always close the connection
            if (repo != null && conn != null) repo.releaseConnection(conn);
        }
    }

    /** Gets a SDX document as SAX events.
     *
     * @param doc           A Document, ie XMLDocument or HTMLDocument, we discern the type from the document lookup.
     * @param consumer      A SAX content handler to feed with events.
     * @param docTypeKnown   If the type of the ParsableDocument desired is unknown, ie. XML or HTML,
     *                      allows users to build one or the other type of document and still retrieve a document
     * <p>
     * The wrapped contentHandler for including events within an XSP page contentHandler should be created using
     * <CODE>IncludeXMLConsumer stripper = new IncludeXMLConsumer(xspContentHandler);</CODE>
     * @throws SDXException
     */
    public void getDocument(ParsableDocument doc, XMLConsumer consumer, boolean docTypeKnown) throws SDXException {
        //ensuring we have valid objects
        super.getDocument(doc, consumer);
        if (!docTypeKnown) {
            //looking-up for the document in repository look-up index
            DatabaseEntity ent = database.getEntity(doc.getId());
            if (ent == null) {
                String[] args = new String[2];
                args[0] = doc.getId();
                args[1] = this.getId();
                throw new SDXException(logger, SDXExceptionCode.ERROR_NO_DOC_EXISTS_DOCBASE, args, null);
            }
            ent.enableLogging(this.logger);
            String doctype = ent.getProperty(PROPERTY_NAME_DOCTYPE);
            if (Utilities.checkString(doctype)) {
                if (doctype.equalsIgnoreCase(Document.DOCTYPE_HTML))
                    doc = new HTMLDocument(doc.getId());
                else //if (doctype.equalsIgnoreCase(Document.DOCTYPE_XML) || doctype.equalsIgnoreCase(Document.DOCTYPE_USER) || doctype.equalsIgnoreCase(Document.DOCTYPE_GROUP))
                    doc = new XMLDocument(doc.getId());
                //TODO:maybe in the future we need to add more checks for other types of parsable documents
                doc.enableLogging(this.logger);
            }

        }
        //now we know have a good doctype so we call our original method
        this.getDocument(doc, consumer);

    }

    /** Supplies the provided output stream with the requested document
     * @param doc	The document.
     * @param os	The output stream.
     * @throws fr.gouv.culture.sdx.exception.SDXException
     */
    public void getDocument(Document doc, OutputStream os) throws SDXException {
        //ensuring we have valid objects
        super.getDocument(doc, os);
        Repository repo = null;
        RepositoryConnection conn = null;
        try {
            //getting the repository in which the document resides
            repo = getRepositoryForDocument(doc);
            //getting a connection for the repository
            if (repo != null)
                conn = repo.getConnection();
            //then call toSAX from the repository
            if (repo != null && conn != null)
                repo.get(doc, os, conn);
        } finally {
            //always close the connection
            if (repo != null && conn != null) repo.releaseConnection(conn);
        }
    }

    /** Provides the requested SDX document as an InputStream
     * @param doc	The document.
     * @return	An input stream.
     * @throws fr.gouv.culture.sdx.exception.SDXException
     */
    public InputStream getDocument(Document doc) throws SDXException {
        //ensuring we have valid objects
        super.getDocument(doc);
        Repository repo = null;
        RepositoryConnection conn = null;
        InputStream is = null;

        try {

            //getting the repository in which the document resides
            repo = getRepositoryForDocument(doc);
            //getting a connection for the repository
            if (repo != null)
                conn = repo.getConnection();
            //getting an InputStream from the repository with a null encoding
            //TODO?: should we support an encoding parameter in the future?-rbp
            if (repo != null && conn != null)
                is = repo.openStream(doc, null, conn);
        } finally {
            //always close the connection
            if (repo != null && conn != null) repo.releaseConnection(conn);
        }
        return is;
    }


    /**This method retrieves a repository id from a document object
     * and then does a lookup for the corresponding repository object.
     *
     * @param doc   Document with desired repository storage id
     * @param defaultRepo Default repo to utilize  if none found
     * @return
     * @throws SDXException
     */
    protected Repository getRepositoryForStorage(Document doc, Repository defaultRepo) throws SDXException {
        Repository repo = null;
        if (doc != null) {
            String repoId = doc.getRepositoryForStorage();
            if (Utilities.checkString(repoId))
                repo = getRepository(repoId);
        }

        if (repo == null) {
            if (defaultRepo != null)
                repo = defaultRepo;
            else
                repo = this.defaultRepository;
        }
        return repo;
    }

    /**Gets the repository object in which a document resides,
     * document should not be null and have a valid id,
     * use Utilities.checkDocument before calling this method*/
    protected Repository getRepositoryForDocument(Document doc) throws SDXException {
        //looking-up for the document in repository look-up index
        DatabaseEntity ent = database.getEntity(doc.getId());
        if (ent == null) {
            String[] args = new String[2];
            args[0] = doc.getId();
            args[1] = this.getId();
            throw new SDXException(logger, SDXExceptionCode.ERROR_NO_DOC_EXISTS_DOCBASE, args, null);
        }

        ent.enableLogging(this.logger);

        Repository repo = null;

        //getting the document's repository
        String repoId = ent.getProperty(PROPERTY_NAME_REPO);
        if (Utilities.checkString(repoId) && repositories != null && repositories.containsKey(repoId))
            repo = (Repository) repositories.get(repoId);

        return repo;
        /* else {
            //we have a serious problem because we somehow have no repositories
            String[] args = new String[1];
            args[0] = doc.getId();
            throw new SDXException(logger, SDXExceptionCode.ERROR_RETRIEVE_REPO_FOR_DOCUMENT, args, null);
        }*/

    }

    /**
     * Deletes a document and any attached document(s) if not used by any other document(s).
     *
     *@param	doc The document to delete.
     * @throws fr.gouv.culture.sdx.exception.SDXException
     */
    protected synchronized void delete(Document doc, boolean isIndexable, boolean isPrimary, boolean isShared, String relation, ContentHandler handler) throws SDXException, SAXException, ProcessingException {
        //ensuring we have valid objects
        Utilities.checkDocument(logger, doc);
        Repository repository = null;
        if (handler != null)
            handler.startElement(Framework.SDXNamespaceURI, Node.Name.DELETION, Framework.SDXNamespacePrefix + ":" + Node.Name.DELETION, new AttributesImpl());
        try {

            //TODO : refactor connection management (in the calling method ? use a pool ?)
            //some set-up for accessing the repository
            repository = getRepositoryForDocument(doc);

            if (isIndexable) deleteIndexableDocumentComponents((IndexableDocument) doc, handler);
            //else {TODO Test this: removing this else{}, this would allow us to delete sub documents
            /*
            * STEP 1 : delete the relations to any parent documents (currently,
            *ONLY attached documents may have multiple owners)
            */
            //TODO: use inner class when javac compiler bug is fixed-rbp
            //            super.lookupDatabase.deleteRelationsToMastersFromDatabase(doc);
            deleteRelationsToMastersFromDatabase(doc);
            //}

            /*
            * STEP 6 : delete the document's entry from the look-up index
            */
            //TODO: use inner class when javac compiler bug is fixed-rbp
//            super.lookupDatabase.deleteDocMetaDataFromRelationsIndex(doc);
            /*Getting the database entity corresponding to this document*/

            /*
            * STEP 7 : deleteing the document itself
            */
            if (repository != null)
                deletePhysicalDocument(doc, repository, handler);

            /*
            * TODO : that should never be the case but it would be safe to destroy all other relations
            * to and from this document, using for example :
            * deleteFromRelationsIndex(doc, null, null, 0, batchDelete);
            */


        } finally {

            if (handler != null) {
                handler.endElement(Framework.SDXNamespaceURI, Node.Name.DELETION, Framework.SDXNamespacePrefix + ":" + Node.Name.DELETION);
			}

        }
    }

    /**Deletes all secondary document and search index components
     * @param doc   The parent document
     */
    protected void deleteIndexableDocumentComponents(IndexableDocument doc, ContentHandler handler) throws SDXException, ProcessingException, SAXException {
        /*Getting the database entity corresponding to this document*/
        DatabaseEntity ent = database.getEntity(doc.getId());
        //if this is null we don't need to do anything because we shouldn't have a document
        if (ent == null) return;

        /*
        * this flag replaces the tests on (doc instanceof IndexableDocument)
        * some instances of IndexableDocument (e.g. TransformedDocuments) are *not* to be indexed)            *
        * TODO : find a better name and improve document -pb
        */

        //casting the document to an indexable document
        IndexableDocument indexableDoc = (IndexableDocument) doc;

        //keeping track of deleted records if oai support is enabled
        addOaiDeletedRecord(indexableDoc);
        /*
        * STEP 1 : deleteing the document's indexation from the search index
        */
        //TODO : test if the deletion is relevant since the document may not be indexed *yet* -pb
        //TODO : consistency : this is now the 4th step in add -pb
        deleteFromSearchIndex(indexableDoc.getId());
        //Warning : given the behaviour of DatabaseEntity.getField, only 1 original document is assumed -pb
        //TODO: use inner class when javac compiler bug is fixed-rbp
        //DatabaseEntity[] originalDocEntry = super.lookupDatabase.getRelated(doc.getId(), RELATION_TYPE_ORIGINAL);
        DatabaseEntity originalDocEntry = null;
        String originalDocId = ent.getProperty(PROPERTY_NAME_ORIGINAL);
        if (Utilities.checkString(originalDocId))
            originalDocEntry = super.database.getEntity(ent.getProperty(PROPERTY_NAME_ORIGINAL));
        /*
        * STEP 2 : deleteing the original document, if relevant
        */
        if (originalDocEntry != null) {
            Document origDoc = new BinaryDocument(); //Sic !
            origDoc.enableLogging(logger);
            origDoc.setId(originalDocEntry.getId());

            //Notice that the original document look-up entry will be deleted as well
            if (handler != null) handler.startElement(Framework.SDXNamespaceURI, PROPERTY_NAME_ORIGINAL, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_ORIGINAL, new AttributesImpl());
            this.delete(origDoc, false, false, false, PROPERTY_NAME_ORIGINAL, handler); //recursive call
            if (handler != null) handler.endElement(Framework.SDXNamespaceURI, PROPERTY_NAME_ORIGINAL, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_ORIGINAL);

            /*TODO: i don't think the below line is necessary any longer since the attached/sub document deletion calls
            deleteRelationsToMastersFromDatabase(Document doc)*/
            //deleteing the relation between the document and the original document
            //deleteFromRelationsIndex(doc, origDoc, RELATION_TYPE_ORIGINAL);
        }

        /*
        * STEP 3 : deleteing the attached documents, if relevant
        */
        //TODO: use inner class when javac compiler bug is fixed-rbp
//                DatabaseEntity[] attachedDocs = super.lookupDatabase.getRelated(doc.getId(), RELATION_TYPE_ATTACHED);
        String[] attachedDocs = ent.getPropertyValues(PROPERTY_NAME_ATTACHED);
        if (attachedDocs != null && attachedDocs.length > 0) {
            if (handler != null) handler.startElement(Framework.SDXNamespaceURI, PROPERTY_NAME_ATTACHED, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_ATTACHED, new AttributesImpl());
            for (int i = 0; i < attachedDocs.length; i++) {
                //getting the attached document
                if (Utilities.checkString(attachedDocs[i])) { //How a null value could be possible ? If so, throw a recoverable exception ? -pb
                    DatabaseEntity l_attachedDocEntity = database.getEntity(attachedDocs[i]);
                    //TODO: use inner class when javac compiler bug is fixed-rbp
					//DatabaseEntity owners[] = super.lookupDatabase.getOwners(l_attachedDocEntity.getId());
                    String[] l_attachedDocOwners = l_attachedDocEntity.getPropertyValues(PROPERTY_NAME_PARENT);
                    //Delete attached document entry if it is owned by our doc and our doc only
                    if (l_attachedDocOwners == null || l_attachedDocOwners.length == 0) {
                        //TODOException : the attached document is not owned ?
                    } else {
                        if (l_attachedDocOwners.length == 1 && l_attachedDocOwners[0] != null && l_attachedDocOwners[0].equals(indexableDoc.getId())) {
                            //Notice that the attached document look-up entry will be deleted as well
                            try {
                                BinaryDocument l_attachedDoc = new BinaryDocument();
                                l_attachedDoc.enableLogging(this.logger);
                                l_attachedDoc.setId(l_attachedDocEntity.getId());
                                this.delete(l_attachedDoc, false, false, false, PROPERTY_NAME_ATTACHED, handler); //recursive call
                            } catch (SDXException e) {
                                if (handler != null) e.toSAX(handler);
                            } catch (SAXException e) {
                                //can only log here as we don't want the entire loop to fail
                                this.logger.error(e.getMessage(), e);
                            } catch (ProcessingException e) {
                                //can only log here as we don't want the entire loop to fail
                                this.logger.error(e.getMessage(), e);
                            }
                        } else if (l_attachedDocOwners.length == 1 && l_attachedDocOwners[0] != null && !l_attachedDocOwners[0].equals(indexableDoc.getId())) {
                            //TODOException : the attached document is owned, but not by our document !
                        } else {
                            //Attached document is owned by more than 1 document : keep it
                            //TODO : check that our document is among the owners
							//remove our doucment from the list of parents/owners
                            database.removeProperty(l_attachedDocEntity.getId(), PROPERTY_NAME_PARENT, indexableDoc.getId());
                        }
                    }
                }
            }
            if (handler != null) handler.endElement(Framework.SDXNamespaceURI, PROPERTY_NAME_ATTACHED, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_ATTACHED);


        }

        /*
        * STEP 4 : deleteing the sub-documents, if relevant
        */
        //TODO: use inner class when javac compiler bug is fixed-rbp
        //DatabaseEntity[] subDocs = super.lookupDatabase.getRelated(doc.getId(), RELATION_TYPE_SUB);
        String[] subDocs = ent.getPropertyValues(PROPERTY_NAME_SUB);
        //if(subDocs != null && subDocs.length > 0) {
		if(subDocs != null && subDocs.length > 0) { //TODO: find a better way to manage the subdocs. At the moment, it search for subdocs in database that just went deleted, so even if subDocs itself is not null, the database will return a null value and launch an exception. -la
            if (handler != null) handler.startElement(Framework.SDXNamespaceURI, PROPERTY_NAME_SUB, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_SUB, new AttributesImpl());
            for (int i = 0; i < subDocs.length; i++) {
                //getting the sub-document
                if (Utilities.checkString(subDocs[i])) { //How a null value could be possible ? If so, throw a recoverable exception ? -pb
                    DatabaseEntity l_subDocEntity = database.getEntity(subDocs[i]);//FIXME: Sure subDocs is not null, but the value returned by the getEntity is. So, are the subdocs already deleted as I can guess with the debug informations, or should we try another way to delete them ? -la
                    //TODO: do something useful here
                    if(l_subDocEntity == null)
                     {
                     	//TODO: SubDoc already deleted in database
                     }
                    else
                     {
                     String[] l_subDocOwners = l_subDocEntity.getPropertyValues(PROPERTY_NAME_PARENT); //FIXME: here is the null pointer exception -la
                      if (l_subDocOwners == null || l_subDocOwners.length == 0) {
                         //TODOException : the sub document is not owned ?
                    } else {
                    	if (l_subDocOwners.length == 1 && l_subDocOwners[0] != null && l_subDocOwners[0].equals(indexableDoc.getId())) {
                    	//Notice that the sub document look-up entry will be deleted as well
              				try {
                                 XMLDocument subDoc = new XMLDocument();
                                 subDoc.enableLogging(this.logger);
                                 subDoc.setId(l_subDocEntity.getId());
                                 this.delete(subDoc, true, false, false, PROPERTY_NAME_SUB, handler); //recursive call
                             } catch (SDXException e) {
                                 if (handler != null) e.toSAX(handler);
                             } catch (SAXException e) {
                                 //can only log here as we don't want the entire loop to fail
                                 Utilities.logError(this.logger, e.getMessage(), e);
                             } catch (ProcessingException e) {
                                 //can only log here as we don't want the entire loop to fail
                             	Utilities.logError(this.logger, e.getMessage(), e);
                             }
              			} else if (l_subDocOwners.length == 1 && l_subDocOwners[0] != null && !l_subDocOwners[0].equals(indexableDoc.getId())) {
                             //TODOException : the sub document is owned, but not by our document !
                         } else {
                             //sub document is owned by more than 1 document : keep it
                             //TODO : check that our document is among the owners
                             //remove our doucment from the list of parents/owners
                             database.removeProperty(l_subDocEntity.getId(), PROPERTY_NAME_PARENT, indexableDoc.getId());
              			}
              		}
                     }                   
                }
            }
            if (handler != null) handler.endElement(Framework.SDXNamespaceURI, PROPERTY_NAME_SUB, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_SUB);

        }
    }

    /**
     * Deletes an array of documents and any attached document(s) if not used by any other document(s).
     *
     * @param	docs	The documents to delete.
     * @param   handler A content handler to feed with information.
     * @throws fr.gouv.culture.sdx.exception.SDXException
     *
     */
    public synchronized void delete(Document[] docs, ContentHandler handler) throws SDXException, SAXException, ProcessingException {

        /* TODORefactor:
        in the future we should implement some logic to lookup the documents and group them by repository,
        get a connection for each repo and then delete using the same connection
         */

        if (docs != null) {
            long start = System.currentTimeMillis();

            //we a batch of documents to delete so we set a true state
            AttributesImpl atts = new AttributesImpl();
            if (handler != null) {
                String appId = Utilities.getStringFromHashtable(Application.APPLICATION_ID, props);
                if (Utilities.checkString(appId))
                    atts.addAttribute("", Node.Name.APP, Node.Name.APP, Node.Type.CDATA, appId);
                atts.addAttribute("", Node.Name.BASE, Node.Name.DB_ID, Node.Type.CDATA, this.getId());
                handler.startElement(Framework.SDXNamespaceURI, Node.Name.DELETIONS, Framework.SDXNamespacePrefix + ":" + Node.Name.DELETIONS, atts);

            }
            //counters
            int deletions = 0;
            int failures = 0;

            try {
                for (int i = 0; i < docs.length; i++) {
                    //if we have the last document we set the flag that the batch delete is finished
                    try {
                        if (docs[i] instanceof IndexableDocument) {
                            boolean isIndexable = true;
                            boolean isPrimary = false;//a sub document could be primary or secondary
                            boolean isShared = true;//a sub document could be shared
                            delete(docs[i], isIndexable, isPrimary, isShared, null, handler);
                       } else if (docs[i] instanceof BinaryDocument) {
                            boolean isIndexable = false;
                            boolean isPrimary = false;//an attached document is only secondary
                            boolean isShared = true;//an attached document could be shared
                            delete(docs[i], isIndexable, isPrimary, isShared, null, handler);
					}
                        deletions++;
                    } catch (SDXException e) {
                        failures++;
                        if (handler != null)
                            e.toSAX(handler);
                        else
                            Utilities.logException(logger, e);
                    }
                }
            } finally {
                if (handler != null) {
                    long finish = System.currentTimeMillis();
                    long elapsed = finish - start;
                    AttributesImpl summaryAtts = new AttributesImpl();
                    summaryAtts.addAttribute("", Node.Name.DELETIONS, Node.Name.DELETIONS, Node.Type.CDATA, Integer.toString(deletions));
                    summaryAtts.addAttribute("", Node.Name.FAILURES, Node.Name.FAILURES, Node.Type.CDATA, Integer.toString(failures));
                    summaryAtts.addAttribute("", Node.Name.DURATION, Node.Name.DURATION, Node.Type.CDATA, String.valueOf(elapsed / 1000));
                    handler.startElement(Framework.SDXNamespaceURI, Node.Name.SUMMARY, Framework.SDXNamespacePrefix + ":" + Node.Name.SUMMARY, summaryAtts);
                    handler.endElement(Framework.SDXNamespaceURI, Node.Name.SUMMARY, Framework.SDXNamespacePrefix + ":" + Node.Name.SUMMARY);

                    handler.endElement(Framework.SDXNamespaceURI, Node.Name.DELETIONS, Framework.SDXNamespacePrefix + ":" + Node.Name.DELETIONS);
                }
                //optimizing any repositories that are not optimized
                this.optimizeRepositories();
                //release any pooled connections
                this.releasePooledRepositoryConnections();
                //optimizing our internal datastructure
                //TODO: use inner class when javac compiler bug is fixed-rbp
//                super.lookupDatabase.optimizeDatabase();
                optimizeDatabase();
            }
        }
    }

    /** Adds a document.
     *
     * @param doc   The document to add.
     * @param repository    The repository where to store the document.
     * @param params	The parameters for this adding action.
     * @param handler	A content handler where to send information about the process (may be null), currently we don't use it, but maybe in the future.
     * TODO : what kind of "informations" ? -pb
     */
    public synchronized void index(IndexableDocument doc, Repository repository, IndexParameters params, ContentHandler handler) throws SDXException, SAXException, ProcessingException {
        IndexableDocument[] docs = new IndexableDocument[1];
        docs[0] = doc;
        index(docs, repository, params, handler);
        /*TODOPerformance: is it really necessary to optimize after every addition to the index, i think so based upon little testing
        *this may be problematic for large document bases?rbp*/
        //when we send one document, the index should only be optimized after the doucument and any subdocuments are added
        //TODO:is this still necessary-rbp
        //this.luceneSearchIndex.optimize();
    }


    protected void rollbackIndexation(IndexableDocument doc, ContentHandler handler) throws SDXException, SAXException, ProcessingException {
        if (doc.getStoreHandler().getDocs().length > 0) {
            delete(doc.getStoreHandler().getDocs(), handler);
            deleteFromSearchIndex(doc.getId());
        }
    }



    //TODO: use inner class when javac compiler bug is fixed-rbp
    //protected class LookupDatabase {

    /**Retrieves all parent documents
     * containing the relationType/docId
     * entry
     *
     * @param relationType  The relation type (sub, attached, original)
     * @param docId The secondary document id
     * @param handler   Then handler to feed with events
     * @return
     * @throws SDXException
     * @throws SAXException
     */
    public String[] getOwners(String relationType, String docId, ContentHandler handler) throws SDXException, SAXException {
        Parameters params = new Parameters();
        params.setParameter(relationType, docId);
        //below line can be used to test database.search() method
        //params.setParameter(MIMETYPE_PROPERTY, "text/xml");
        String[] owners = database.search(params);
        if (handler != null && owners != null && owners.length > 0) {
            AttributesImpl atts = new AttributesImpl();
            atts.addAttribute("", Node.Name.NO, Node.Name.NO, Node.Type.CDATA, Integer.toString(owners.length));
            atts.addAttribute("", Node.Name.RELATION, Node.Name.RELATION, Node.Type.CDATA, relationType);
            atts.addAttribute("", Node.Name.ID, Node.Name.ID, Node.Type.CDATA, docId);
            handler.startElement(Framework.SDXNamespaceURI, Node.Name.OWNERS, Utilities.prefixNodeNameSDX(Node.Name.OWNERS), atts);
            for (int i = 0; i < owners.length; i++) {
                String ownerId = owners[i];
                if (ownerId != null && !"".equals(ownerId)) {
                    AttributesImpl atts2 = new AttributesImpl();
                    atts2.addAttribute("", Node.Name.ID, Node.Name.ID, "CDATA", ownerId);
                    handler.startElement(Framework.SDXNamespaceURI, Node.Name.DOCUMENT, Utilities.prefixNodeNameSDX(Node.Name.DOCUMENT), atts2);
                    handler.endElement(Framework.SDXNamespaceURI, Node.Name.DOCUMENT, Utilities.prefixNodeNameSDX(Node.Name.DOCUMENT));
                }
            }
            handler.endElement(Framework.SDXNamespaceURI, Node.Name.OWNERS, Utilities.prefixNodeNameSDX(Node.Name.OWNERS));

        }

        return owners;
    }

    /**Retrieves the related documents (sub, attached, original)
     * of the parent document for the relationType provided
     *
     * @param docId The parent document id
     * @param relationType  The relation type (sub, attached, original)
     * @param handler The handler to feed with events
     * @return
     * @throws SDXException
     * @throws SAXException
     */
    public String[] getRelated(String docId, String relationType, ContentHandler handler) throws SDXException, SAXException {
        if (!Utilities.checkString(docId)) return null;
         DatabaseEntity doc = database.getEntity(docId);
        if (doc == null) {
            String[] args = new String[2];
            args[0] = docId;
            args[1] = this.getId();
            throw new SDXException(this.logger, SDXExceptionCode.ERROR_NO_DOC_EXISTS_DOCBASE, args, null);
        }
        String[] related = this.database.getPropertyValues(docId, relationType);
        if (handler != null && related != null && related.length > 0) {
            AttributesImpl atts = new AttributesImpl();
            atts.addAttribute("", Node.Name.NO, Node.Name.NO, Node.Type.CDATA, Integer.toString(related.length));
            atts.addAttribute("", Node.Name.RELATION, Node.Name.RELATION, Node.Type.CDATA, relationType);
            atts.addAttribute("", Node.Name.ID, Node.Name.ID, Node.Type.CDATA, docId);
            handler.startElement(Framework.SDXNamespaceURI, relationType, Utilities.prefixNodeNameSDX(relationType), atts);
            for (int i = 0; i < related.length; i++) {
                String relatedId = related[i];
                if (relatedId != null && !"".equals(relatedId)) {
                    AttributesImpl atts2 = new AttributesImpl();
                    atts2.addAttribute("", Node.Name.ID, Node.Name.ID, "CDATA", relatedId);
                    handler.startElement(Framework.SDXNamespaceURI, Node.Name.DOCUMENT, Utilities.prefixNodeNameSDX(Node.Name.DOCUMENT), atts2);
                    handler.endElement(Framework.SDXNamespaceURI, Node.Name.DOCUMENT, Utilities.prefixNodeNameSDX(Node.Name.DOCUMENT));
                }
            }
            handler.endElement(Framework.SDXNamespaceURI, relationType, Utilities.prefixNodeNameSDX(relationType));

        }

        return related;

    }


    //TODO : possibly move this method to a LuceneRelationsIndex class or to LuceneDatase -pb
    protected void deleteRelationsToMastersFromDatabase(Document doc) throws SDXException {
        //deleting properties of database entities which have this document listed as an ATTACHED document
        super.database.removeProperty(PROPERTY_NAME_ATTACHED, doc.getId());
        super.database.removeProperty(PROPERTY_NAME_SUB, doc.getId());

    }

    //TODO: use inner class when javac compiler bug is fixed-rbp
    //}


  
    protected synchronized int handleParameters(Document doc, Repository repo, IndexParameters params, boolean isIndexable, boolean isPrimary, String relation, ContentHandler handler) throws SDXException, SAXException, ProcessingException {
        //checking to see if the doc exists in the repositories via the repository look-up index
        String docId = doc.getId();
        Repository repoToUse = repo;
        Repository repoFromInternalLookup = null;
        boolean documentExistsInParamRepo = false;
        String repoIDFromInternalLookup = database.getPropertyValue(docId, PROPERTY_NAME_REPO);
        repoFromInternalLookup = this.getRepository(repoIDFromInternalLookup);
        if (repoToUse != null)
            documentExistsInParamRepo = repoToUse.exists(docId, getPooledRepositoryConnection((repoToUse.getId())));
        if (!documentExistsInParamRepo && repoFromInternalLookup != null)
            repoToUse = repoFromInternalLookup;


        int status = DOC_ADD_STATUS_ADDED;
        //doc exists... handling using params
        int handleSameId = -1;
        if (isPrimary)
            handleSameId = params.handleSameId();
        else {
            //we have a sub document or an attached document
            if (isIndexable)
                handleSameId = params.handleSubDocumentSameId();
            else
                handleSameId = params.handleAttachedDocumentSameId();
        }
        String repoToUseId = "";
        if (repoToUse != null) repoToUseId = repoToUse.getId();
        switch (handleSameId) {
            case IndexParameters.SAME_ID_ERROR:
                status = DOC_ADD_STATUS_FAILURE;
                String[] args = new String[3];
                args[0] = docId;
                args[1] = this.id;
                args[2] = repoToUseId;
                throw new SDXException(logger, SDXExceptionCode.ERROR_DOC_EXISTS, args, null);
            case IndexParameters.SAME_ID_IGNORE:
                //TODO Logging: see with Martin -- where to put this string?
                Utilities.logInfo(logger, "ignoring document : " + doc.getId());
                status = DOC_ADD_STATUS_IGNORED;
                String[] args2 = new String[3];
                args2[0] = docId;
                args2[1] = this.id;
                args2[2] = repoToUseId;//TODOException: a better message for ignore
                throw new SDXException(logger, SDXExceptionCode.ERROR_DOC_EXISTS, args2, null);//breaking the document addition loop
            case IndexParameters.SAME_ID_REFRESH_SHARED_DOCUMENT:
                if (isIndexable && Utilities.checkString(repoIDFromInternalLookup))
                    deleteIndexableDocumentComponents((IndexableDocument) doc, handler);//deleting the master document components
                if (Utilities.checkString(repoIDFromInternalLookup) || documentExistsInParamRepo)
                    deletePhysicalDocument(doc, repoToUse, handler);//deleting the physical document and it's entry in our lookup database
                status = DOC_ADD_STATUS_REFRESHED;
                break;
            case IndexParameters.SAME_ID_REPLACE:
            default:
                if (Utilities.checkString(repoIDFromInternalLookup)){
                    boolean isShared = !isPrimary;//a secondary document could be shared among primary documents
					delete(doc, isIndexable, isPrimary, isShared, relation, handler);
					status = DOC_ADD_STATUS_REPLACED;
                }
                else if (!isIndexable && !isPrimary && documentExistsInParamRepo) {
                    //a binary document, which at this point exists ONLY (not internally) in the repository passed as a parameter
                    deletePhysicalDocument(doc, repoToUse, handler);
                    status = DOC_ADD_STATUS_ADDED;
                }
                break;
        }

        return status;
    }

    /**Set's the default pipeline parameters and ensures the params have a pipeline
     *
     * @param params The params object provided by the user at indexation time
     */
    protected IndexParameters setBaseParameters(IndexParameters params) {
        //if we get null params, we create default ones
        if (params == null)
            params = new IndexParameters();
        //providing a logger for the params
        params.enableLogging(logger);

        //checking pipeline parameters,
        if (params.getPipelineParams() == null)
            params.setPipelineParams(new Parameters());

        //adding the user, if it exists
        fr.gouv.culture.sdx.user.User tmpUser = params.getUser(); //Quick patch to increase performance -pb
        if (tmpUser != null)
            params.getPipelineParams().setParameter(SDX_USER, tmpUser.getId());
        //if their is not a pipline specified we use our default pipeline from the application configuration
        //TODO?:what if this has already been done outside of our code, we reset them after adding our infos, it think it is ok?-rbp
        /*raises problems when the developer reUses the same index parameters for different document bases, once set after the below code,
        then we have no handle on its validity as the correct pipeline*/
        if (params.getPipeline() == null) params.setPipeline(this.indexationPipeline);

        return params;
    }

    /** Adds some documents.
     *
     * @param docs The documents to add.
     * @param repository The repository where to store the documents. If null is passed, the default repository will be used.
     * @param params The parameters for this adding action.
     * @param handler	A content handler where to send information about the process (may be null)
     * TODO : what kind of "informations" ? -pb
     */
    public synchronized void index(IndexableDocument[] docs, Repository repository, IndexParameters params, ContentHandler handler) throws SDXException, SAXException, ProcessingException {

        /** boolean indicating a batch indexation so the the search
         * index can be optimized only once at the end of the batch,
         * false by default to indicate single document indexation.
         */
        //ensuring we have valid parameters
        params = setBaseParameters(params);

        if (handler != null) handler.startElement(Framework.SDXNamespaceURI, Node.Name.UPLOAD_DOCUMENTS, Framework.SDXNamespacePrefix + ":" + Node.Name.UPLOAD_DOCUMENTS, new AttributesImpl());

        long start = System.currentTimeMillis();
        boolean batchIndex = false;
        int docsAdded = 0;
        int docsFailed = 0;

        //if we have a very big patch we split them into smaller batches
        if (docs != null && docs.length > 0) {
            double batchMax = params.getBatchMax();
            //we reset the batch max if it is a smaller batch
            if (batchMax > docs.length) batchMax = docs.length;
            //we calculate the number of batches needed
            int batches = (int) Math.ceil((docs.length / batchMax));
            //setting the index of the document for indexing in the docs array
            int docIdx = 0;
            //selecting a batch
            for (int i = 0; i < batches; i++) {
                //we have batch of documents to index so we set a true state
                batchIndex = true;
                //using the same connection to the store for each document
                RepositoryConnection conn = null;
                //if no repository is provided, using the default repository
                if (repository == null) {
                    String[] args = new String[1];
                    args[0] = this.id;
                    SDXException sdxE = new SDXException(null, SDXExceptionCode.ERROR_USING_DEFAULT_REPO, args, null);
                    Utilities.logInfo(logger, sdxE.getMessage());
                    //we are using the default because no repository was found
                    repository = defaultRepository;
                }

                //ensuring we have a valid repository id to work with
                if (repository != null && !Utilities.checkString(repository.getId()))
                    throw new SDXException(logger, SDXExceptionCode.ERROR_INVALID_REPO_ID, null, null);
                try {

                    if (repository != null)
                        conn = getPooledRepositoryConnection(repository.getId());

                    //indexing documents upto batchMax with the same connection
                    for (int k = 1; docIdx < docs.length && k <= batchMax; k++) {
                        //if we have the last document we set the flag that the batch index and delete are finished, so we can optimize
                        //if ((k % batchMax) == 0 || (docIdx == docs.length - 1))
                        //    batchIndex = false;

                        try {
                            //indexing the next document
                            index(docs[docIdx], repository, conn, params, handler, batchIndex);
                            docsAdded++;//parent document added
                            docsAdded = docsAdded + docs[docIdx].getStoreHandler().getDocs().length;//any subordinate additions
                        } catch (SDXException e) {
                            docsFailed++;
                            if (handler != null && params.getSendIndexationEvents() >= IndexParameters.SEND_ERRORS_EVENTS) {
                                AttributesImpl failAtts = new AttributesImpl();
                                failAtts.addAttribute("", Node.Name.STATUS, Node.Name.STATUS, Node.Type.CDATA, Node.Value.FAILURE);
                                if (docs[docIdx] != null && docs[docIdx].getId() != null)
                                    failAtts.addAttribute("", Node.Name.ID, Node.Name.ID, Node.Type.CDATA, docs[docIdx].getId());
                                if (repository != null)
                                    failAtts.addAttribute("", Node.Name.REPO, Node.Name.REPO, Node.Type.CDATA, repository.getId());
                                failAtts.addAttribute("", Node.Name.BASE, Node.Name.BASE, Node.Type.CDATA, this.getId());
                                String appId = Utilities.getStringFromHashtable(Application.APPLICATION_ID, props);
                                if (Utilities.checkString(appId))
                                    failAtts.addAttribute("", Node.Name.APP, Node.Name.APP, Node.Type.CDATA, appId);
                                handler.startElement(Framework.SDXNamespaceURI, Node.Name.UPLOAD_DOCUMENT, Framework.SDXNamespacePrefix + ":" + Node.Name.UPLOAD_DOCUMENT, failAtts);
                                e.toSAX(handler);
                                handler.endElement(Framework.SDXNamespaceURI, Node.Name.UPLOAD_DOCUMENT, Framework.SDXNamespacePrefix + ":" + Node.Name.UPLOAD_DOCUMENT);
                            }
                            rollbackIndexation(docs[docIdx], handler);
                            //catch and then continue adding other docs, message should be already logged, but pass some info's to the handler in the future
                        } finally {
                            //incrementing to the next document
                            docIdx++;
                            /*TODO:it may be a good idea to optimize here as we don't know how complex a document may be
                            and it could have many attached documents/subdocuments
                            //optimizing our internal datastructure
                            this.optimizeDatabase();
                            */
                        }

                    }
                    //TODOLogging:add some summary info and info for each document uploaded successfully

                } finally {
                    //merging the batch in memory
                    compactSearchIndex();
                    //optimizing any repositories that are not optimized
                    this.optimizeRepositories();
                    //release any pooled connections
                    this.releasePooledRepositoryConnections();
                    //optimizing our internal datastructure
                    //TODO: use inner class when javac compiler bug is fixed-rbp
//                    super.lookupDatabase.optimizeDatabase();
                    optimizeDatabase();
                }
            }
        }
        long finish = System.currentTimeMillis();
        long elapsed = finish - start;

        if (handler != null && params.getSendIndexationEvents() >= IndexParameters.SEND_STATS_EVENTS) {
            AttributesImpl summaryAtts = new AttributesImpl();
            summaryAtts.addAttribute("", Node.Name.ADDITIONS, Node.Name.ADDITIONS, Node.Type.CDATA, Integer.toString(docsAdded));
            // summaryAtts.addAttribute("", "replacements", "replacements", ConfigurationNode.Type.CDATA, Integer.toString(docsAdded));
            summaryAtts.addAttribute("", Node.Name.FAILURES, Node.Name.FAILURES, Node.Type.CDATA, Integer.toString(docsFailed));
            summaryAtts.addAttribute("", Node.Name.DURATION, Node.Name.DURATION, Node.Type.CDATA, String.valueOf(elapsed / 1000));
            handler.startElement(Framework.SDXNamespaceURI, Node.Name.SUMMARY, Framework.SDXNamespacePrefix + ":" + Node.Name.SUMMARY, summaryAtts);
            handler.endElement(Framework.SDXNamespaceURI, Node.Name.SUMMARY, Framework.SDXNamespacePrefix + ":" + Node.Name.SUMMARY);
        }

        if (handler != null) handler.endElement(Framework.SDXNamespaceURI, Node.Name.UPLOAD_DOCUMENTS, Framework.SDXNamespacePrefix + ":" + Node.Name.UPLOAD_DOCUMENTS);

    }

    /** Adds a document from an array of documents using the same connection.
     *
     * @param doc			The document to add.
     * @param repository	The repository where to store the document.
     * @param conn          The repository's connection, we want to use only one connection for all the documents
     * @param params     The parameters for this adding action.
     * @param handler	A content handler where to send information about the process (may be null), currently we don't use it, but maybe in the future.
     * TODO : what kind of "informations" ? -pb
     */
    protected synchronized void index(IndexableDocument doc, Repository repository, RepositoryConnection conn, IndexParameters params, ContentHandler handler, boolean batchIndex) throws SDXException, SAXException, ProcessingException {
        //ensuring we have a valid document object, not null, but the id can be null so we don't check it like in other cases below
        if (doc == null)
            throw new SDXException(logger, SDXExceptionCode.ERROR_DOC_NULL, null, null);

        if (handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS)
            handler.startElement(Framework.SDXNamespaceURI, Node.Name.UPLOAD_DOCUMENT, Framework.SDXNamespacePrefix + ":" + Node.Name.UPLOAD_DOCUMENT, new AttributesImpl());

        try {
            //we always use our document base's logger during indexation, if another has been provided it will not be used
            doc.enableLogging(this.logger);

            //setting the id generator properties
            doc.setIdGenerator(this.idGen);
            //giving the document access to the handler for event logging
            if (handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS)
                doc.setMessageHandler(handler);

            /*
            * setting the document's url as a pipe parameter
            * we should always have pipelineParams at this point
            * because setBaseParameters has already been called
            */
            java.net.URL tmpURL = doc.getURL(); //Quick patch to increase performance -pb
            Parameters tmpParameters = params.getPipelineParams(); //Quick patch to increase performance -pb
            String tempUrlString = "";
            if (tmpURL != null)
                tempUrlString = tmpURL.toExternalForm();
            tmpParameters.setParameter(DOC_URL, tempUrlString);
            //adding our defaults
            //getting current date info
            Date date = fr.gouv.culture.sdx.utils.Date.getUtcIso8601Date();
            //formmating a humanReadable date
            tmpParameters.setParameter(SDX_DATE, DateFormat.getDateTimeInstance(DateFormat.FULL, DateFormat.FULL).format(date));
            //formatting an iso 8601 date
            //TODORefactor: this date format into our date utilities-rbp
            // params.getPipelineParams().setParameter(SDX_FORMATTED_DATE, new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssz").format(date));
            // FG: to get exact system date, formatted by our date utilities
//        params.getPipelineParams().setParameter(SDX_ISO8601_DATE, fr.gouv.culture.sdx.utils.Date.formatDate(new Date()));
            //updated our date utilities to handle the ISP8601 date type
            tmpParameters.setParameter(SDX_ISO8601_DATE, fr.gouv.culture.sdx.utils.Date.formatUtcISO8601Date(date));
            tmpParameters.setParameter(SDX_DATE_MILLISECONDS, Long.toString(date.getTime()));
            //setting the parameters to the pipeline
            //TODO?:what if this has already been done outside of our code, we reset them after adding our infos,
            //I think it is ok for now, but maybe bad in some complex cases-rbp
            fr.gouv.culture.sdx.pipeline.Pipeline tmpPipeline = params.getPipeline();  //Quick patch to increase performance -pb
            tmpPipeline.setParameters(tmpParameters);
            //indexing document using the pipeline from the index params
            tmpPipeline.setConsumer(doc);
            Parser parser = null;
            try {
                parser = (Parser) super._manager.lookup(Parser.ROLE);
                doc.startIndexing(parser, tmpPipeline);
                //after parsing we want to retrieve any transformed documents
                byte[] tmpBytes = tmpPipeline.getTransformedBytes(); //Quick patch to increase performance -pb
                File tmpFile = tmpPipeline.getTransformedFile(); //Quick patch to increase performance -pb
                if (tmpBytes != null)
                    doc.setTransformedDocument(tmpBytes);
                else if (tmpFile != null)
                    doc.setTransformedDocument(tmpFile);
                //TODO : potential other case here -pb
            } catch (ComponentException e) {
                String[] args = new String[1];
                args[0] = e.getMessage();
                throw new SDXException(logger, SDXExceptionCode.ERROR_ACQUIRE_PARSER, args, e);
            } finally {
                if (parser != null) super._manager.release(parser);
            }

            add(batchIndex, doc, repository, conn, params, true, true, null, null, handler, null);
        } finally {
            if (handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS)
                handler.endElement(Framework.SDXNamespaceURI, Node.Name.UPLOAD_DOCUMENT, Framework.SDXNamespacePrefix + ":" + Node.Name.UPLOAD_DOCUMENT);
        }


    }

    //SDX search architecture/Lucene specific
    /**
     *
     * @param originalDoc The document to add
     * @param repository The repository to use
     * @param conn  The repository connection to use
     * @param params The document addition parameters
     * @param handler   The content handler to feed with events
     * @param isIndexable   is the document indexable
     * @param isPrimary     is the document a primary or secondary document
     * @param batchIndex    is this a batch index
     * @throws SDXException
     * @throws SAXException
     * @throws ProcessingException
     */
   
    protected void add(boolean batchIndex, Document originalDoc, Repository repository, RepositoryConnection conn, IndexParameters params, boolean isIndexable, boolean isPrimary, String relation, String parentId, ContentHandler handler, String attachedDocId) throws SDXException, SAXException, ProcessingException {
        //ensuring we have a valid document
        Utilities.checkDocument(logger, originalDoc);


        //boolean handleConnection = false;
        Repository storeRepo = repository;
        RepositoryConnection storeRepoConn = conn;
        int status = -1;
        String repoId = "";
        //setting the status based upon parameters handled
        status = handleParameters(originalDoc, repository, params, isIndexable, isPrimary, relation, handler);
		if (status < DOC_ADD_STATUS_ADDED) return;//if the document exists and shouldnt be added exit
        if (handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS)
            handler.startElement(Framework.SDXNamespaceURI, Node.Name.ADDITION, Framework.SDXNamespacePrefix + ":" + Node.Name.ADDITION, new AttributesImpl());
        try {

            //some set-up for accessing the repository
            /*
             * TODO : check if the *provided* connection if fit for the repository ?
             * the opportunity to default the repository could result in providing an irrelevant connection -pb
             */
            if (storeRepo == null) storeRepo = defaultRepository;

            //determining the storage repository
            storeRepo = getRepositoryForStorage(originalDoc, storeRepo);

            if (storeRepo != null)
                repoId = storeRepo.getId();

            //if the provided repo is not the desired store repo, or the connection provided was null we handle the connection
            if (storeRepo != null && (storeRepoConn == null || storeRepo != repository)) {
                storeRepoConn = getPooledRepositoryConnection(storeRepo.getId());
            }


            Document docToStore = originalDoc; //By default, we store the original document
			String docToStoreId = "";
            boolean storeTransformed = false;

            DatabaseEntity docLookupEntry = null;
            //TODO: use inner class when javac compiler bug is fixed-rbp
            //super.lookupDatabase.addDocMetaDataToDatabase(docToStore, repository);
            String[] parentDocIds = new String[1];
            parentDocIds[0] = parentId;//add the new parent id
            if (status == DOC_ADD_STATUS_REFRESHED) {
                DatabaseEntity entityToRefresh = super.database.getEntity(docToStore.getId());
                if (entityToRefresh != null){
                String[] oldParents = entityToRefresh.getPropertyValues(PROPERTY_NAME_PARENT);
                parentDocIds = new String[oldParents.length + 1];
                for (int i=0 ;i < oldParents.length; i++)
                    parentDocIds[i+1] = oldParents[i];
                }
            }
            docLookupEntry = createEntityForDocMetaData(docToStore, storeRepo, parentDocIds);

            /*if we are in an attached doc we give the good*/
            
            if (attachedDocId != null){
            	docToStoreId = attachedDocId;
            }
            
            /*
            * this flag replaces the tests on (doc instanceof IndexableDocument)
            * some instances of IndexableDocument (e.g. TransformedDocuments) are *not* to be indexed)            *
            * TODO : find a better name and improve document -pb
            */
            if (isIndexable) {

                //Code safe here : null is assumed in case we can't get a transformed document
                IndexableDocument transformedDoc = ((IndexableDocument) originalDoc).getTransformedDocument();

                //TODO : make the code below more generic by providing a isWrtiable() method to repositories
                //only writable repositories can store original documents -pb

                //determining which document to store
                if (transformedDoc != null && !(storeRepo instanceof URLRepository) && Utilities.checkString(transformedDoc.getId())) {
                    //modifying the original object's id and filename
                    originalDoc.setId("o_" + originalDoc.getId());
                    // FIXME: I put the following comment because it doesn't work when getPreferredFilename returns null
//                    originalDoc.setPreferredFilename("o_" + originalDoc.getPreferredFilename());
                    docToStore = transformedDoc; //store the transformed
                    storeTransformed = true;
                }

				docToStoreId = docToStore.getId();



                /*
                * STEP 1 : adding the original document, if relevant
                */
                if (storeTransformed && (params.isSaveOriginalDocument() && this.keepOriginalDocuments)) {
                    if (handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS)
                        handler.startElement(Framework.SDXNamespaceURI, PROPERTY_NAME_ORIGINAL, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_ORIGINAL, new AttributesImpl());
                    try {
                        //Notice that the original document look-up entry will be added as well
                        this.add(batchIndex, originalDoc, repository, conn, params, false, false, PROPERTY_NAME_ORIGINAL, null, handler, null); //Recursive call -pb
                        ((IndexableDocument) docToStore).getStoreHandler().addDoc(originalDoc);
                        //adding the relation between the document and the original document
                        //TODO: use inner class when javac compiler bug is fixed-rbp
//                    super.lookupDatabase.addDocRelationToDatabase(docToStore, originalDoc, RELATION_TYPE_ORIGINAL, null);
                        docLookupEntry.addProperty(PROPERTY_NAME_ORIGINAL, originalDoc.getId());
                    } finally {
                        if (handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS)
                            handler.endElement(Framework.SDXNamespaceURI, PROPERTY_NAME_ORIGINAL, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_ORIGINAL);
                    }


                }

                /*
                * STEP 2 : adding the attached documents, if relevant
                */
                Enumeration attachedDocs = ((IndexableDocument) originalDoc).getAttachedDocuments();
                if (attachedDocs != null) {
                    int attDocsAdded = 0;
                    boolean startElemSent = false;
                    try {
                        while (attachedDocs.hasMoreElements()) {
                            if (attDocsAdded == 0 && handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS) {
                                handler.startElement(Framework.SDXNamespaceURI, PROPERTY_NAME_ATTACHED, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_ATTACHED, new AttributesImpl());
                                startElemSent = true;
                            }
                            //getting the attached document
                            BinaryDocument attachedDoc = (BinaryDocument) attachedDocs.nextElement();
                            //Notice that the attached document look-up entry will be added as well
                            this.add(batchIndex, attachedDoc, repository, conn, params, false, false, PROPERTY_NAME_ATTACHED, docToStoreId, handler, attachedDoc.getId()); //Recursive call -pb
                            attDocsAdded++;
                            ((IndexableDocument) docToStore).getStoreHandler().addDoc(attachedDoc);
                            //adding the relation between the document and the attached document
                            //TODO : consider *ordered* attached documents ; it could be useful -pb
                            //TODO: use inner class when javac compiler bug is fixed-rbp
                            //super.lookupDatabase.addDocRelationToDatabase(docToStore, attachedDoc, RELATION_TYPE_ATTACHED, null);
                            docLookupEntry.addProperty(PROPERTY_NAME_ATTACHED, attachedDoc.getId());
                            /*TODO: pierrick please see the handleParams() method and see if we can add
                            *any other proper behavior there
                            * WARNING :
                            * This implementation assumes that the attached document
                            * is the same than the one that is already stored.
                            * One can imagine that, for a *same* id, other properties
                            * (mimetype, data) could be different.
                            * We should be able to use IndexParamters and/or attachedDocLookupEntry
                            * to determine the different possible behaviours :
                            * always replace
                            * never replace (ignore)
                            * create new instance (in which case, the id management won't be easy)
                            * replace if "master document" (document to be implemented)
                            * ...
                            *
                            * For now, we do nothing here since the attached document
                            * is already present, probably (certainly ;-) owned by other documents -pb
                            */
                        }
                    } finally {
                        if (startElemSent && handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS) {
                            AttributesImpl summaryAtts = new AttributesImpl();
                            summaryAtts.addAttribute("", Node.Name.ADDITIONS, Node.Name.ADDITIONS, Node.Type.CDATA, Integer.toString(attDocsAdded));
                            handler.startElement(Framework.SDXNamespaceURI, Node.Name.SUMMARY, Framework.SDXNamespacePrefix + ":" + Node.Name.SUMMARY, summaryAtts);
                            handler.endElement(Framework.SDXNamespaceURI, Node.Name.SUMMARY, Framework.SDXNamespacePrefix + ":" + Node.Name.SUMMARY);
                            handler.endElement(Framework.SDXNamespaceURI, PROPERTY_NAME_ATTACHED, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_ATTACHED);
                        }
                    }
                }


                /*
                * STEP 3 : adding the sub-documents, if relevant
                */
                Enumeration subDocs = ((IndexableDocument) originalDoc).getSubDocuments();
                if (subDocs != null) {
                    int subDocsAdded = -1;
                    boolean startElemSent = false;
                    try {
                        while (subDocs.hasMoreElements()) {
                            if (subDocsAdded == 0 && handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS) {
                                handler.startElement(Framework.SDXNamespaceURI, PROPERTY_NAME_SUB, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_SUB, new AttributesImpl());
                                startElemSent = true;
                            }
                            //getting the sub-document
                            IndexableDocument subDoc = (IndexableDocument) subDocs.nextElement();
                            //Notice that the sub-document look-up entry will be added as well
                            this.add(batchIndex, subDoc, repository, conn, params, true, false, PROPERTY_NAME_SUB, docToStoreId, handler, null); //Recursive call -pb
                            subDocsAdded++;
                            ((IndexableDocument) docToStore).getStoreHandler().addDoc(subDoc);
                            //adding the relation between the document and the attached document
                            //TODO: use inner class when javac compiler bug is fixed-rbp
//                        super.lookupDatabase.addDocRelationToDatabase(docToStore, subDoc, RELATION_TYPE_SUB, null);
                            docLookupEntry.addProperty(PROPERTY_NAME_SUB, subDoc.getId());
                            /*TODO: pierrick please see the handleParams() method and see if we can add
                            *any other proper behavior there
                            * WARNING :
                            * This implementation assumes that the attached document
                            * is the same than the one that is already stored.
                            * One can imagine that, for a *same* id, other properties
                            * (mimetype, data) could be different.
                            * We should be able to use IndexParamters and/or attachedDocLookupEntry
                            * to determine the different possible behaviours :
                            * always replace
                            * never replace (ignore)
                            * create new instance (in which case, the id management won't be easy)
                            * replace if "master document" (document to be implemented)
                            * ...
                            *
                            * For now, we do nothing here since the attached document
                            * is already present, probably (certainly ;-) owned by other documents -pb
                            */

                        }
                    } finally {
                        if (startElemSent && handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS) {
                            AttributesImpl summaryAtts = new AttributesImpl();
                            summaryAtts.addAttribute("", Node.Name.ADDITIONS, Node.Name.ADDITIONS, Node.Type.CDATA, Integer.toString(subDocsAdded));
                            handler.startElement(Framework.SDXNamespaceURI, Node.Name.SUMMARY, Framework.SDXNamespacePrefix + ":" + Node.Name.SUMMARY, summaryAtts);
                            handler.endElement(Framework.SDXNamespaceURI, Node.Name.SUMMARY, Framework.SDXNamespacePrefix + ":" + Node.Name.SUMMARY);
                            handler.endElement(Framework.SDXNamespaceURI, PROPERTY_NAME_SUB, Framework.SDXNamespacePrefix + ":" + PROPERTY_NAME_SUB);
                        }
                    }
                }

                /*
               * STEP 4 : adding the document's indexation to the search index
               */
                /*possibly differ this step -pb
                * yes i differed it as we don't want to store anything
                * about the indexation document until its parts are
                * sucessfully added
                * Well... I mean the possibilty to delay the actual indexing of documents.
                * Currently, this method does two things :
                * 1) delegate document management to the relevant repositories
                * 2) create an indexation document and store it in the (Lucene) searchIndex
                * As point 2 is somewhat ressource consuming, it would be nice to process it
                * when server gets idle -pb
                */

                /*
                * Resolution of :
                * a) document id
                * b) attached documents info
                * c) search index document
                */
                Object indexationDoc = getIndexationDocument((IndexableDocument) originalDoc, docToStoreId, repoId, params);
                //TOTOpb : understand how ce could have gotten a valid id before this :-)
                //ANSWER: auto id generation or id set by user during pipeline (ex. XsltTransformatin)-rbp

                if (indexationDoc != null) {
                    addToSearchIndex(indexationDoc, batchIndex);
                    removeOaiDeletedRecord((IndexableDocument) originalDoc);
                }
            }


            /*
            * STEP 5 : adding the document itself
            */
            if (storeRepo != null)
                storeRepo.add(docToStore, storeRepoConn);


            /*
            * STEP 6 : adding the document's entry to the look-up index
            */
            super.database.save(docLookupEntry);


            /*
            * STEP 7: sending our info events
            */

            if (handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS) {
                AttributesImpl addAtts = new AttributesImpl();
                addAtts.addAttribute("", Node.Name.STATUS, Node.Name.STATUS, Node.Type.CDATA, String.valueOf(_documentAdditionStatus[status]));
                addAtts.addAttribute("", Node.Name.ID, Node.Name.ID, Node.Type.CDATA, docToStoreId);
                addAtts.addAttribute("", Node.Name.REPO, Node.Name.REPO, Node.Type.CDATA, repoId);
                addAtts.addAttribute("", Node.Name.BASE, Node.Name.BASE, Node.Type.CDATA, this.getId());
                String appId = Utilities.getStringFromHashtable(Application.APPLICATION_ID, props);
                if (Utilities.checkString(appId))
                    addAtts.addAttribute("", Node.Name.APP, Node.Name.APP, Node.Type.CDATA, appId);
                String mimeType = docToStore.getMimeType();
                if (Utilities.checkString(mimeType))
                    addAtts.addAttribute("", Node.Name.MIMETYPE, Node.Name.MIMETYPE, Node.Type.CDATA, mimeType);

                String length = Integer.toString(docToStore.getLength());
                if (Utilities.checkString(length))
                    addAtts.addAttribute("", Node.Name.BYTE_LENGTH, Node.Name.BYTE_LENGTH, Node.Type.CDATA, length);

                handler.startElement(Framework.SDXNamespaceURI, Node.Name.DOCUMENT, Framework.SDXNamespacePrefix + ":" + Node.Name.DOCUMENT, addAtts);
                handler.endElement(Framework.SDXNamespaceURI, Node.Name.DOCUMENT, Framework.SDXNamespacePrefix + ":" + Node.Name.DOCUMENT);
            }
        } finally {
            if (handler != null && params.getSendIndexationEvents() == IndexParameters.SEND_ALL_EVENTS)
                handler.endElement(Framework.SDXNamespaceURI, Node.Name.ADDITION, Framework.SDXNamespacePrefix + ":" + Node.Name.ADDITION);
            //cleaning-up
            /*we are going to release all from the pool at the end of the batch
            if (handleConnection) {
                if (useStoreRepo)
                    releasePooledRepositoryConnection(storeRepo.getId());
                else
                    releasePooledRepositoryConnection(repository.getId());
            } */
        }
    }

	public void init() throws SDXException {
        super.init();
        addParentRelationsToChildren();
    }

	private void addParentRelationsToChildren() throws SDXException {
        final Logger l_finalLogger = this.logger;
        new Thread() {
            public void run() {
                try {
                    DatabaseEntity dbFormat = database.getEntity(SDX_DATABASE_FORMAT);
                    if (dbFormat != null && dbFormat.containsValue(SDX_DATABASE_VERSION, SDX_DATABASE_VERSION_2_3))
                        return;//the database is already compliant we exit
                    //searching for all entities that reference a child "attached" document
                    Parameters l_attachedDocParams = new Parameters();
                    l_attachedDocParams.setParameter(PROPERTY_NAME_ATTACHED, database.getWildcardSearchToken());
                    String[] l_attachedDocParents = database.search(l_attachedDocParams);
                    for (int i = 0; i < l_attachedDocParents.length; i++) {
                    	DatabaseEntity l_entity = database.getEntity(l_attachedDocParents[i]);
                    	String l_entityId = l_entity.getId();
                    	String[] l_attachedDocsIds = l_entity.getPropertyValues(PROPERTY_NAME_ATTACHED);
                        for (int j = 0; j < l_attachedDocsIds.length; j++) {
                            String l_attachedDocId = l_attachedDocsIds[j];
                            DatabaseEntity l_attachedDoc = database.getEntity(l_attachedDocId);
                            if (l_attachedDoc != null && l_attachedDoc.containsValue(PROPERTY_NAME_PARENT, l_entityId))
                                database.addProperty(l_attachedDocId, PROPERTY_NAME_PARENT, l_entityId);
                        }
                    }
                    //searching for all entities that reference a child "sub" document
                    Parameters l_subDocParams = new Parameters();
                    l_attachedDocParams.setParameter(PROPERTY_NAME_SUB, database.getWildcardSearchToken());
                    String[] l_subDocParents = database.search(l_subDocParams);

                    for (int i = 0; i < l_subDocParents.length; i++) {
                        DatabaseEntity l_entity = database.getEntity(l_subDocParents[i]);
                        String l_entityId = l_entity.getId();
                        String[] l_subDocsIds = l_entity.getPropertyValues(PROPERTY_NAME_SUB);
                        for (int j = 0; j < l_subDocsIds.length; j++) {
                            String l_subDocId = l_subDocsIds[j];
                            DatabaseEntity l_subDoc = database.getEntity(l_subDocId);
                            if (l_subDoc != null && l_subDoc.containsValue(PROPERTY_NAME_PARENT, l_entityId))
                                database.addProperty(l_subDocId, PROPERTY_NAME_PARENT, l_entityId);
                        }
                    }

                    //adding an entry to indicate that the database is updated to version 2.3
                    DatabaseEntity dbe = new DatabaseEntity(SDX_DATABASE_FORMAT);
                    dbe.addProperty(SDX_DATABASE_VERSION, SDX_DATABASE_VERSION_2_3);
                    database.save(dbe);


                } catch (SDXException e) {
                    l_finalLogger.error(e.getMessage(), e);
                }
            }
        }.start();
    }
	
}
