---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-TEXT.ADB
--  Description : GUI Widget Text entry
--
--  Copyright (C) 2001, Martin C. Carlisle <carlislem@acm.org>
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Ada.Text_IO;
with Gui_Enum;
with File_Helpers;
with textentry_dialog_window;
with mcc.Common_Dialogs;
with Generate_Helpers;
with mcc.Gui.Widget;          use type mcc.Gui.Widget.Widget_Pointer;
with mcc.text_entry_types;
with state;

package body gui.Widget.Text is
   -- reads information from file into Text_Entry,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Text_Entry) is
   begin
      Read_Widget (GUI_Widget (Widget));
      if File_Helpers.Token_Index > File_Helpers.N_Tokens then
         return;
      end if;
      Widget.Data_Item         := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      Widget.Data_Type         := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      Widget.Base_Type         := mcc.text_entry_types.Default_Base_Type;
      -- test for EOL for backward compatibility
      if File_Helpers.Token_Index > File_Helpers.N_Tokens then
         return;
      end if;
      declare
         Base_Type : constant String :=
            File_Helpers.Token (File_Helpers.Token_Index).all;
      begin
         Widget.Base_Type := mcc.text_entry_types.Base_Type'Value (Base_Type);
      exception
         when others =>
            Ada.Text_IO.Put_Line
              ("Syntax error at base type: " &
               Base_Type &
               ", using default (" &
               mcc.text_entry_types.Base_Type'Image
                  (mcc.text_entry_types.Default_Base_Type) &
               ")");
      end;
   end Read_Widget;

   -- Writes information to file from Text_Entry
   procedure Write_Widget (Widget : in Text_Entry) is
   begin
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.TextEntry) & " ");
      Write_Widget (GUI_Widget (Widget));
      if Widget.Data_Item /= null and Widget.Data_Type /= null then
         File_Helpers.Put (" """);
         File_Helpers.Put_String (Widget.Data_Item.all);
         File_Helpers.Put (""" """);
         File_Helpers.Put_String (Widget.Data_Type.all);
         File_Helpers.Put
           (""" " & mcc.text_entry_types.Base_Type'Image (Widget.Base_Type));
      end if;
      File_Helpers.P;
   end Write_Widget;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Text_Entry) is
   begin
      Generate_Helpers.Generate_With ("Mcc.Gui.Widget.Text_Entry");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Text_Entry) is
   begin
      File_Helpers.P
        (Widget.Name.all &
         " : aliased Mcc.Gui.Widget.Text_Entry.Text_Entry;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99
   procedure Generate_Widget_Creation
     (Widget      : in Text_Entry;
      Window_Name : in String)
   is
      use File_Helpers;
   begin
      P ("Mcc.Gui.Widget.Text_Entry.Create");
      P ("  (Obj    => " & Widget.Name.all & ",");
      P ("   Parent => " & Window_Name & ",");
      P ("   X      => " & mcc.Img (Widget.x) & ",");
      P ("   Y      => " & mcc.Img (Widget.y) & ",");
      P ("   Width  => " & mcc.Img (Widget.Width) & ",");
      P ("   Height => " & mcc.Img (Widget.Height) & ");");
   end Generate_Widget_Creation;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Text_Entry;
      Container : in out mcc.Gui.Container.Container'Class)
   is
   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.Gui.Widget.Text_Entry.Text_Entry;
      end if;

      mcc.Gui.Widget.Text_Entry.Create
        (Obj    =>
           mcc.Gui.Widget.Text_Entry.Text_Entry (Widget.The_Widget.all),
         Parent => Container,
         X      => Widget.x,
         Y      => Widget.y,
         Width  => Widget.Width,
         Height => Widget.Height);
      Display_Widget (GUI_Widget (Widget), Container);
   exception
      when others =>
         mcc.Common_Dialogs.Ok_Box ("Can't display: " & Widget.Name.all);
   end Display_Widget;

   procedure Set_Properties (Widget : in out Text_Entry) is
   begin
      -- fill in by copying to global
      mcc.text_entry_types.Default_Base_Type := Widget.Base_Type;

      textentry_dialog_window.Generate_and_Fill_Window;

      -- where are the entries?
      Widget.Properties         :=
        textentry_dialog_window.textentry_dialog_window'Access;
      Widget.Name_Entry         := textentry_dialog_window.entry1'Access;
      Widget.X_Entry            := textentry_dialog_window.entry2'Access;
      Widget.Y_Entry            := textentry_dialog_window.entry3'Access;
      Widget.Width_Entry        := textentry_dialog_window.entry4'Access;
      Widget.Height_Entry       := textentry_dialog_window.entry5'Access;
      Widget.Data_Item_Entry    := textentry_dialog_window.entry6'Access;
      Widget.Base_Type_Dropdown :=
        textentry_dialog_window.base_dropdown'Access;
      Widget.Data_Type_Entry    := textentry_dialog_window.entry7'Access;
      Set_Properties (GUI_Widget (Widget));

      if state.Get_Current_Window.Novice_Mode then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Item_Entry.all,
            Text => "disabled for novice");
         mcc.Gui.Widget.Text_Entry.Disable (Widget.Data_Item_Entry.all);
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Data_Type_Entry.all,
            Text => "disabled for novice");
         mcc.Gui.Widget.Text_Entry.Disable (Widget.Data_Type_Entry.all);
      else
         if Widget.Data_Item /= null then
            mcc.Gui.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Data_Item_Entry.all,
               Text => Widget.Data_Item.all);
         end if;

         if Widget.Data_Type /= null then
            mcc.Gui.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Data_Type_Entry.all,
               Text => Widget.Data_Type.all);
         end if;
      end if;
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Text_Entry) is
      Ok : Boolean;
   begin
      Apply_Properties (GUI_Widget (Widget));
      declare
         Data_Item : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Data_Item_Entry.all);
      begin
         if Data_Item /= "disabled for novice" then
            Widget.Data_Item := new String'(Data_Item);
         end if;
      end;
      declare
         Data_Type     : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Data_Type_Entry.all);
         Base_Type_Pos : Integer;
      begin
         if Data_Type /= "disabled for novice" then
            Widget.Data_Type := new String'(Data_Type);
            Base_Type_Pos    :=
               mcc.Gui.Widget.Dropdown.Get_Selected
                 (Widget.Base_Type_Dropdown.all);
            if Base_Type_Pos > 0 then
               Widget.Base_Type :=
                  mcc.text_entry_types.Base_Type'Val (Base_Type_Pos - 1);
            end if;
         end if;
      end;

      -- copy out of global into record
      textentry_dialog_window.Read_Window
        (Success         => Ok,
         Highlight_Error => False,
         Beep_On_Error   => False);
      Widget.Base_Type := mcc.text_entry_types.Default_Base_Type;
   end Apply_Properties;

   procedure Check_Properties
     (Widget : in out Text_Entry;
      Ok     : out Boolean)
   is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);

      if Ok then
         textentry_dialog_window.Read_Window
           (Success         => Ok,
            Highlight_Error => True,
            Beep_On_Error   => True);
      else
         mcc.Gui.Bell;
      end if;
   end Check_Properties;

   procedure Generate_Action_Context_Clause (Widget : in Text_Entry) is
      use type mcc.text_entry_types.Base_Type;
   begin
      if Widget.Data_Item /= null and then Widget.Data_Item.all /= "" then
         Generate_Helpers.Generate_With_For_FQN (Widget.Data_Item.all);
      end if;
      if Widget.Data_Type /= null and then Widget.Data_Type.all /= "" then
         Generate_Helpers.Generate_With_For_FQN (Widget.Data_Type.all);
      end if;
      if Widget.Base_Type in mcc.text_entry_types.Float_Types then
         Generate_Helpers.Generate_With ("Mcc.Text_Entry_Types");
      elsif Widget.Base_Type = mcc.text_entry_types.Unsigned then
         Generate_Helpers.Generate_With ("Interfaces");
      elsif Widget.Base_Type = mcc.text_entry_types.Unbounded_String
        and then Widget.Data_Item /= null
        and then Widget.Data_Item.all /= ""
      then
         Generate_Helpers.Generate_With ("Ada.Strings.Unbounded");
      end if;
   end Generate_Action_Context_Clause;

   function Has_Anything_To_Fill_Or_Read
     (Widget : in Text_Entry)
      return   Boolean
   is
   begin
      return Widget.Data_Item /= null and then Widget.Data_Item.all /= "";
   end Has_Anything_To_Fill_Or_Read;

   procedure Generate_Fill_Action (Widget : in Text_Entry)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      -- don't bother if there's no data item
      if Widget.Data_Item = null or else Widget.Data_Item.all = "" then
         return;
      end if;

      case Widget.Base_Type is
         when mcc.text_entry_types.Unbounded_String =>
            P ("Mcc.Gui.Widget.Text_Entry.Set_Text");
            P ("  (Obj    => " & Widget.Name.all & ",");
            P ("   Text   => Ada.Strings.Unbounded.To_String (" &
               Widget.Data_Item.all & "));");
         when mcc.text_entry_types.Integer =>
            P ("Mcc.Gui.Widget.Text_Entry.Set_Text");
            P ("  (Obj    => " & Widget.Name.all & ",");
            P ("   Text   => Integer (" & Widget.Data_Item.all & "));");
         when mcc.text_entry_types.Unsigned =>
            P ("Mcc.Gui.Widget.Text_Entry.Set_Text");
            P ("  (Obj    => " & Widget.Name.all & ",");
            P ("   Text   => Interfaces.Unsigned_32 (" &
               Widget.Data_Item.all & "));");
         when mcc.text_entry_types.Float_1 |
              mcc.text_entry_types.Float_2 |
              mcc.text_entry_types.Float_3 |
              mcc.text_entry_types.Float_E =>
            P ("Mcc.Gui.Widget.Text_Entry.Set_Text");
            P ("  (Obj    => " & Widget.Name.all & ",");
            P ("   Text   => Mcc.Text_Entry_Types.Image (Float (" &
               Widget.Data_Item.all &
               "), Mcc.Text_Entry_Types." &
               mcc.text_entry_types.Base_Type'Image (Widget.Base_Type) &
               "));");
         when mcc.text_entry_types.Enumeration =>
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= ""
            then
               P ("Mcc.Gui.Widget.Text_Entry.Set_Text");
               P ("  (Obj    => " & Widget.Name.all & ",");
               P ("   Text   => " & Widget.Data_Type.all &
                  "'Image (" & Widget.Data_Item.all & "));");
            end if;
      end case;
   end Generate_Fill_Action;

   procedure Generate_Read_Action (Widget : in Text_Entry) is
      use type mcc.text_entry_types.Base_Type;
      use File_Helpers;

      procedure Print_Declare_Block is
      begin
         P ("declare");
         P ("   X : String := Mcc.Gui.Widget.Text_Entry.Get_Text");
         P ("                    (Obj => " & Widget.Name.all & ");");
         P ("begin");
      end Print_Declare_Block;

      procedure Print_Exception_Block is
      begin
         P ("exception when Constraint_Error =>");
         P ("   if Highlight_Error then");
         P ("      Mcc.Gui.Widget.Text_Entry.Highlight(" &
            Widget.Name.all & ");");
         P ("   end if;");
         P ("   raise;");
         P ("end;");
      end Print_Exception_Block;
   begin
      -- don't bother if there's no data item
      if Widget.Data_Item = null or else Widget.Data_Item.all = "" then
         return;
      end if;

      case Widget.Base_Type is
         when mcc.text_entry_types.Unbounded_String =>
            Print_Declare_Block;
            P ("   " & Widget.Data_Item.all & " := " &
               "Ada.Strings.Unbounded.To_Unbounded_String(X);");
            P ("end;");
         when mcc.text_entry_types.Integer =>
            P ("declare");
            P ("   X : Integer := Mcc.Gui.Widget.Text_Entry.Get_Text");
            P ("                    (" & Widget.Name.all & ");");
            P ("begin");
            P ("   " & Widget.Data_Item.all & " := ", Newline => False);
            -- only add a cast if we need to
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= ""
            then
               P ("Integer (X);", Indent => False);
            else
               P ("X;", Indent => False);
            end if;
            Print_Exception_Block;
         when mcc.text_entry_types.Unsigned =>
            P ("declare");
            P ("   use Interfaces;");
            P ("   X : Unsigned_32 := Mcc.Gui.Widget.Text_Entry.Get_Text");
            P ("                    (" & Widget.Name.all & ");");
            P ("begin");
            P ("   " & Widget.Data_Item.all & " := ", Newline => False);
            -- only add a cast if we need to
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= ""
            then
               Put ("Unsigned_32 (X)");
            else
               Put ("X");
            end if;
            P (";", Indent => False);
            Print_Exception_Block;
         when mcc.text_entry_types.Float_1 |
              mcc.text_entry_types.Float_2 |
              mcc.text_entry_types.Float_3 |
              mcc.text_entry_types.Float_E =>
            Print_Declare_Block;
            P ("   " & Widget.Data_Item.all & " := ", Newline => False);
            -- if data type is blank, use standard.float
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= ""
            then
               Put (Widget.Data_Type.all & "'Value (X)");
            else
               Put ("Float'Value (X)");
            end if;
            P (";", Indent => False);
            Print_Exception_Block;
         when mcc.text_entry_types.Enumeration =>
            if Widget.Data_Type /= null
              and then Widget.Data_Type.all /= ""
            then
               Print_Declare_Block;
               P ("   " & Widget.Data_Item.all &
                  " := " & Widget.Data_Type.all & "'Value(X);");
               Print_Exception_Block;
            end if;
      end case;
   end Generate_Read_Action;

end Gui.Widget.Text;
