---------------------------------------------------------------
--
--  RAPID - Rapid Ada Portable Interface Designer
--
--  GUI-WIDGET-DROPDOWN.ADB
--  Description : GUI Widget Dropdown
--
--  Copyright (C) 2000, Jonathan Busch and Martin Carlisle
--
-- RAPID is free software; you can redistribute it and/or
-- modify it without restriction.  However, we ask that you
-- please retain the original author information, and clearly
-- indicate if it has been modified.
--
-- RAPID is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty
-- of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
--
-- As a special exception, if other files instantiate generics from
-- this unit, or you link this unit with other files to produce an
-- executable, this unit does not by itself cause the resulting
-- executable to be covered by the GNU General Public License.
-- This exception does not however invalidate any other reasons
-- why the executable file might be covered by the GNU Public
-- License.
---------------------------------------------------------------
with Gui_Enum;
with File_Helpers;
with mcc.Common_Dialogs;
with dropdown_dialog_window;
with Ada.Exceptions;
with Ada.Integer_Text_IO;
with mcc.Gui.Widget.Dropdown; use type mcc.Gui.Widget.Widget_Pointer;
with mcc.Gui.Colors;
with Generate_Helpers;
with state;

package body gui.Widget.Dropdown is

   -- reads information from file into Dropdown,
   -- assumes keyword already read.
   procedure Read_Widget (Widget : in out Dropdown) is
   begin -- Read_Widget
      Read_Widget (GUI_Widget (Widget));
      -- Get number of rows for list when open
      Widget.Number_Rows       :=
         Natural'Value (File_Helpers.Token (File_Helpers.Token_Index).all);
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      -- Get Colors
      Widget.FG_Color          := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      Widget.BG_Color          := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;

      if File_Helpers.Token_Index > File_Helpers.N_Tokens then
         return;
      end if;
      Widget.Enumeration_Var   := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
      Widget.Enumeration_Type  := Get_String;
      File_Helpers.Token_Index := File_Helpers.Token_Index + 1;
   end Read_Widget;

   -- Writes information to file from Dropdown
   procedure Write_Widget (Widget : in Dropdown) is
   begin -- Write_Widget
      File_Helpers.Put (Gui_Enum.Img (Gui_Enum.Dropdown) & " ");
      Write_Widget (GUI_Widget (Widget));

      File_Helpers.Put (Natural'Image (Widget.Number_Rows));
      File_Helpers.Put (" " & Widget.FG_Color.all);
      File_Helpers.Put (" " & Widget.BG_Color.all);
      File_Helpers.Put ("""");
      if Widget.Enumeration_Var /= null then
         File_Helpers.Put (Widget.Enumeration_Var.all);
      end if;
      File_Helpers.Put (""" """);
      if Widget.Enumeration_Type /= null then
         File_Helpers.Put (Widget.Enumeration_Type.all);
      end if;
      File_Helpers.P ("""");
   end Write_Widget;

   -- mcc 05/23/00
   procedure Generate_Action_Context_Clause (Widget : in Dropdown) is
   begin
      if Widget.Enumeration_Var /= null
        and then Widget.Enumeration_Var.all /= ""
      then
         Generate_Helpers.Generate_With_For_FQN (Widget.Enumeration_Var.all);
      end if;
      if Widget.Enumeration_Type /= null
        and then Widget.Enumeration_Type.all /= ""
      then
         Generate_Helpers.Generate_With_For_FQN (Widget.Enumeration_Type.all);
      end if;
   end Generate_Action_Context_Clause;

   -- SAG 02.Dec.1999
   procedure Generate_Callback_Context_Clause
     (Widget     : in Dropdown;
      Windowname : in String)
   is
   begin
      Generate_Action_Context_Clause (Widget);  -- CHECK: Do we need this?
      Generate_Helpers.Generate_With ("Mcc.Gui.Widget.Dropdown");
      Generate_Helpers.Generate_With (Windowname);
   end Generate_Callback_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Context_Clause (Widget : in Dropdown) is
   begin
      Generate_Helpers.Generate_With ("Mcc.Gui.Widget.Dropdown");
      Generate_Helpers.Generate_With ("Mcc.Gui.Colors");
   end Generate_Widget_Context_Clause;

   -- wbw 6/6/99
   procedure Generate_Widget_Declaration (Widget : in Dropdown) is
   begin
      File_Helpers.P
        (Widget.Name.all &
         " : aliased Mcc.Gui.Widget.Dropdown.Dropdown;");
   end Generate_Widget_Declaration;

   -- wbw 5/10/99 modified by mcc 05/24/00
   procedure Generate_Widget_Creation
     (Widget      : in Dropdown;
      Window_Name : in String)
   is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      P ("Mcc.Gui.Widget.Dropdown.Create");
      P ("  (Obj         => " & Widget.Name.all & ",");
      P ("   Parent      => " & Window_Name & ",");
      P ("   X           =>" & Integer'Image (Widget.x) & ",");
      P ("   Y           =>" & Integer'Image (Widget.y) & ",");
      P ("   Width       =>" & Integer'Image (Widget.Width) & ",");
      P ("   Height      =>" & Integer'Image (Widget.Height) & ",");
      P ("   Number_Rows =>" & Integer'Image (Widget.Number_Rows) &
         ");");

      if Widget.Enumeration_Type /= null
        and then Widget.Enumeration_Type.all /= ""
      then
         P ("for E in " & Widget.Enumeration_Type.all & " loop");
         P ("   Mcc.Gui.Widget.Dropdown.Add_Entry");
         P ("     (Obj      => " & Widget.Name.all & ",");
         P ("      Location => " & Widget.Enumeration_Type.all & "'Pos(E),");
         P ("      Text     => " &
            Widget.Enumeration_Type.all &
            "'Image(E));");
         P ("end loop;");
      end if;

      if Widget.BG_Color.all /= "default" then
         P ("Mcc.Gui.Widget.Dropdown.Set_Background_Color");
         P ("  (Obj       => " & Widget.Name.all & ",");
         P ("   New_Color => Mcc.Gui.Colors.Named_Color");
         P ("      (Mcc.Gui.Colors." & Widget.BG_Color.all & "));");
      end if;

      if Widget.FG_Color.all /= "default" then
         P ("Mcc.Gui.Widget.Dropdown.Set_Foreground_Color");
         P ("  (Obj       => " & Widget.Name.all & ",");
         P ("   New_Color => Mcc.Gui.Colors.Named_Color");
         P ("      (Mcc.Gui.Colors." & Widget.FG_Color.all & "));");
      end if;
   end Generate_Widget_Creation;

   function Has_Anything_To_Fill_Or_Read
     (Widget : in Dropdown)
      return   Boolean
   is
   begin
      return Widget.Enumeration_Var /= null
            and then Widget.Enumeration_Var.all /= ""
            and then Widget.Enumeration_Type /= null
            and then Widget.Enumeration_Type.all /= "";
   end Has_Anything_To_Fill_Or_Read;

   procedure Generate_Fill_Action (Widget : in Dropdown) is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      if Has_Anything_To_Fill_Or_Read (Widget) then
         P ("Mcc.Gui.Widget.Dropdown.Select_Item");
         P ("  (Obj      => " & Widget.Name.all & ",");
         P ("   Number   => " & Widget.Enumeration_Type.all &
            "'Pos(" & Widget.Enumeration_Var.all & ") + 1);");
      end if;
   end Generate_Fill_Action;

   procedure Generate_Read_Action (Widget : in Dropdown) is
      procedure P (Text : String) is
      begin
         File_Helpers.P (Text);
      end P;
   begin
      if Has_Anything_To_Fill_Or_Read (Widget) then
         P ("declare");
         P ("   Position : Integer := " &
            "Mcc.Gui.Widget.Dropdown.Get_Selected(" &
            Widget.Name.all & ") - 1;");
         P ("begin");
         P ("   " & Widget.Enumeration_Var.all & " := " &
            Widget.Enumeration_Type.all & "'Val(Position);");
         P ("end;");
      end if;
   end Generate_Read_Action;

   -- display the widget to a window
   procedure Display_Widget
     (Widget    : in out Dropdown;
      Container : in out mcc.Gui.Container.Container'Class)
   is

   begin
      if Widget.The_Widget = null then
         Widget.The_Widget := new mcc.Gui.Widget.Dropdown.Dropdown;
      end if;

      mcc.Gui.Widget.Dropdown.Create
        (Obj         =>
           mcc.Gui.Widget.Dropdown.Dropdown (Widget.The_Widget.all),
         Parent      => Container,
         X           => Widget.x,
         Y           => Widget.y,
         Width       => Widget.Width,
         Height      => Widget.Height,
         Number_Rows => Widget.Number_Rows);

      if Widget.FG_Color.all /= "default" then
         mcc.Gui.Widget.Set_Foreground_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.Gui.Colors.Named_Color (mcc.Gui.Colors.Color_Enum'Value
                                             (Widget.FG_Color.all)));
      end if;
      if Widget.BG_Color.all /= "default" then
         mcc.Gui.Widget.Set_Background_Color
           (Obj       => Widget.The_Widget.all,
            New_Color =>
              mcc.Gui.Colors.Named_Color (mcc.Gui.Colors.Color_Enum'Value
                                             (Widget.BG_Color.all)));
      end if;

      Display_Widget (GUI_Widget (Widget), Container);

   exception
      when e : others =>
         mcc.Common_Dialogs.Ok_Box
           ("Can't display: " &
            Widget.Name.all &
            ASCII.LF &
            Ada.Exceptions.Exception_Information (e));
   end Display_Widget;

   procedure Set_Properties (Widget : in out Dropdown) is
   begin
      dropdown_dialog_window.Generate_Window;
      Widget.Properties             :=
        dropdown_dialog_window.dropdown_dialog_window'Access;
      Widget.Name_Entry             := dropdown_dialog_window.entry1'Access;
      Widget.X_Entry                := dropdown_dialog_window.entry2'Access;
      Widget.Y_Entry                := dropdown_dialog_window.entry3'Access;
      Widget.Width_Entry            := dropdown_dialog_window.entry4'Access;
      Widget.Height_Entry           := dropdown_dialog_window.entry5'Access;
      Widget.FG_Entry               := dropdown_dialog_window.entry6'Access;
      Widget.BG_Entry               := dropdown_dialog_window.entry7'Access;
      Widget.Rows_Entry             :=
        dropdown_dialog_window.Row_Entry'Access;
      Widget.Enumeration_Type_Entry := dropdown_dialog_window.entry8'Access;
      Widget.Enumeration_Var_Entry  := dropdown_dialog_window.entry9'Access;
      Set_Properties (GUI_Widget (Widget));

      mcc.Gui.Widget.Text_Entry.Set_Text
        (Obj  => Widget.Rows_Entry.all,
         Text => Widget.Number_Rows);

      if Widget.BG_Color /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.BG_Entry.all,
            Text => Widget.BG_Color.all);
      end if;
      if Widget.FG_Color /= null then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.FG_Entry.all,
            Text => Widget.FG_Color.all);
      end if;
      if state.Get_Current_Window.Novice_Mode then
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Enumeration_Var_Entry.all,
            Text => "disabled for novice");
         mcc.Gui.Widget.Text_Entry.Disable (Widget.Enumeration_Var_Entry.all);
         mcc.Gui.Widget.Text_Entry.Set_Text
           (Obj  => Widget.Enumeration_Type_Entry.all,
            Text => "disabled for novice");
         mcc.Gui.Widget.Text_Entry.Disable
           (Widget.Enumeration_Type_Entry.all);
      else
         if Widget.Enumeration_Var /= null then
            mcc.Gui.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Enumeration_Var_Entry.all,
               Text => Widget.Enumeration_Var.all);
         end if;
         if Widget.Enumeration_Type /= null then
            mcc.Gui.Widget.Text_Entry.Set_Text
              (Obj  => Widget.Enumeration_Type_Entry.all,
               Text => Widget.Enumeration_Type.all);
         end if;
      end if;
   end Set_Properties;

   procedure Apply_Properties (Widget : in out Dropdown) is
   begin
      Apply_Properties (GUI_Widget (Widget));

      begin
         Widget.Number_Rows :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.Rows_Entry.all);
      exception
         when others =>
            Widget.Number_Rows := -99;
      end;
      declare
         FG_Color : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.FG_Entry.all);
      begin
         if FG_Color = "" then
            Widget.FG_Color := new String'("default");
         else
            Widget.FG_Color := new String'(FG_Color);
         end if;
      end;
      declare
         BG_Color : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text (Widget.BG_Entry.all);
      begin
         if BG_Color = "" then
            Widget.BG_Color := new String'("default");
         else
            Widget.BG_Color := new String'(BG_Color);
         end if;
      end;
      declare
         Enumeration_Var : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text
              (Widget.Enumeration_Var_Entry.all);
      begin
         if Enumeration_Var /= "disabled for novice" then
            Widget.Enumeration_Var := new String'(Enumeration_Var);
         end if;
      end;
      declare
         Enumeration_Type : String :=
            mcc.Gui.Widget.Text_Entry.Get_Text
              (Widget.Enumeration_Type_Entry.all);
      begin
         if Enumeration_Type /= "disabled for novice" then
            Widget.Enumeration_Type := new String'(Enumeration_Type);
         end if;
      end;

   end Apply_Properties;

   procedure Check_Properties (Widget : in out Dropdown; Ok : out Boolean) is
   begin
      Check_Properties (GUI_Widget (Widget), Ok);

      if Ok and then Widget.Number_Rows < 0 then
         Ok := False;
         mcc.Gui.Widget.Text_Entry.Highlight (Widget.Rows_Entry.all);
      end if;

      if Ok then
         Ok := (Widget.FG_Color /= null) and then (Widget.FG_Color.all /= "");
         if Ok and then Widget.FG_Color.all /= "default" then
            declare
               Color : mcc.Gui.Colors.Color_Enum;
            begin
               Color := mcc.Gui.Colors.Color_Enum'Value (Widget.FG_Color.all);
            exception
               when others =>
                  Ok := False;
            end;
         end if;
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.FG_Entry.all);
         end if;
      end if;
      if Ok then
         Ok := (Widget.BG_Color /= null) and then (Widget.BG_Color.all /= "");
         if Ok and then Widget.BG_Color.all /= "default" then
            declare
               Color : mcc.Gui.Colors.Color_Enum;
            begin
               Color := mcc.Gui.Colors.Color_Enum'Value (Widget.BG_Color.all);
            exception
               when others =>
                  Ok := False;
            end;
         end if;
         if not Ok then
            mcc.Gui.Widget.Text_Entry.Highlight (Widget.BG_Entry.all);
         end if;
      end if;

      if not Ok then
         mcc.Gui.Bell;
      end if;
   end Check_Properties;

end Gui.Widget.Dropdown;
