/************************************************************************
 *                                                                      *
 * Open File Manager - ncurses file manager for GNU/Linux               *
 * (c) 2001, 2002 Slawomir Strumecki, Raphael Bugajewski                *
 *                                                                      *
 * This program is free software, you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to the Free Software          *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 *
 *                                                                      *
 ************************************************************************
 *                                                                      *
 * If you want to contact us, please use the following address(es):     *
 *                                                                      *
 *     Raphael Bugajewski               Slawomir Strumecki              *
 *     Kl. Mittelstr. 1                 mailto: <logospam@poczta.fm>    *
 *     13585 Berlin / Germany                                           *
 *     Tel.: +49 (175) 331 93 92                                        *
 *     mailto: <born@bugajewski.de>                                     *
 *                                                                      *
 ************************************************************************/

#include "misc.h"
#include "fildir.h"
#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <unistd.h>


int dp_break;


int cmpSize(const void *s, const void *d)
{
    DirStruct *a,*b;
    
    a=*(DirStruct **)s;
    b=*(DirStruct **)d;
    if (! (a->dir ||  b->dir)) return ( (a->size>b->size) ? 1 : ((a->size<b->size) ? -1 : 0) );
    if (a->dir && b->dir) strcasecmp(a->name,b->name);
    return (a->dir ? -1 : 1);
}

int cmpDirFirst(const void *s, const void *d)
{
    DirStruct *a,*b;
    
    a=*(DirStruct **)s;
    b=*(DirStruct **)d;
    if (a->dir==b->dir) return strcasecmp(a->name,b->name);
    return ((a->dir) ? -1 : 1);
}

int cmpName(const void *s, const void *d)
{
    return strcasecmp((*(DirStruct **)s)->name,(*(DirStruct **)d)->name);
}

/* we need to implement a loop with a stack, but I'm too lazy ;-) */

void dirQsort(DirStruct **tab, size_t n, int (*fptr)(DirStruct *s, DirStruct *d))
{
    int l,r;
    int left,right,v;
    DirStruct *tmp;
    
    if (n<2) return;
    left=0;right=n-1;
    l=left;r=right;
    v=r/2-1;
    
    fprintf(stderr,"v(%d): %s\n",v,tab[v]->name);
	
    /*<partition>*/
    
	do {
	
	/* we are searching for a not fitting on the right side ... */
	
	while (fptr(tab[v],tab[r])<=0) {r--;}

	/* and on the left side */
	
	while (fptr(tab[v],tab[l])>=0) {l++;}
    
	/* if different then swap */
	
	if (l<r) {
	    fprintf(stderr,"swap: %s <-> %s\n",tab[l]->name,tab[r]->name);
	    tmp=tab[r];
	    tab[r]=tab[l];
	    tab[l]=tmp;
	}
    } while(l<r);
    fprintf(stderr,"l(%d): %s r(%d): %s\n",l,tab[l]->name,r,tab[r]->name);
    tmp=tab[r];
    tab[r]=tab[v];
    tab[v]=tmp;

    /*</partition>*/
    
}

/* sorts the dst structure */

DirStruct *dirSort(DirStruct *dst,int mode, int keep_prev)
{
    
	/* temporary variable */	
    DirStruct *tds=NULL;

	/* number of inodes */
    unsigned long c;

	/* helping variable */
    unsigned long i;

	/* all inodes */
    DirStruct **tdsa;

    int	(*compr)(const void *, const void *);

    if (dst==NULL) return NULL;

	/* if there is nothing to sort or we don't need to sort */	
    if ((dst->files_count+dst->dir_count==0) || (mode==SORT_UNSORTED)) return dst;

	/* number of inodes */
    c=dst->last->index-dst->index+1;

	/* if there is nothing to sort */	
    if (c<2) return dst;

    tds=dst;
    tdsa=(DirStruct **)xmalloc(c*sizeof(DirStruct));
    for (i=0;((i<c) && (tds!=NULL));i++) {
	tdsa[i]=tds;
	tds=tds->next;
    }
    if (i!=c) c=i;
    
    if (mode==SORT_NAME) compr=cmpName;
    else if (mode==SORT_DIR_FIRST) compr=cmpDirFirst;
    else if (mode==SORT_SIZE) compr=cmpSize;
    qsort(tdsa,c,sizeof(*tdsa),compr);


	/* moving from table to list */

    tdsa[0]->index=dst->index;
    tdsa[0]->last=tdsa[c-1];
    if (keep_prev) {
	tdsa[0]->prev=dst->prev;
	tdsa[0]->first=dst->first;
    } else {
	tdsa[0]->prev=NULL;
	tdsa[0]->first=tdsa[0];
    }
    tdsa[c-1]->next=NULL;
    for (i=1;i<c;i++) {
	tds=tdsa[i];
	tds->prev=tdsa[i-1];
	tds->prev->next=tdsa[i];
	tds->index=tds->prev->index+1;
	tds->first=tds->prev->first;
	tds->last=tdsa[c-1];
    } 
    tds=tdsa[0];    
    xfree(tdsa);
    return tds;
}

/* reads the directory content */
/* return NULL only if there is enough memory */

DirStruct *dirSuck(char *dname) 
{ 
    DIR *cat;
    struct dirent *node;
    struct stat stbuf;
    DirStruct *tmp=NULL,*tmp2=NULL;
    size_t	DSsize=sizeof(DirStruct);
    char *tname;
    long tnsize;
    unsigned long dc=0,fc=0;
    unsigned long tindex=1;

	cat=opendir(dname);
	/* if we can't read the directory 
	   we create a node */
	if (cat==NULL) {
	    tmp=(DirStruct *)xmalloc(DSsize);
	    if (tmp!=NULL) {
		tmp->name=(char*)xmalloc(3);
		sprintf(tmp->name,"..");
		tmp->path=NULL;
		tmp->mode=040755;
		tmp->dir=1;
		tmp->link=0;
		tmp->prev=NULL;
		tmp->next=NULL;
		tmp->first=tmp;
		tmp->last=tmp;
		tmp->corrupt=0;
		tmp->exe=0;
		tmp->index=1;
		tmp->selected=0;
		tmp->files_count=0;
		tmp->dir_count=0;
		tmp->size=4096;
	    }
	    return tmp;
	}
	
	node=readdir(cat);
    	while ((node=readdir(cat))!=NULL) {

		/* we are writing the pointer to the first DirStruct in the list */		
    	    if (tmp2==NULL) {
	    	tmp=(DirStruct *)xmalloc(DSsize); 
		if(tmp==NULL) return NULL;
		tmp->first=tmp;
    		tmp->prev=NULL;
		tmp2=tmp;
	    }
	    else {
	    	tmp->next=(DirStruct *)xmalloc(DSsize); 
		if (tmp->next==NULL) return NULL;
		tmp->next->prev=tmp;
		tmp->next->first=tmp->first;
		tmp=tmp->next;
	    }
	    
	    tmp->path=NULL;	
	    tmp->name=NULL;
	    tnsize=strlen(node->d_name)+1;
	    tname=(char *)xmalloc(tnsize);
	    //strcpy(tname,node->d_name);
	    sprintf(tname,"%s",node->d_name);
	    tmp->name=tname;
	    tmp->index=tindex++;
	    tmp->selected=FALSE;
	    	
	    if (stat(node->d_name,&stbuf)==-1) {
		tmp->corrupt=TRUE;
		tmp->dir=FALSE; 
	    }
	    else {
		tmp->corrupt=FALSE;
		tmp->size=stbuf.st_size;
		tmp->atime=stbuf.st_atime;
		tmp->mtime=stbuf.st_mtime;
		tmp->ctime=stbuf.st_ctime;
		tmp->mode=stbuf.st_mode;
		tmp->uid=stbuf.st_uid;
		tmp->gid=stbuf.st_gid;
		tmp->exe=(((stbuf.st_mode&S_IXUSR)==S_IXUSR)||((stbuf.st_mode&S_IXGRP)==S_IXGRP)||((stbuf.st_mode&S_IXOTH)==S_IXOTH));
		tmp->dir=((stbuf.st_mode&S_IFMT)==S_IFDIR); 
	    }
		
	    if (lstat(node->d_name,&stbuf)==0) {
	    	tmp->link=S_ISLNK(stbuf.st_mode);
	    	tmp->size=stbuf.st_size;
	    	tmp->mode=stbuf.st_mode;
	    	//if (tmp->corrupt) tmp->mode=stbuf.st_mode&S_IFMT;
	    }
			
	    if (tmp->dir) dc++; 
	    else fc++; 
	}
		 		
	tmp->next=NULL;
	closedir(cat);
	tmp2->last=tmp;
	tmp2->dir_count=--dc;
	tmp2->files_count=fc;
	
	/* copying dirs count and files count to all nodes */
	tmp=tmp2;
	while (tmp->next!=NULL) {
	    tmp->next->dir_count=tmp->dir_count;
	    tmp->next->files_count=tmp->files_count;
	    tmp->next->last=tmp->last;
	    tmp=tmp->next;
	}
	
	return tmp2;
}

/* duplicate DirStruct structure */
DirStruct *dirDuplicate(DirStruct *ds)
{
    DirStruct *tmp;

    tmp=NULL;
    if (ds!=NULL) {
	tmp=(DirStruct *)xmalloc(sizeof(DirStruct));
	if (tmp!=NULL) {
	    tmp->name=strdup(ds->name);
	    tmp->path=NULL;
	    tmp->mode=ds->mode;
	    tmp->dir=ds->dir;
	    tmp->link=ds->link;
	    tmp->prev=NULL;
	    tmp->next=NULL;
	    tmp->first=NULL;
	    tmp->last=NULL;
	    tmp->corrupt=ds->corrupt;
	    tmp->exe=ds->exe;
	    tmp->index=ds->index;
	    tmp->dir_count=0;
	    tmp->files_count=0;
	    tmp->selected=0;
	    tmp->files_count=0;
	    tmp->dir_count=0;
	    tmp->size=ds->size;
	}
    }
    return tmp;
}


/* frees the memory used by dst */

int dirFree(DirStruct *dst)
{
    DirStruct *tds;

    if (dst!=NULL) {
	//dst=dst->first;
	while (dst!=NULL) {
		tds=dst->next;
		xfree(dst->name);
		if (dst->path!=NULL) xfree(dst->path);    
		xfree(dst);
		dst=tds; 
	}
    }
    return 0;	
}

DirStruct *dirFind(DirStruct *ds, char *name)
{
    DirStruct *tmp;

    if ((ds!=NULL) && (name!=NULL)) {
	tmp=ds->first;

	/* we need to expand the term , we don't need to check all */
	
	while(tmp!=NULL){
	    if (strcmp(name,tmp->name)==0) return tmp;
	    tmp=tmp->next;
	}
//	fprintf(stderr,"DIRFIND: nie znaleziono %s. skonczono na %s\n",name,tmp->name);
    }
    return NULL;
}

int dirProcess(DirStruct *ds, void (*fptr)(DirStruct *,int) ,int all)
{
    DirStruct **dstab;
    DirStruct *subdir;
    int last;
    int levels;
    int tmp;

    dp_break=0;
    dstab=(DirStruct **)xmalloc(DP_DIR_DEEP_LEVEL*sizeof(DirStruct));
    last=0;
    levels=1;

	/* sets to... */
    
    ds=ds->first;

	/* loop reading files/directories - we need to improve it */

    do {
	ds=ds->next;

	/* only possible, if first directory */
	
	if (ds==NULL) return 0;
	if (ds->selected || last) {
	    if (ds->path!=NULL) chdir(ds->path);
	    if ((ds->dir) && (! ds->link)) {
		(*fptr)(ds,DP_DIRIN);
		tmp=chdir(ds->name);
		if (tmp==0) {
//		    fprintf(stderr,"DIRPROCESS: Entered to %s\n",ds->name);
		    subdir=dirSuck(".");

			/* we changed the directory */
			
		    if (subdir!=NULL){
			dstab[last++]=ds;
			ds=subdir;
		    }
		    else chdir("..");
		}
		if (last>=levels*DP_DIR_DEEP_LEVEL){
		    dstab=(DirStruct **)realloc((DirStruct **)dstab,++levels*DP_DIR_DEEP_LEVEL*sizeof(DirStruct));
		}
	    }

		/* if not a directory */
		
	    else {
		(*fptr)(ds,DP_FILE);
	    }
	}

	/* if the directory is passed through, but not the starting directory */
	
	while ( (ds->next==NULL)&&(last) ) {
	    last--;

		/* damn, theoritically here should be some errors */
		
	    chdir("..");
	    dirFree(ds);
	    ds=dstab[last];
//	    fprintf(stderr,"DIRPROCESS: Back from %s\n",ds->name);    
	    (*fptr)(ds,DP_DIROUT);
	}
    } while ((ds->next!=NULL) && (! dp_break));
    
    if (dp_break) 
	while (last--) {
	    chdir("..");
	    dirFree(dstab[last]);
	}
    xfree(dstab);
    return dp_break;
}

void dirProcessBreak(int errno)
{
    dp_break=errno;
}
