/* This file is part of the MPFRCPP Library.

  Copyright (c) 2006 -- 2007 Alexey V. Beshenov <bav.272304@gmail.com>.

  The MPFRCPP Library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public License as
  published by the Free Software Foundation; either version 2.1 of the
  License, or (at your option) any later version.

  The MPFRCPP Library is distributed in the hope that it will be
  useful, but WITHOUT ANY WARRANTY; without even the implied warranty
  of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with the MPFRCPP Library; see the file COPYING.LIB. If
  not, write to the Free Software Foundation, Inc., 51 Franklin Street,
  Fifth Floor, Boston, MA 02110-1301, USA. */

/**
 * @file numeric_function.hpp
 * @date 2007-04-03
 * MPFRCPP numeric functors base.
 * INTERNAL HEADER, NOT TO BE USED DIRECTLY.
 */

#ifndef INCLUDED_BY_MPFRCPP
    #error THIS IS AN INTERNAL HEADER ONLY, NOT TO BE USED DIRECTLY
#endif

#ifndef MPFRCPP_CORE_NUMERIC_FUNCTION
#define MPFRCPP_CORE_NUMERIC_FUNCTION

#include "precision.hpp"
#include "round_mode.hpp"

#include <set>
#include <algorithm>

namespace mpfrcpp {

    //------------------------------------------------------------

    /**
     * MPFRCPP numeric function base.
     */

    class NumericFunction {
            Precision pr_;
            RoundMode rm_;
        public:
            NumericFunction () throw();
            NumericFunction ( const Precision&, const RoundMode& ) throw();
            NumericFunction ( const Precision& ) throw();
            NumericFunction ( const RoundMode& ) throw();

            Precision getPrecision () const throw();
            RoundMode getRoundMode () const throw();
            void setPrecision ( const Precision& ) throw();
            void setRoundMode ( const RoundMode& ) throw();
    };

    //------------------------------------------------------------

    NumericFunction::NumericFunction () throw() :
            pr_( Real::getParameters().getDefaultPrecision() ),
    rm_( Real::getParameters().getDefaultRoundMode() ) {}

    NumericFunction::NumericFunction
    ( const Precision &pr, const RoundMode &rm ) throw() :
    pr_( pr ), rm_( rm ) {}

    NumericFunction::NumericFunction ( const Precision &pr ) throw() :
    pr_( pr ), rm_( Real::getParameters().getDefaultRoundMode() ) {}

    NumericFunction::NumericFunction ( const RoundMode &rm ) throw() :
    pr_( Real::getParameters().getDefaultPrecision() ), rm_( rm ) {}

    Precision NumericFunction::getPrecision () const throw() {
        return pr_;
    }
    RoundMode NumericFunction::getRoundMode () const throw() {
        return rm_;
    }

    void NumericFunction::setPrecision ( const Precision &pr ) throw() {
        pr_ = pr;
    }
    void NumericFunction::setRoundMode ( const RoundMode &rm ) throw() {
        rm_ = rm;
    }

    //------------------------------------------------------------

    /**
     * Set of NumericFunction objects (stored by pointers).
     * Should be used for synchronous changing of precision
     * and round mode.
     */

    class NumericFunctions {
            std::set<NumericFunction*> functions_;
        public :
            NumericFunctions() throw();

            void erase ( NumericFunction* ) throw();
            void erase ( NumericFunction& ) throw();
            void insert ( NumericFunction& ) throw();
            // NOTE: insertion keeps precision and round mode

            // Set Precision / RoundMode for each stored function:
            void setPrecision ( const Precision& ) throw();
            void setRoundMode ( const RoundMode& ) throw();
    };

    //------------------------------------------------------------

    namespace internal {

        /**
         * INTERNAL functor
         */

        class setPrecision {
                Precision pr_;
            public:
                setPrecision ( const Precision& pr ) throw() : pr_( pr ) {}
                void operator() ( NumericFunction* x ) const throw() {
                    x->setPrecision( pr_ );
                }
        };

        /**
         * INTERNAL functor
         */

        class setRoundMode {
                RoundMode rm_;
            public:
                setRoundMode ( const RoundMode& rm ) throw() : rm_( rm ) {}
                void operator() ( NumericFunction* x ) const throw() {
                    x->setRoundMode( rm_ );
                }
        };

    }    // namespace internal

    NumericFunctions::NumericFunctions () throw() {}

    void NumericFunctions::erase ( NumericFunction* p ) throw() {
        functions_.erase( p );
    }

    void NumericFunctions::erase ( NumericFunction& p ) throw() {
        functions_.erase( &p );
    }

    void NumericFunctions::insert ( NumericFunction& p ) throw() {
        functions_.insert( &p );
    }

    void NumericFunctions::setPrecision ( const Precision &pr ) throw() {
        std::for_each ( functions_.begin(), functions_.end(),
                        internal::setPrecision( pr ) );
    }
    void NumericFunctions::setRoundMode ( const RoundMode &rm ) throw() {
        std::for_each ( functions_.begin(), functions_.end(),
                        internal::setRoundMode( rm ) );
    }

    //------------------------------------------------------------

} // namespace mpfrcpp

#endif    // MPFRCPP_CORE_NUMERIC_FUNCTION
