# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import pymobius
import pymobius.app.firefox
import pymobius.app.geckofx
import pymobius.app.internet_explorer
import mobius

ATTR_NAME = 'data.files_opened'

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Model class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class model (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize object
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, item):
    self.__item = item

    if item.has_attribute (ATTR_NAME):
      self.__entries = item.get_attribute (ATTR_NAME)
      self.__entries_loaded = True

    else:
      self.__entries = []
      self.__entries_loaded = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get data from model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get (self):
    if not self.__entries_loaded:
      self.__retrieve_entries ()
      self.__entries_loaded = True
      self.__item.set_attribute (ATTR_NAME, self.__entries)

    return self.__entries

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get data from model
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def refresh (self):
    self.__item.remove_attribute (ATTR_NAME)
    self.__entries = []
    self.__entries_loaded = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_entries (self):
    self.__retrieve_firefox ()
    self.__retrieve_geckofx ()
    self.__retrieve_internet_explorer ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Firefox
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_firefox (self):
    model = pymobius.app.firefox.model (self.__item)
    
    for profile in model.get_profiles ():
      for h in profile.get_history ():
        url = mobius.io.uri (h.url)

        if url.get_scheme == 'file':
          entry = pymobius.Data ()
         
          entry.path = url.get_path ().replace ('/', '\\')
          entry.timestamp = str (h.timestamp) + ' (UTC)'
          entry.username = h.username
          entry.app_id = 'firefox'
          entry.app_name = 'Mozilla Firefox'
          entry.item = self.__item.get_attribute ('name')

          entry.metadata = []
          entry.metadata.append (('Typed', h.typed))
          entry.metadata.append (('URL', h.url))
          entry.metadata.append (('Profile', h.profile_name))
          entry.metadata.append (('Profile path', h.profile_path))

          self.__entries.append (entry)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from GeckoFX
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_geckofx (self):
    model = pymobius.app.geckofx.model (self.__item)
    
    for profile in model.get_profiles ():
      for h in profile.get_history ():
        url = mobius.io.uri (h.url)

        if url.get_scheme == 'file':
          entry = pymobius.Data ()
         
          entry.path = url.get_path ().replace ('/', '\\')
          entry.timestamp = str (h.timestamp) + ' (UTC)'
          entry.username = h.username
          entry.app_id = 'geckofx'
          entry.app_name = 'GeckoFX'
          entry.item = self.__item.get_attribute ('name')

          entry.metadata = []
          entry.metadata.append (('Typed', h.typed))
          entry.metadata.append (('URL', h.url))
          entry.metadata.append (('Profile', h.profile_name))
          entry.metadata.append (('Profile path', h.profile_path))

          self.__entries.append (entry)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Retrieve data from Internet Explorer
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __retrieve_internet_explorer (self):
    model = pymobius.app.internet_explorer.model (self.__item)
    
    for profile in model.get_profiles ():
      for h in profile.get_history ():
        url = mobius.io.uri (h.url)

        if url.get_scheme == 'file':
          entry = pymobius.Data ()
          entry.path = url.get_path ().replace ('/', '\\')
          
          if h.timestamp_utc:
            entry.timestamp = str (h.timestamp) + ' (UTC)'
          else:
            entry.timestamp = str (h.timestamp) + ' (local time)'

          entry.username = h.username
          entry.app_id = 'internet_explorer'
          entry.app_name = 'Internet Explorer'
          entry.item = self.__item.get_attribute ('name')

          entry.metadata = []
          entry.metadata.append (('Record type', h.type))
          entry.metadata.append (('Record offset', '0x%08x' % h.offset))
          entry.metadata.append (('Record size', h.size))
          entry.metadata.append (('Record is active', h.is_active))
          entry.metadata.append (('Record primary time (UTC)', h.primary_time))

          if h.file_type in ('daily', 'weekly'):
            entry.metadata.append (('Record secondary time (local time)', h.secondary_time))
          else:
            entry.metadata.append (('Record secondary time (UTC)', h.secondary_time))

          entry.metadata.append (('URL', h.url))
          entry.metadata.append (('index.dat path', h.index_dat_path))
          entry.metadata.append (('index.dat creation time', h.index_dat_creation_time))
          entry.metadata.append (('index.dat last modification time', h.index_dat_last_modification_time))
            
          self.__entries.append (entry)
