// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017,2018,2019,2020 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include "imagefile_impl_raw.h"
#include <mobius/datetime/conv_iso_string.h>
#include <mobius/io/file.h>

#include <iostream>

namespace mobius
{
namespace imagefile
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief construct object
//! \param url imagefile URL
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
imagefile_impl_raw::imagefile_impl_raw (const std::string& url)
  : url_ (url)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new reader for imagefile
//! \return reader
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::io::reader
imagefile_impl_raw::new_reader () const
{
  auto f = mobius::io::new_file_by_url (url_);
  return f.new_reader ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief create new writer for imagefile
//! \return writer
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::io::writer
imagefile_impl_raw::new_writer () const
{
  auto f = mobius::io::new_file_by_url (url_);
  return f.new_writer ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief get metadata
//! \return imagefile metadata
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
mobius::metadata
imagefile_impl_raw::get_metadata () const
{
  return mobius::metadata
  {
    {
      "url",
      "URL",
      "std::string",
      get_url ()
    },
    {
      "type",
      "type",
      "std::string",
      get_type ()
    },
    {
      "size",
      "size",
      "size_type",
      std::to_string (get_size ()) + " bytes"
    },
    {
      "sectors",
      "number of sectors",
      "size_type",
      std::to_string (get_sectors ())
    },
    {
      "sector_size",
      "sector size",
      "size_type",
      std::to_string (get_sector_size ()) + " bytes"
    },
    {
      "last_metadata_time",
      "last metadata modification date/time",
      "mobius::datetime::datetime",
      to_string (get_last_metadata_time ())
    },
    {
      "last_modification_time",
      "last modification date/time",
      "mobius::datetime::datetime",
      to_string (get_last_modification_time ())
    },
    {
      "last_access_time",
      "last access date/time",
      "mobius::datetime::datetime",
      to_string (get_last_access_time ())
    },
    {
      "user_name",
      "user name",
      "std::string",
      get_user_name ()
    },
    {
      "group_name",
      "group name",
      "std::string",
      get_group_name ()
    },
  };
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief load metadata
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
void
imagefile_impl_raw::_load_metadata () const
{
  if (metadata_loaded_)
    return;

  auto f = mobius::io::new_file_by_url (url_);

  if (f && f.exists ())
    {
      size_ = f.get_size ();
      sectors_ = (size_ + sector_size_ - 1) / sector_size_;
      last_metadata_time_ = f.get_metadata_time ();
      last_modification_time_ = f.get_modification_time ();
      last_access_time_ = f.get_access_time ();
      user_name_ = f.get_user_name ();
      group_name_ = f.get_group_name ();
    }

  metadata_loaded_ = true;
}

} // namespace imagefile
} // namespace mobius
