// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015,2016,2017 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <mobius/datetime/datetime.h>
#include <mobius/datetime/conv_julian.h>
#include <mobius/datetime/conv_unix_timestamp.h>
#include <chrono>

namespace mobius
{
namespace datetime
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief constructor
//! \param y year
//! \param m month (1-12)
//! \param d day (1-31)
//! \param hh hour
//! \param mm minute (0 - 59)
//! \param ss second (0 - 59)
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
datetime::datetime (int y, int m, int d, int hh, int mm, int ss) noexcept
  : date_ (y, m, d),
    time_ (hh, mm, ss),
    is_valid_ (true)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief construct datetime from date and time
//! \param d date object
//! \param t time object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
datetime::datetime (const date& d, const time& t) noexcept
  : date_ (d),
    time_ (t),
    is_valid_ (true)
{
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief subtract a timedelta from a datetime
//! \param dt datetime object
//! \param delta timedelta object
//! \return new datetime object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const datetime
now ()
{
  auto clock_now = std::chrono::system_clock::now ();
  time_t t = std::chrono::system_clock::to_time_t (clock_now);

  return datetime_from_unix_timestamp (t);
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief compare two datetimes
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
bool
operator== (const datetime& d1, const datetime& d2) noexcept
{
  return bool (d1) == bool (d2) &&
         d1.get_date () == d2.get_date () &&
         d1.get_time () == d2.get_time ();
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief add a timedelta to a datetime
//! \param dt datetime object
//! \param delta timedelta object
//! \return new datetime object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const datetime
operator+ (const datetime& dt, const timedelta& delta) noexcept
{
  using resolution_type = timedelta::resolution_type;
  resolution_type seconds = dt.get_time ().to_day_seconds () + delta.get_seconds ();
  resolution_type days = 0;

  constexpr resolution_type SECONDS_OF_DAY = 86400;

  if (seconds >= 0)
    {
      days = (seconds / SECONDS_OF_DAY) + delta.get_days ();
      seconds = seconds % SECONDS_OF_DAY;
    }
  else
    {
      seconds += SECONDS_OF_DAY;
      days--;
    }

  // create new timedelta, with seconds = 0
  mobius::datetime::timedelta delta2 (delta);
  delta2.set_days (days);
  delta2.set_seconds (0);

  // return new datetime
  return mobius::datetime::datetime (dt.get_date () + delta2, time (seconds % SECONDS_OF_DAY));
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief subtract a timedelta from a datetime
//! \param dt datetime object
//! \param delta timedelta object
//! \return new datetime object
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
const datetime
operator- (const datetime& dt, const timedelta& delta) noexcept
{
  return dt + timedelta (
           -delta.get_years (),
           -delta.get_days (),
           -delta.get_seconds (),
           -delta.get_nanoseconds ()
         );
}

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//! \brief write datetime representation to std::ostream
//! \param stream ostream reference
//! \param dt datetime object
//! \return reference to ostream
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
std::ostream&
operator<< (std::ostream& stream, const datetime& dt)
{
  if (dt)
    {
      char buffer[128] = {0};
      auto d = dt.get_date ();
      auto t = dt.get_time ();

      sprintf (buffer, "%04d-%02d-%02d %02d:%02d:%02d",
               d.get_year (),
               d.get_month (),
               d.get_day (),
               t.get_hour (),
               t.get_minute (),
               t.get_second ());

      stream << buffer;
    }

  return stream;
}

} // namespace datetime
} // namespace mobius
