#ifndef __MOBIUS_BYTEARRAY_H__
#define __MOBIUS_BYTEARRAY_H__

// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
// Mobius Forensic Toolkit
// Copyright (C) 2008,2009,2010,2011,2012,2013,2014,2015 Eduardo Aguiar
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by the
// Free Software Foundation; either version 2, or (at your option) any later
// version.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
// Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program. If not, see <http://www.gnu.org/licenses/>.
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
#include <memory>
#include <cstddef>
#include <cstdint>
#include <string>
#include <initializer_list>

namespace mobius
{
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
//!\brief bytearray class with reference count and copy on write mechanism
//!\author Eduardo Aguiar
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class bytearray
{
public:
  using value_type = std::uint8_t;
  using reference = value_type&;
  using const_reference = const value_type&;
  using pointer = value_type *;
  using const_pointer = const value_type *;
  using size_type = std::size_t;
  using difference_type = ptrdiff_t;
  using const_iterator = const uint8_t *;
  using iterator = uint8_t *;

  bytearray ();
  bytearray (const char *);
  bytearray (const std::string&);
  bytearray (const uint8_t *, size_type);
  bytearray (const bytearray&);
  bytearray (bytearray&&) noexcept;
  bytearray (const std::initializer_list<uint8_t>&);
  explicit bytearray (size_type);

  const_reference operator[](size_type) const;
  reference operator[](size_type);
  bool operator==(const bytearray&) const;
  bytearray& operator=(const bytearray&);
  bytearray& operator^=(const bytearray&);

  size_type size () const;
  bool empty () const;
  const_pointer data () const;
  pointer data ();
  void fill (value_type);
  void resize (size_type);
  const std::string to_hexstring () const;
  bytearray slice (size_type, size_type) const;

  const_iterator begin () const;
  const_iterator end () const;
  iterator begin ();
  iterator end ();

private:
  size_type size_;
  std::unique_ptr<std::uint8_t[]> pdata_;
};

} // namespace mobius

#endif