# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Icon data
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''
iVBORw0KGgoAAAANSUhEUgAAACAAAAAgCAYAAABzenr0AAAHxUlEQVRYhcWXS4hlVxVA1znnft6v
uqq7ujvdHezuxE8HSQgqBoKJSRQxaDAZCM7EOFARDGgwAyPoTFRwEKOQCOogOFHQgSQiBgcxKjHB
lgySdLrzq0rXq9979T733Xu+28GrrkpI2pHQm8v9cNhnr3322fvsq0SEKyn6iloHsksvDz/+DCEl
irKg0BoNpOiREJEkxJCQJEjcXTGlkBhJKZGSICmhkhCAxjps43GN5YcPfOF/A6jbfgDAT+6/HZEE
IiCCQqEAELQCpfXuV9oFgCQKwZC0IAEUczijdjVFuDT/u8mtx1fmK/D5Ww7/fKO/8jURsVorpdTc
uOzCvE3kra/C7rV3k129pAVpi3z1M8eeeeTx/m3vBlA7h/rFk//m/AvPy+13fJq1wSYpKSAQVUYe
EoGaUQ3N9gQbHGIMWgCtMQAh0iRH5qFJCaUjKiiuOrjI6dPHeeyx33Li1AfUWw3/7IlX9kMwFaEW
YeIqqsZSAF4ZJNUEUcxmDls1NMnTNAktNd5GROeIm+LSHDRm4CpHp9Pm/aeu5pWVVR780aMstT3d
Nzb3DH79W/fBWwE2t8YMh1PObYzY7A/3BspOiQsO4w0xeJQ2JBKonHyxnC+zLNIKkSQZuBknjh0h
BHjsyad46cIFbrzuFJPhBkuHDgLwsc/d/Y4wZNZanHOsvb5GMxrtDUwai40R2ziMMTSzQNZRRG+w
rkamERunZFqj85KlA12e/udZXr7wOkvLBzi1vMTFN4coqVDjMXfe+xXw9TsBGqVwSnH01AkGVXdv
YLrjuObYIviAT+AGG4SdyHQ2oIiLxOMR7ZYIjSULibPnLlDWgVvfe4I0g+grDi4rKrfAXfd/h/Fg
6932IdnMWhrnuLi+waya7Q2c+dLdyEaDlC26bVhsAQF6nbkjzQgaCxFPijkf6UHZhdCARPARspCY
WgBk4cTy3tz3feMOHvrpXxVA5tEEDLGdkZq9usR7WvDaYIcqCi08A6/JteCcYzKLjEYzxtMJKijE
WkRn0CTEWCKGPAW8nReuiOOjZ84gBH78/V+BqWHp1G4Iqgrb1NSVYON+kj83hL4XSlGYvE3QkIoM
ZQONadBFRnmwg4QpKS6hUiAlR2pydFT4caKm4fSRDkcOL/ObP/+L1T8+Dcd6sNTDHDkwB9heeZNx
f4P+i+eQarKfBdt38fp/LpCXXbwDkQYJhmQd0SlMR6MATYbyQ0wQPA1lbrBoWott3nfVcZ57dY3n
H/0TJE3nzluQ/kXqqk/cdTaLM0eyHmYznNoPwbqFbPEQUpa02wqlDhNNhvOeFppqOEG2xoTpmFTm
5CgChu0L2yy2Iklyfv3w7yEk9KlDtDttqrUVqCtoAszcHCB5R4weURHx+wVr8zTk/QzpHiQoiCGQ
ZRmm1cHQpnf0EO6khYmlWX2DnZUNjE70zlzFysCx9eoa3PwhFpRhNhxSTSuoR7CxBkQwnTmArx2x
Dtixo5G4B9DfgmLL0ZqsUyMUvkAyjw8lEi0ymRFiohUgmkix3GUyGPLq385CBe3lJUKwTJwHZpju
AtJWJG1gPIDe0m4IdldgZmfEmPYARh1YXlhgsyzpeJj2AsgCup7C2DCbRuJwwLhpcJMhaTCBaQ0L
i5iuUA8vwmAGzkKyxGoKnRwMIB7SPD8zpRVaK9pk2P0tQPO782ysnifoxCjLwau5stFAxHiQqkGM
kMig26Vc6uJngmDQB65FLzuic6jpmFRV4MewswbTEcj8eM/ydg9TdDhwcJlJcvsEL52Duo/W+Zw4
64IPkEGKGQRPTB7jIyr34Cw2FOhOF4KGwsJAEOtRvoHGw2wMjUAKEOfhzgYbW0yHOzz17LNE2Q8B
t12P9pEYCqANsYIMJBQUyeIkkpPwqUJVCvKEsjKHLRSMc4L2ZFkgVBpwoDMoHViBtAtwz52fZPWV
F7jxwzczU/me/e898hfMNcdwaQA6BzEIkMcJrtUl8xqfalQo5k2LM5CX0NLgCoxySAqEpgESSjTi
GxAD0UMMAOilVpsshqdzVGw7u7YQXSiaRlhZR9wYGTVINUbGFbquCD6HyYwQG1TQQILCQQKiQ09B
pwbvwIiFrEAbhYiF0oC0QNI8DCnMW7Lrbrjplkuef/OLtwLw4N9flnpzE1olJkDSGglxbkwXYFqg
GkDAt+ae+0hwDmyGigOCL0AHUjUB20BsYDYB78HvFiIuI4XuIQsJVAZFA5eaVJ1BpsAnCArqBpSB
WkGngFyQNMYkQ7TrOBtQ0qDLFlEiUnbADSHP3wnwwC+f4Nv3fnyehqMpbG+CShALIIDSYP2+gndQ
5oCGbjl/GgemDX4H1AJM15FhnzTd3s0CByeWYbT9doCHnnqNaz94/d7cJxcM7UMn8WWPdgk6FGR5
wMSIBMiKhDrQg24HegV4gcwTdgQjFSHlpDwSig46b2F9wm9v4bZWSetre3ayf7w2PwFvOLLfDQF8
4vbP4p1DsoyUlUi7wGQtUseAutSFR2giuDCvEV6Y1+aISg5d1RDsvBgZMMsn0ctXU96035xcdg98
+eSLveDscYRFlBxNIR5+s7+5wFBaQKYjmqRUUlFppUkpglJojSQyIUuJJe1Qyl197OhIi9oB+gk1
KrLB6mUB7vnuHwC4/1OHKuD85QD/X3LFf06vOMB/ASD6kkHcRCPVAAAAAElFTkSuQmCC'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Working Area
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class WorkingArea (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Create working area
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, window_manager):
    self.window_manager = window_manager
    self.mediator = window_manager.mediator.copy ()
    self.uid = None
    self.window = self.mediator.call ('ui.new-window')
    self.window.connect ('delete-event', self.on_delete_event)
    self.widget = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set default size
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_default_size (self, width, height):
    self.window.set_default_size (width, height)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief set title of working area
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_title (self, title=''):
    app_name = self.mediator.call ('app.get-name')
    app_version = self.mediator.call ('app.get-version')

    w_title = '%s v%s' % (app_name, app_version)
    if title:
      w_title += ' - ' + title

    self.window.set_title (w_title)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set icon
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_icon (self, icon_data):
    if icon_data:
      icon = self.mediator.call ('ui.render-icon', icon_data, 64, 64)
      self.window.set_icon (icon)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Set widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_widget (self, widget):
    self.widget = widget
    self.window.add (widget)
    widget.working_area = self

    on_widget_started = getattr (widget, 'on_widget_started', None)
    if on_widget_started:
      self.on_widget_started ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Get widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_widget (self):
    return self.widget

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Show working area
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def show (self):
    config = self.mediator.call ('app.get-config', 'sdi-window-manager') or {}
    data = config.get (self.id)

    if data:
      self.window.move (data[0], data[1])
      self.window.resize (data[2], data[3])

    self.window.present ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Event: on_delete_event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_delete_event (self, widget, *args):

    # check if widget can be destroyed
    can_destroy = True

    if self.widget:
      on_widget_stopped = getattr (self.widget, 'on_widget_stopped', None)
      if on_widget_stopped:
        can_destroy = not on_widget_stopped ()

    # destroy it if necessary
    if can_destroy:
      self.window_manager.destroy (self.uid)

    else:
      return True


# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'sdi-window-manager'
    self.name = 'Single Document Interface'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Single document interface window manager'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

    # specific data
    self.__next_working_area_uid = 1
    self.__working_area = {}
    self.__working_area_by_id = {}

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.advertise ('ui.working-area.new', self.svc_new_working_area)
    self.mediator.advertise ('ui.working-area.del', self.svc_del_working_area)
    self.mediator.advertise ('ui.working-area.get', self.svc_get_working_area)
    self.mediator.advertise ('ui.working-area.close', self.svc_close_working_area)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Destroy working area
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def destroy (self, uid):
    working_area = self.__working_area.pop (uid)

    # save window position
    x, y = working_area.window.get_position ()
    width, height = working_area.window.get_size ()

    config = self.mediator.call ('app.get-config', 'sdi-window-manager') or {}
    config[working_area.id] = [x, y, width, height]
    self.mediator.call ('app.set-config', 'sdi-window-manager', config)

    # destroy working area
    working_area.window.destroy ()

    uids = self.__working_area_by_id.get (working_area.id)
    uids.remove (uid)

    if uids:
      self.__working_area_by_id[id] = uids
    else:
      self.__working_area_by_id.pop (working_area.id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief service: ui.working-area.get
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_get_working_area (self, id):
    uids = self.__working_area_by_id.get (id)

    if uids:
      return self.__working_area.get (uids[0])

    return None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @service ui.working-area.new
  # @brief create a new working area
  # @return working area
  # @begin-sequence
  #   create WorkingArea
  #   return WorkingArea
  # @end-sequence
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_new_working_area (self, id):
    working_area = WorkingArea (self)
    working_area.uid = self.__next_working_area_uid
    working_area.id = id

    self.__working_area[working_area.uid] = working_area
    self.__working_area_by_id.setdefault (id, []).append (working_area.uid)
    self.__next_working_area_uid += 1

    return working_area

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief service: ui.working-area.del
  # @brief return: True - removed, False otherwise
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_del_working_area (self, id):
    uids = self.__working_area_by_id.get (id, [])

    for uid in uids:
      self.destroy (uid)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief service: ui.working-area.close
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def svc_close_working_area (self, id):
    uids = self.__working_area_by_id.get (id, [])

    for uid in uids:
      working_area = self.__working_area.get (uid)
      working_area.on_delete_event (None)
