# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# Mobius Forensic Toolkit
# Copyright (C) 2008, 2009 Eduardo Aguiar
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General
# Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program. If not, see <http://www.gnu.org/licenses/>.
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
import gtk
import gtk.gdk
import gobject
import cairo

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
REPORT_ICON, REPORT_NAME, REPORT_ID, REPORT_STATE, REPORT_OBJ = range (5)
STATE_FOLDER, STATE_NOT_MODIFIED, STATE_MODIFIED = range (3)
DND_TARGET_OPTION, DND_TARGET_STMT = range (2)
DND_TARGETS =  [('application/x-mobius-report-opt', 0, DND_TARGET_OPTION), ('application/x-mobius-report-stmt', 0, DND_TARGET_STMT)]

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Statement colors constants
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
TEXT_COLOR = gtk.gdk.Color (65535, 65535, 30000)
FOR_COLOR = gtk.gdk.Color (20000, 65535, 65535)
IF_COLOR = gtk.gdk.Color (60000, 55000, 30000)
CALL_COLOR = gtk.gdk.Color (10000, 50000, 50000)

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief StatementBlockWidget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class StatementBlockWidget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self):
    gtk.VBox.__init__ (self)
    self.set_spacing (10)
    self.connect ('drag-motion', self.on_drag_motion)
    self.connect ('drag-leave', self.on_drag_leave)
    self.connect ('drag-data-received', self.on_drag_data_received)
    self.connect_after ('expose_event', self.on_drag_highlight_expose)
    self.drag_dest_set (gtk.DEST_DEFAULT_ALL, DND_TARGETS, gtk.gdk.ACTION_COPY)
    self.add_placeholder ()

    # internal vars
    self.empty = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief add placeholder 'do nothing'
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def add_placeholder (self):
    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_OUT)
    frame.show ()
    self.pack_start (frame, False)

    label = gtk.Label ()
    label.set_markup ('<i>do nothing</i>')
    label.show ()
    frame.add (label)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief insert child widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def insert_child (self, widget, pos = -1):

    # remove placeholder, if necessary
    if self.empty:
      for child in self.get_children ():
        self.remove (child)

    # add widget
    self.pack_start (widget, False)
    if not self.empty and pos != -1:
      self.reorder_child (widget, pos)

    # change emptiness
    self.empty = False

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief remove child widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def remove_child (self, widget):
    self.remove (widget)

    # if it is empty, insert placeholder
    if not self.get_children ():
      self.add_placeholder ()
      self.empty = True

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: drag-motion
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_drag_motion (self, widget, context, x, y, timestamp, *args):
    widget.drag_unhighlight ()

    # if widget is empty, highlight placeholder
    if widget.empty:
      child = widget.get_children ()[0]
      rect = child.get_allocation ()
      self.set_data ('highlight-placeholder', rect)
      self.set_data ('highlight-y', None)

    # else, find nearest intermediate position
    else:
      rect = widget.get_allocation ()
      y = y + rect.y
 
      highlight_y = -1
      highlight_distance = 50000

      for child in widget.get_children ():
        rect = child.get_allocation ()

        d = abs (rect.y - y)
        if d < highlight_distance:
          highlight_distance = d
          highlight_y = rect.y - 5

        d = abs ((rect.y + rect.height) - y)
        if d < highlight_distance:
          highlight_distance = d
          highlight_y = rect.y + rect.height + 5

      self.set_data ('highlight-placeholder', None)
      self.set_data ('highlight-y', highlight_y)

    # redraw
    widget.queue_draw ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: drag-leave
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_drag_leave (self, widget, context, timestamp, *args):
    self.set_data ('highlight-last-y', self.get_data ('highlight-y'))
    self.set_data ('highlight-y', None)
    self.set_data ('highlight-placeholder', None)
    self.queue_draw ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: drag-data-received
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_drag_data_received (self, widget, context, x, y, selection, info, utime, *args):
    y = self.get_data ('highlight-last-y')
    self.set_data ('highlight-y', None)
    self.set_data ('highlight-last-y', None)
    self.set_data ('highlight-placeholder', None)
    self.drag_unhighlight ()
    self.queue_draw ()

    # evaluate drop position
    if self.empty:
      pos = -1

    else:
      pos = 0

      for child in self.get_children ():
        rect = child.get_allocation ()
        if rect.y < y:
          pos += 1

    # if it is an option from pallette, emit 'drag-option-received'
    if info == DND_TARGET_OPTION:
      self.emit ('drag-option-received', pos, selection.data)

    # if it is a widget, emit 'drag-stmt-received'
    elif info == DND_TARGET_STMT:
      rc = self.emit ('drag-stmt-received', pos)
      if rc:
        context.finish (False, utime)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: after expose (drag_highlight)
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_drag_highlight_expose (self, widget, event, *args):
    y = self.get_data ('highlight-y')
    placeholder = self.get_data ('highlight-placeholder')

    if y:
     rect = self.get_allocation ()
     cr = widget.window.cairo_create ()
     cr.set_source_rgb (0.0, 0.0, 1.0)
     cr.set_line_width (1.0)
     cr.move_to (rect.x + 10, y)
     cr.line_to (rect.x + rect.width - 10, y)
     cr.stroke ()

    elif placeholder:
     cr = widget.window.cairo_create ()
     cr.set_source_rgb (0.0, 0.0, 1.0)
     cr.set_line_width (1.0)
     cr.rectangle (placeholder)
     cr.stroke ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
# @brief StatementBlockWidget signals
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
gobject.type_register (StatementBlockWidget)
gobject.signal_new ('drag-option-received', StatementBlockWidget,
   gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN, (int, str))
gobject.signal_new ('drag-stmt-received', StatementBlockWidget,
   gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN, (int,))

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Report Editor widget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class ReportWidget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    gtk.VBox.__init__ (self)
    self.set_spacing (5)
    self.mediator = mediator.copy ()
    self.report = None
    self.statement_icons = {}

    self.sw = gtk.ScrolledWindow ()
    self.sw.set_policy (gtk.POLICY_AUTOMATIC, gtk.POLICY_AUTOMATIC)
    self.sw.show ()
    self.pack_start (self.sw)

    tooltips = gtk.Tooltips ()

    toolbar = gtk.Toolbar ()
    toolbar.set_style (gtk.TOOLBAR_ICONS)
    toolbar.set_tooltips (True)
    toolbar.show ()
    self.pack_start (toolbar, False, False)

    # trash bin
    toolitem = gtk.ToolButton ()
    toolitem.set_use_drag_window (True)
    toolitem.drag_dest_set (gtk.DEST_DEFAULT_ALL, [('application/x-mobius-report-stmt', 0, DND_TARGET_STMT)], gtk.gdk.ACTION_COPY)
    toolitem.connect ('drag-data-received', self.on_drop_trash_can)

    pixbuf = self.mediator.call ('ui.render-icon', TRASH_BIN_ICON, 32, 32)
    image = gtk.image_new_from_pixbuf (pixbuf)
    image.show ()

    toolitem.set_label_widget (image)
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    toolitem = gtk.SeparatorToolItem ()
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    # statement icons
    toolitem = self.build_toolitem ('for')
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    toolitem = self.build_toolitem ('if')
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    toolitem = self.build_toolitem ('call')
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    toolitem = self.build_toolitem ('output')
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    toolitem = self.build_toolitem ('verbatim')
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    toolitem = self.build_toolitem ('text')
    toolitem.show ()
    toolbar.insert (toolitem, -1)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get color for statement
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_statement_color (self, stmt):
    if stmt == 'for':
      return FOR_COLOR

    elif stmt == 'if':
      return IF_COLOR

    elif stmt == 'call':
      return CALL_COLOR

    return TEXT_COLOR

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief get statement pallette icon
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def get_statement_icon (self, stmt):

    # return icon if it is cached
    pixmap = self.statement_icons.get (stmt)
    if pixmap:
      return pixmap

    # draw icon
    width = 72
    height = 32
    depth = 24

    color = self.get_statement_color (stmt)
    r = float (color.red) / 65535.0
    g = float (color.green) / 65535.0
    b = float (color.blue) / 65535.0

    pixmap = gtk.gdk.Pixmap (None, width, height, depth)
    cr = pixmap.cairo_create ()

    # rectangle
    linear = cairo.LinearGradient (width / 2, 0, width / 2, height)
    linear.add_color_stop_rgb (0, r / 1.5, g / 1.5, b / 1.5)
    linear.add_color_stop_rgb (1, r, g, b)

    cr.set_source (linear)
    cr.set_line_width (1.0)
    cr.rectangle (0, 0, width, height)
    cr.fill ()

    cr.set_source_rgb (0.0, 0.0, 0.0)
    cr.rectangle (0, 0, width, height)

    # statement text
    cr.select_font_face ('sans', cairo.FONT_SLANT_NORMAL, cairo.FONT_WEIGHT_NORMAL)
    cr.set_font_size (12.0)
    extents = cr.text_extents (stmt)
    cr.move_to ((width - extents[2])/2, (height - extents[3])/2 + 10)
    cr.show_text (stmt)

    cr.stroke ()

    # return pixmap and terminate
    self.statement_icons[stmt] = pixmap
    return pixmap

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build dragable toolitem
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_toolitem (self, stmt):
    toolitem = gtk.ToolButton ()
    toolitem.set_use_drag_window (True)
    toolitem.drag_source_set (gtk.gdk.BUTTON1_MASK, [('application/x-mobius-report-opt', 0, 0)], gtk.gdk.ACTION_COPY)
    toolitem.connect ('drag-data-get', self.on_drag_data_get, stmt)

    # generate image for statement
    pixmap = self.get_statement_icon (stmt)
    image = gtk.image_new_from_pixmap (pixmap, None)
    image.show ()

    # set toolitem
    toolitem.set_label_widget (image)
    toolitem.drag_source_set_icon (pixmap.get_colormap (), pixmap, None)

    return toolitem

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief clear widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def clear (self):
    child = self.sw.get_child ()
    if child:
      self.sw.remove (child)
    self.report = None

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief set report
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_report (self, report):
    self.clear ()
    self.report = report

    w = self.build_statement (report)
    self.sw.add_with_viewport (w)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build statement
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_statement (self, obj):
    statement = obj.statement.replace ('-', '_')
    method = getattr (self, 'build_%s' % statement, None)

    if method:
      widget = method (obj)
      widget.obj = obj
      widget.show ()

    else:
      widget = None
 
    return widget

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build report
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_report (self, obj):
    vbox = gtk.VBox ()
    vbox.set_spacing (5)
    vbox.set_border_width (10)
    vbox.show ()

    hbox = gtk.HBox ()
    hbox.set_border_width (5)
    hbox.set_spacing (3)
    hbox.show ()
    vbox.pack_start (hbox, False)

    label = gtk.Label ()
    label.set_markup ('<b>report</b> %s <b>(</b>' % obj.name.rsplit ('.')[-1])
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.set_text (obj.args)
    entry.connect ('changed', self.on_entry_changed, obj, 'args')
    entry.show ()
    hbox.pack_start (entry)

    label = gtk.Label ()
    label.set_markup ('<b>)</b>')
    label.show ()
    hbox.pack_start (label, False)

    w = self.build_statement (obj.code)
    w.show ()
    vbox.pack_start (w)

    return vbox

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build text
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_text (self, obj):
    eventbox = gtk.EventBox ()
    eventbox.modify_bg (gtk.STATE_NORMAL, TEXT_COLOR)

    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_OUT)
    frame.show ()
    eventbox.add (frame)

    hbox = gtk.HBox ()
    hbox.set_border_width (5)
    hbox.set_spacing (3)
    hbox.show ()
    frame.add (hbox)

    label = gtk.Label ()
    label.set_markup ('<b>text</b>')
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.set_text (obj.text)
    entry.connect ('changed', self.on_entry_changed, obj, 'text')
    entry.show ()
    hbox.pack_start (entry)

    checkbutton = gtk.CheckButton ('newline')
    checkbutton.set_active (obj.newline)
    checkbutton.connect ('toggled', self.on_checkbutton_toggled, obj, 'newline')
    checkbutton.show ()
    hbox.pack_start (checkbutton, False)

    return eventbox

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build verbatim
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_verbatim (self, obj):
    eventbox = gtk.EventBox ()
    eventbox.modify_bg (gtk.STATE_NORMAL, TEXT_COLOR)

    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_OUT)
    frame.show ()
    eventbox.add (frame)

    hbox = gtk.HBox ()
    hbox.set_border_width (5)
    hbox.set_spacing (3)
    hbox.show ()
    frame.add (hbox)

    label = gtk.Label ()
    label.set_markup ('<b>verbatim</b>')
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.set_text (obj.text)
    entry.connect ('changed', self.on_entry_changed, obj, 'text')
    entry.show ()
    hbox.pack_start (entry)

    checkbutton = gtk.CheckButton ('newline')
    checkbutton.set_active (obj.newline)
    checkbutton.connect ('toggled', self.on_checkbutton_toggled, obj, 'newline')
    checkbutton.show ()
    hbox.pack_start (checkbutton, False)

    return eventbox

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build output
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_output (self, obj):
    eventbox = gtk.EventBox ()
    eventbox.modify_bg (gtk.STATE_NORMAL, TEXT_COLOR)

    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_OUT)
    frame.show ()
    eventbox.add (frame)

    hbox = gtk.HBox ()
    hbox.set_border_width (5)
    hbox.set_spacing (3)
    hbox.show ()
    frame.add (hbox)

    label = gtk.Label ()
    label.set_markup ('<b>output</b>')
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.set_text (obj.filename)
    entry.connect ('changed', self.on_entry_changed, obj, 'filename')
    entry.show ()
    hbox.pack_start (entry)

    return eventbox

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build call
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_call (self, obj):
    eventbox = gtk.EventBox ()
    eventbox.modify_bg (gtk.STATE_NORMAL, CALL_COLOR)

    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_OUT)
    frame.show ()
    eventbox.add (frame)

    hbox = gtk.HBox ()
    hbox.set_border_width (5)
    hbox.set_spacing (3)
    hbox.show ()
    frame.add (hbox)

    label = gtk.Label ()
    label.set_markup ('<b>call</b>')
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.set_text (obj.report)
    entry.connect ('changed', self.on_entry_changed, obj, 'report')
    entry.show ()
    hbox.pack_start (entry, False)

    label = gtk.Label ()
    label.set_markup ('<b>(</b>')
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.set_text (obj.args)
    entry.connect ('changed', self.on_entry_changed, obj, 'args')
    entry.show ()
    hbox.pack_start (entry)

    label = gtk.Label ()
    label.set_markup ('<b>)</b>')
    label.show ()
    hbox.pack_start (label, False)

    return eventbox

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build if
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_if (self, obj):
    eventbox = gtk.EventBox ()
    eventbox.modify_bg (gtk.STATE_NORMAL, IF_COLOR)

    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_OUT)
    frame.show ()
    eventbox.add (frame)

    vbox = gtk.VBox ()
    vbox.set_border_width (5)
    vbox.set_spacing (10)
    vbox.obj = obj
    vbox.show ()
    frame.add (vbox)

    hbox = gtk.HBox ()
    hbox.set_spacing (3)
    hbox.show ()
    vbox.pack_start (hbox, False)

    label = gtk.Label ()
    label.set_markup ('<b>if</b>')
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.set_text (obj.condition)
    entry.connect ('changed', self.on_entry_changed, obj, 'condition')
    entry.show ()
    hbox.pack_start (entry)

    w = self.build_statement (obj.if_code)
    w.show ()
    vbox.pack_start (w)

    for item in obj.elifs:
      w, w_code = self.build_if_elif (item)

      w.show ()
      vbox.pack_start (w, False)

      w_code.show ()
      vbox.pack_start (w_code, False)

    # else code
    w = self.build_statement (obj.else_code)
    w.show ()
    vbox.pack_end (w)

    # else
    hbox = gtk.HBox ()
    hbox.set_spacing (3)
    hbox.show ()
    vbox.pack_end (hbox, False)

    label = gtk.Label ()
    label.set_markup ('<b>else</b>')
    label.show ()
    hbox.pack_start (label, False)

    # 'add elif' button
    button = gtk.Button ()
    button.connect ('clicked', self.on_add_elif, vbox, obj)
    button.show ()
    hbox.pack_end (button, False)

    button_hbox = gtk.HBox ()
    button_hbox.show ()
    button.add (button_hbox)

    image = gtk.image_new_from_stock (gtk.STOCK_ADD, gtk.ICON_SIZE_MENU)
    image.show ()
    button_hbox.pack_start (image, False)

    label = gtk.Label ('elif')
    label.show ()
    button_hbox.pack_start (label, False)

    return eventbox

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build elif
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_if_elif (self, obj):
    hbox = gtk.HBox ()
    hbox.set_spacing (3)

    label = gtk.Label ()
    label.set_markup ('<b>elif</b>')
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.set_text (obj.condition)
    entry.connect ('changed', self.on_entry_changed, obj, 'condition')
    entry.show ()
    hbox.pack_start (entry)

    # 'remove elif' button
    button = gtk.Button ()
    button.show ()
    hbox.pack_end (button, False)

    button_hbox = gtk.HBox ()
    button_hbox.show ()
    button.add (button_hbox)

    image = gtk.image_new_from_stock (gtk.STOCK_REMOVE, gtk.ICON_SIZE_MENU)
    image.show ()
    button_hbox.pack_start (image, False)

    label = gtk.Label ('elif')
    label.show ()
    button_hbox.pack_start (label, False)

    w = self.build_statement (obj.code)

    button.connect ('clicked', self.on_remove_elif, hbox, w, obj)

    return hbox, w

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build for
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_for (self, obj):
    eventbox = gtk.EventBox ()
    eventbox.modify_bg (gtk.STATE_NORMAL, FOR_COLOR)

    frame = gtk.Frame ()
    frame.set_shadow_type (gtk.SHADOW_OUT)
    frame.show ()
    eventbox.add (frame)

    vbox = gtk.VBox ()
    vbox.set_border_width (5)
    vbox.set_spacing (10)
    vbox.show ()
    frame.add (vbox)

    hbox = gtk.HBox ()
    hbox.set_spacing (3)
    hbox.show ()
    vbox.pack_start (hbox, False)

    label = gtk.Label ()
    label.set_markup ('<b>for</b>')
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.set_text (obj.vars)
    entry.connect ('changed', self.on_entry_changed, obj, 'vars')
    entry.show ()
    hbox.pack_start (entry, False)

    label = gtk.Label ()
    label.set_markup ('<b>in</b>')
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.set_text (obj.values)
    entry.connect ('changed', self.on_entry_changed, obj, 'values')
    entry.show ()
    hbox.pack_start (entry)

    w = self.build_statement (obj.code)
    w.show ()
    vbox.pack_start (w)

    return eventbox

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief build block
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def build_block (self, obj):
    hbox = gtk.HBox ()

    label = gtk.Label ('    ')
    label.show ()
    hbox.pack_start (label, False)

    vbox = StatementBlockWidget ()
    vbox.set_border_width (10)
    vbox.obj = obj
    vbox.connect ('drag-option-received', self.on_drag_option_received)
    vbox.connect ('drag-stmt-received', self.on_drag_stmt_received)
    vbox.show ()
    hbox.pack_start (vbox)

    for stmt in obj.statements:
      w = self.build_statement (stmt)
      w.show ()
      vbox.insert_child (w, -1)
      self.set_widget_draggable (w)

    return hbox

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief set widget draggagle
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def set_widget_draggable (self, widget):
    widget.drag_source_set (gtk.gdk.BUTTON1_MASK, [('application/x-mobius-report-stmt', 0, DND_TARGET_STMT)], gtk.gdk.ACTION_COPY)
    widget.connect ('drag-begin', self.on_drag_begin)
    widget.connect ('drag-data-get', self.on_drag_data_get)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_entry_changed
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_entry_changed (self, widget, obj, attrid, *args):
    setattr (obj, attrid, widget.get_text ())
    self.emit ('changed')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_checkbutton_toggled
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_checkbutton_toggled (self, widget, obj, attrid, *args):
    setattr (obj, attrid, widget.get_active ())
    self.emit ('changed')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_drag_begin
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_drag_begin (self, widget, context, *args):

    # future (python-gtk 2.14):
    if hasattr (widget, 'get_snapshot'):
      pixbuf = widget.get_snapshot ()
      widget.drag_source_set_icon_pixbuf (pixbuf)

    # set generic DND icon
    else:
      pixmap = self.get_statement_icon (widget.obj.statement)
      widget.drag_source_set_icon (pixmap.get_colormap (), pixmap, None)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_drag_option_received
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_drag_option_received (self, widget, pos, stmt_id):

    # insert new statement
    stmt = self.mediator.call ('report.new-statement', stmt_id)

    if pos == -1:
      widget.obj.statements.append (stmt)
    else:
      widget.obj.statements.insert (pos, stmt)

    # create widget and insert into block widget
    stmt_widget = self.build_statement (stmt)
    stmt_widget.show ()
    widget.insert_child (stmt_widget, pos)
    self.set_widget_draggable (stmt_widget)

    # emit 'changed'
    self.emit ('changed')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_drag_stmt_received
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_drag_stmt_received (self, dest_block_widget, pos):
    source_block_widget = self.dnd_widget.parent
    stmt = self.dnd_widget.obj

    # check if dnd_widget is parent of dest_block_widget
    widget = dest_block_widget

    while widget.parent:
      if widget.parent == self.dnd_widget:
        return True
      widget = widget.parent
        
    # if source and dest widgets are the same, reorder DND widget
    if source_block_widget == dest_block_widget:
      for idx, child in enumerate (source_block_widget.obj.statements):
        if id (child) == id (stmt):
          old_pos = idx

      if pos > old_pos:
        pos = pos - 1

      source_block_widget.obj.statements.pop (old_pos)
      source_block_widget.obj.statements.insert (pos, stmt)
      source_block_widget.reorder_child (self.dnd_widget, pos)

    # otherwise, remove from source and add to dest
    else:
      if pos == -1:
        dest_block_widget.obj.statements.append (stmt)
      else:
        dest_block_widget.obj.statements.insert (pos, stmt)

      for idx, child in enumerate (source_block_widget.obj.statements):
        if id (child) == id (stmt):
          source_block_widget.obj.statements.pop (idx)

      source_block_widget.remove_child (self.dnd_widget)
      dest_block_widget.insert_child (self.dnd_widget, pos)

    # emit 'changed'
    self.emit ('changed')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_drop_trash_can
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_drop_trash_can (self, widget, context, x, y, selection, info, utime, *args):

    # remove from block widget
    block_widget = self.dnd_widget.parent
    block_widget.remove_child (self.dnd_widget)

    # remove from model
    for idx, child in enumerate (block_widget.obj.statements):
      if id (child) == id (self.dnd_widget.obj):
        block_widget.obj.statements.pop (idx)

    # emit 'changed'
    self.emit ('changed')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_drag_data_get
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_drag_data_get (self, widget, context, selection, info, utime, *args):
    if info == DND_TARGET_OPTION:
      selection.set ('application/x-mobius-report-opt', 8, args[0])

    elif info == DND_TARGET_STMT:
      selection.set ('application/x-mobius-report-opt', 8, '')
      self.dnd_widget = widget

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_add_elif
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_add_elif (self, widget, vbox, obj, *args):

    # add elif to model
    stmt = self.mediator.call ('report.new-statement', 'elif')
    obj.elifs.append (stmt)

    # add elif to if widget
    w, w_code = self.build_if_elif (stmt)

    w.show ()
    vbox.pack_start (w, False)

    w_code.show ()
    vbox.pack_start (w_code, False)

    # emit 'changed'
    self.emit ('changed')

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_remove_elif
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_remove_elif (self, widget, hbox, elif_code, obj, *args):

    # remove elif from widget
    vbox = hbox.parent

    for child in vbox.get_children ():
      if child == hbox or child == elif_code:
        vbox.remove (child)

    # remove elif from model
    for idx, e in enumerate (vbox.obj.elifs):
      if id (e) == id (obj):
        vbox.obj.elifs.pop (idx)

    # emit 'changed'
    self.emit ('changed')

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
# @brief ReportWidget signals
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
gobject.type_register (ReportWidget)
gobject.signal_new ('changed', ReportWidget,
   gobject.SIGNAL_RUN_LAST, gobject.TYPE_BOOLEAN, ())

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Widget
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Widget (gtk.VBox):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief Initialize widget
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def __init__ (self, mediator):
    gtk.VBox.__init__ (self)
    self.mediator = mediator.new_client_mediator ()
    self.set_border_width (5)
    self.set_spacing (10)
    self.show ()

    # menubar
    menubar = gtk.MenuBar ()
    menubar.show ()
    self.pack_start (menubar, False, False)

    item = gtk.MenuItem ('_File')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    self.new_menuitem = gtk.ImageMenuItem (gtk.STOCK_NEW)
    self.new_menuitem.connect ("activate", self.on_report_new)
    self.new_menuitem.set_sensitive (False)
    self.new_menuitem.show ()
    menu.append (self.new_menuitem)

    self.save_menuitem = gtk.ImageMenuItem (gtk.STOCK_SAVE)
    self.save_menuitem.connect ("activate", self.on_report_save)
    self.save_menuitem.set_sensitive (False)
    self.save_menuitem.show ()
    menu.append (self.save_menuitem)

    item = gtk.ImageMenuItem (gtk.STOCK_CLOSE)
    item.connect ("activate", self.on_extension_close)
    item.show ()
    menu.append (item)

    item = gtk.MenuItem ('_Edit')
    item.show ()
    menubar.append (item)

    menu = gtk.Menu ()
    menu.show ()
    item.set_submenu (menu)

    self.new_folder_menuitem = gtk.MenuItem ('_New folder')
    self.new_folder_menuitem.connect ("activate", self.on_report_new_folder)
    self.new_folder_menuitem.set_sensitive (False)
    self.new_folder_menuitem.show ()
    menu.append (self.new_folder_menuitem)

    self.remove_menuitem = gtk.ImageMenuItem (gtk.STOCK_REMOVE)
    self.remove_menuitem.connect ("activate", self.on_report_remove)
    self.remove_menuitem.set_sensitive (False)
    self.remove_menuitem.show ()
    menu.append (self.remove_menuitem)

    # toolbar
    self.tooltips = gtk.Tooltips ()

    toolbar = gtk.Toolbar ()
    toolbar.set_style (gtk.TOOLBAR_ICONS)
    toolbar.set_tooltips (True)
    toolbar.show ()
    self.pack_start (toolbar, False, False)

    self.new_toolitem = gtk.ToolButton (gtk.STOCK_NEW)
    self.new_toolitem.set_sensitive (False)
    self.new_toolitem.connect ("clicked", self.on_report_new)
    self.new_toolitem.show ()
    self.new_toolitem.set_tooltip (self.tooltips, "create report")
    toolbar.insert (self.new_toolitem, -1)

    self.save_toolitem = gtk.ToolButton (gtk.STOCK_SAVE)
    self.save_toolitem.set_sensitive (False)
    self.save_toolitem.connect ("clicked", self.on_report_save)
    self.save_toolitem.show ()
    self.save_toolitem.set_tooltip (self.tooltips, "save report")
    toolbar.insert (self.save_toolitem, -1)

    toolitem = gtk.SeparatorToolItem ()
    toolitem.show ()
    toolbar.insert (toolitem, -1)

    self.new_folder_toolitem = gtk.ToolButton (gtk.STOCK_DIRECTORY)
    self.new_folder_toolitem.set_sensitive (False)
    self.new_folder_toolitem.connect ("clicked", self.on_report_new_folder)
    self.new_folder_toolitem.show ()
    self.new_folder_toolitem.set_tooltip (self.tooltips, "create folder")
    toolbar.insert (self.new_folder_toolitem, -1)

    self.remove_toolitem = gtk.ToolButton (gtk.STOCK_REMOVE)
    self.remove_toolitem.set_sensitive (False)
    self.remove_toolitem.connect ("clicked", self.on_report_remove)
    self.remove_toolitem.show ()
    self.remove_toolitem.set_tooltip (self.tooltips, "remove folder/report")
    toolbar.insert (self.remove_toolitem, -1)

    # hpaned
    hpaned = gtk.HPaned ()
    hpaned.show ()
    self.pack_start (hpaned)

    # treeview
    frame = gtk.Frame ()
    frame.show ()
    hpaned.pack1 (frame)

    sw = gtk.ScrolledWindow ()
    sw.set_policy (gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
    sw.show ()
    frame.add (sw)

    datastore = gtk.TreeStore (str, str, str, int, object)
    datastore.set_sort_column_id (REPORT_NAME, gtk.SORT_ASCENDING)

    self.treeview = gtk.TreeView (datastore)
    self.treeview.set_headers_visible (False)
    self.populate_treeview ()
    self.treeview.show ()
    sw.add (self.treeview)

    renderer = gtk.CellRendererPixbuf ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'stock_id', REPORT_ICON)
    self.treeview.append_column (tvcolumn)

    renderer = gtk.CellRendererText ()
    tvcolumn = gtk.TreeViewColumn ()
    tvcolumn.pack_start (renderer, True)
    tvcolumn.add_attribute (renderer, 'text', REPORT_NAME)
    self.treeview.append_column (tvcolumn)

    selection = self.treeview.get_selection ()
    selection.connect ('changed', self.on_treeview_selection_changed)

    # editor
    self.report_widget = ReportWidget (self.mediator)
    self.report_widget.connect ('changed', self.on_report_changed)
    self.report_widget.show ()
    hpaned.pack2 (self.report_widget)

    # buttons
    hbox = gtk.HBox ()
    hbox.show ()
    self.pack_start (hbox, False)

    button = gtk.Button (stock=gtk.STOCK_CLOSE)
    button.connect ('clicked', self.on_extension_close)
    button.show ()
    hbox.pack_end (button, False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief populate report treeview
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def populate_treeview (self):

    # insert root node
    datastore = self.treeview.get_model ()
    root = datastore.append (None, (gtk.STOCK_DIRECTORY, 'report', None, STATE_FOLDER, None))

    # insert folder and report nodes
    dir_iterators = {}

    for item_type, item_name in self.mediator.call ('report.iter'):
      pos = item_name.rfind ('.')

      if pos != -1:
        basename = item_name[pos + 1:]
        dirname = item_name[:pos]
      else:
        dirname = ''
        basename = item_name

      # get parent iterator
      if dirname:
        iter = dir_iterators.get (dirname)
      else:
        iter = root

      # insert folder
      if item_type == 'FOLDER':
        child = datastore.append (iter, (gtk.STOCK_DIRECTORY, basename, item_name, STATE_FOLDER, None))
        dir_iterators[item_name] = child

      # insert report
      else:
        datastore.append (iter, (gtk.STOCK_FILE, basename, item_name, STATE_NOT_MODIFIED, None))

    # expand root node
    self.treeview.expand_row (datastore.get_path (root), False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief check if there are modified reports
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def is_modified (self):
    model = self.treeview.get_model ()
    iter = model.get_iter_root ()
    is_modified = False
    pending = []

    while iter and not is_modified:
      state = model.get_value (iter, REPORT_STATE)

      if state == STATE_MODIFIED:
        is_modified = True

      else:
        child = model.iter_children (iter)
        if child:
          pending.append (child)

        iter = model.iter_next (iter)
        if not iter and pending:
          iter = pending[0]
          pending = pending[1:]

    return is_modified

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief get new report/folder full id
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def get_new_item_full_id (self):
    selection = self.treeview.get_selection ()
    model, iter = selection.get_selected ()

    if not iter:
      return None

    # show dialog to enter item ID
    dialog = gtk.Dialog ('Enter new item ID', None, gtk.DIALOG_MODAL,
      (gtk.STOCK_OK, gtk.RESPONSE_OK, gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL))
    dialog.set_position (gtk.WIN_POS_CENTER)
    dialog.set_type_hint (gtk.gdk.WINDOW_TYPE_HINT_DIALOG)
    dialog.set_border_width (10)

    hbox = gtk.HBox ()
    hbox.set_spacing (3)
    hbox.set_border_width (10)
    hbox.show ()
    dialog.vbox.pack_start (hbox)

    label = gtk.Label ('ID')
    label.show ()
    hbox.pack_start (label, False)

    entry = gtk.Entry ()
    entry.show ()
    hbox.pack_start (entry)

    rc = dialog.run ()
    item_id = entry.get_text ()

    dialog.destroy ()

    # if user pressed OK, evaluate full ID
    if rc == gtk.RESPONSE_OK and item_id:
      parent_id = model.get_value (iter, REPORT_ID)
 
      if parent_id:
        full_id = parent_id + '.' + item_id
      else:
        full_id = item_id

    else:
      full_id = None

    return full_id

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle widget stopped event
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_widget_stopped (self):

    # if there are unsaved reports, show save/ignore/cancel dialog
    if self.is_modified ():
      dialog = gtk.MessageDialog (None,
                  gtk.DIALOG_MODAL,
                  gtk.MESSAGE_QUESTION,
                  gtk.BUTTONS_YES_NO,
                  "Save changes before closing?")
      dialog.add_button (gtk.STOCK_CANCEL, gtk.RESPONSE_CANCEL)
      rc = dialog.run ()
      dialog.destroy ()

      if rc == gtk.RESPONSE_CANCEL:
        return True

      elif rc == gtk.RESPONSE_YES:
        pass # save

    # clean-up code
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief handle close button
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_extension_close (self, widget, *args):
    self.mediator.call ('ui.working-area.close', self.working_area.id)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: on_treeview_selection_changed
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_treeview_selection_changed (self, selection, *args):
    model, iter = selection.get_selected ()

    if iter:
      state = model.get_value (iter, REPORT_STATE)

      if state == STATE_FOLDER:
        self.on_folder_selected (model, iter)
      else:
        self.on_report_selected (model, iter)

    else:
      self.on_nothing_selected ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: on_nothing_selected
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_nothing_selected (self):
    self.report_widget.clear ()

    self.new_menuitem.set_sensitive (False)
    self.new_toolitem.set_sensitive (False)
    self.save_menuitem.set_sensitive (False)
    self.save_toolitem.set_sensitive (False)
    self.new_folder_menuitem.set_sensitive (False)
    self.new_folder_toolitem.set_sensitive (False)
    self.remove_menuitem.set_sensitive (False)
    self.remove_toolitem.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: on_folder_selected
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_folder_selected (self, model, iter):
    self.report_widget.clear ()

    is_removable = (model.get_path (iter) != (0, ))
    self.new_menuitem.set_sensitive (True)
    self.new_toolitem.set_sensitive (True)
    self.save_menuitem.set_sensitive (False)
    self.save_toolitem.set_sensitive (False)
    self.new_folder_menuitem.set_sensitive (True)
    self.new_folder_toolitem.set_sensitive (True)
    self.remove_menuitem.set_sensitive (is_removable)
    self.remove_toolitem.set_sensitive (is_removable)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: on_report_selected
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_report_selected (self, model, iter):

    # get report object
    report = model.get_value (iter, REPORT_OBJ)
    report_id = model.get_value (iter, REPORT_ID)
    state = model.get_value (iter, REPORT_STATE)

    if not report:
      report = self.mediator.call ('report.load', report_id)
      model.set_value (iter, REPORT_OBJ, report)

    # update window
    self.report_widget.set_report (report)

    is_modified = (state == STATE_MODIFIED)
    self.new_menuitem.set_sensitive (False)
    self.new_toolitem.set_sensitive (False)
    self.save_menuitem.set_sensitive (is_modified)
    self.save_toolitem.set_sensitive (is_modified)
    self.new_folder_menuitem.set_sensitive (False)
    self.new_folder_toolitem.set_sensitive (False)
    self.remove_menuitem.set_sensitive (True)
    self.remove_toolitem.set_sensitive (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: on_report_changed
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_report_changed (self, widget, *args):
    model, iter = self.treeview.get_selection ().get_selected ()

    if iter:
      model.set_value (iter, REPORT_STATE, STATE_MODIFIED)
      model.set_value (iter, REPORT_ICON, gtk.STOCK_EDIT)
      self.save_menuitem.set_sensitive (True)
      self.save_toolitem.set_sensitive (True)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: on_report_save
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_report_save (self, widget, *args):
    model, iter = self.treeview.get_selection ().get_selected ()

    if iter:
      report = model.get_value (iter, REPORT_OBJ)
      report_id = model.get_value (iter, REPORT_ID)
      self.mediator.call ('report.save', report)
      model.set_value (iter, REPORT_STATE, STATE_NOT_MODIFIED)
      model.set_value (iter, REPORT_ICON, gtk.STOCK_FILE)

      self.save_menuitem.set_sensitive (False)
      self.save_toolitem.set_sensitive (False)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: on_report_new
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_report_new (self, widget, *args):
    full_id = self.get_new_item_full_id ()

    if full_id:
      item_id = full_id.rsplit ('.', 1)[-1]
      report = self.mediator.call ('report.new', full_id)

      selection = self.treeview.get_selection ()
      model, iter = selection.get_selected ()
      child = model.append (iter, (gtk.STOCK_EDIT, item_id, full_id, STATE_MODIFIED, report))
      self.treeview.expand_row (model.get_path (iter), False)
      selection.select_iter (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: on_report_new_folder
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_report_new_folder (self, widget, *args):
    full_id = self.get_new_item_full_id ()

    if full_id:
      item_id = full_id.rsplit ('.', 1)[-1]
      self.mediator.call ('report.new-folder', full_id)

      selection = self.treeview.get_selection ()
      model, iter = selection.get_selected ()
      child = model.append (iter, (gtk.STOCK_DIRECTORY, item_id, full_id, STATE_FOLDER, None))
      self.treeview.expand_row (model.get_path (iter), False)
      selection.select_iter (child)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  # @brief event: on_report_remove
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-= 
  def on_report_remove (self, widget, *args):
    selection = self.treeview.get_selection ()
    model, iter = selection.get_selected ()

    if not iter:
      return

    # show confirmation dialog
    item_id = model.get_value (iter, REPORT_ID)
    state = model.get_value (iter, REPORT_STATE)

    if state == STATE_FOLDER:
      item_type = 'folder'
    else:
      item_type = 'report'

    dialog = gtk.MessageDialog (None,
      gtk.DIALOG_MODAL,
      gtk.MESSAGE_QUESTION,
      gtk.BUTTONS_YES_NO,
      "You are about to remove %s '%s'. Are you sure?" % (item_type, item_id))

    rc = dialog.run ()
    dialog.destroy ()

    if rc != gtk.RESPONSE_YES:
      return

    # remove report/folder
    if state == STATE_FOLDER:
      self.mediator.call ('report.remove-folder', item_id)
    else:
      self.mediator.call ('report.remove', item_id)

    # update window
    has_next = model.remove (iter)

    if has_next:
      selection.select_iter (iter)

    else:
      self.on_nothing_selected ()

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Icon
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
ICON_DATA = '''
PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiIHN0YW5kYWxvbmU9Im5vIj8+Cjwh
LS0gQ3JlYXRlZCB3aXRoIElua3NjYXBlIChodHRwOi8vd3d3Lmlua3NjYXBlLm9yZy8pIC0tPgo8
c3ZnIHhtbG5zOnN2Zz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIHhtbG5zPSJodHRwOi8v
d3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkv
eGxpbmsiIHhtbG5zOmlua3NjYXBlPSJodHRwOi8vd3d3Lmlua3NjYXBlLm9yZy9uYW1lc3BhY2Vz
L2lua3NjYXBlIiB2ZXJzaW9uPSIxLjAiIHdpZHRoPSIxOTkuNjk2NzgiIGhlaWdodD0iMjY3LjAy
MDIiIGlkPSJzdmcyIiBzdHlsZT0iZGlzcGxheTogaW5saW5lOyI+CiAgPGRlZnMgaWQ9ImRlZnM0
Ij4KICAgIDxwYXR0ZXJuIHBhdHRlcm5Vbml0cz0idXNlclNwYWNlT25Vc2UiIHdpZHRoPSIzMC4w
NjYwMjAiIGhlaWdodD0iNS4xODA1Nzc4IiBpZD0iV2F2eSIgaW5rc2NhcGU6c3RvY2tpZD0iV2F2
eSI+CiAgICAgIDxwYXRoIHN0eWxlPSJmaWxsOiBibGFjazsgc3Ryb2tlOiBub25lOyIgZD0iTSA3
LjU5NywwLjA2MSBDIDUuMDc5LC0wLjE4NyAyLjY1NiwwLjMwMiAtMC4wMSwxLjc4OCBMIC0wLjAx
LDMuMDYxIEMgMi43NzMsMS40MzEgNS4xNzMsMS4wNTIgNy40NzIsMS4yODAgQyA5Ljc3MCwxLjUw
OCAxMS45NjksMi4zNjEgMTQuMjUzLDMuMjE4IEMgMTguODIwLDQuOTMxIDIzLjgwNCw2LjY3NiAz
MC4wNjYsMy4wNjEgTCAzMC4wNjIsMS43ODggQyAyMy42MjIsNS40OTcgMTkuMjQ2LDMuNzcwIDE0
LjY5MSwyLjA2MSBDIDEyLjQxMywxLjIwNyAxMC4xMTUsMC4zMTEgNy41OTcsMC4wNjEgeiAiIGlk
PSJwYXRoNDY1MSIvPgogICAgPC9wYXR0ZXJuPgogICAgPGxpbmVhckdyYWRpZW50IGlkPSJsaW5l
YXJHcmFkaWVudDMyOTUiPgogICAgICA8c3RvcCBpZD0ic3RvcDMyOTciIHN0eWxlPSJzdG9wLWNv
bG9yOiByZ2IoMjU1LCAyNTUsIDI1NSk7IHN0b3Atb3BhY2l0eTogMTsiIG9mZnNldD0iMCIvPgog
ICAgICA8c3RvcCBpZD0ic3RvcDMyOTkiIHN0eWxlPSJzdG9wLWNvbG9yOiByZ2IoMjU1LCAyNTUs
IDI1NSk7IHN0b3Atb3BhY2l0eTogMDsiIG9mZnNldD0iMSIvPgogICAgPC9saW5lYXJHcmFkaWVu
dD4KICAgIDxsaW5lYXJHcmFkaWVudCBpZD0ibGluZWFyR3JhZGllbnQ2OTAyIj4KICAgICAgPHN0
b3AgaWQ9InN0b3A2OTA0IiBzdHlsZT0ic3RvcC1jb2xvcjogcmdiKDg1LCA0NSwgMzMpOyBzdG9w
LW9wYWNpdHk6IDE7IiBvZmZzZXQ9IjAiLz4KICAgICAgPHN0b3AgaWQ9InN0b3A2OTA2IiBzdHls
ZT0ic3RvcC1jb2xvcjogcmdiKDEyMywgNjYsIDQ5KTsgc3RvcC1vcGFjaXR5OiAxOyIgb2Zmc2V0
PSIxIi8+CiAgICA8L2xpbmVhckdyYWRpZW50PgogICAgPGxpbmVhckdyYWRpZW50IGlkPSJsaW5l
YXJHcmFkaWVudDY4OTQiPgogICAgICA8c3RvcCBpZD0ic3RvcDY4OTYiIHN0eWxlPSJzdG9wLWNv
bG9yOiByZ2IoMjQ2LCAxNzMsIDIwNSk7IHN0b3Atb3BhY2l0eTogMTsiIG9mZnNldD0iMCIvPgog
ICAgICA8c3RvcCBpZD0ic3RvcDY4OTgiIHN0eWxlPSJzdG9wLWNvbG9yOiByZ2IoMTk2LCAyMSwg
OTgpOyBzdG9wLW9wYWNpdHk6IDE7IiBvZmZzZXQ9IjEiLz4KICAgIDwvbGluZWFyR3JhZGllbnQ+
CiAgICA8bGluZWFyR3JhZGllbnQgaWQ9ImxpbmVhckdyYWRpZW50Njg1NSI+CiAgICAgIDxzdG9w
IGlkPSJzdG9wNjg1NyIgc3R5bGU9InN0b3AtY29sb3I6IHJnYigyNTUsIDI1NSwgMjU1KTsgc3Rv
cC1vcGFjaXR5OiAxOyIgb2Zmc2V0PSIwIi8+CiAgICAgIDxzdG9wIGlkPSJzdG9wNjg1OSIgc3R5
bGU9InN0b3AtY29sb3I6IHJnYigyNTUsIDI1NSwgMjU1KTsgc3RvcC1vcGFjaXR5OiAwOyIgb2Zm
c2V0PSIxIi8+CiAgICA8L2xpbmVhckdyYWRpZW50PgogICAgPGZpbHRlciBpZD0iZmlsdGVyMzY2
MyI+CiAgICAgIDxmZUdhdXNzaWFuQmx1ciBpZD0iZmVHYXVzc2lhbkJsdXIzNjY1IiBzdGREZXZp
YXRpb249IjAuNDgwMDk0NTgiIGlua3NjYXBlOmNvbGxlY3Q9ImFsd2F5cyIvPgogICAgPC9maWx0
ZXI+CiAgICA8ZmlsdGVyIGlkPSJmaWx0ZXIzNjQwIj4KICAgICAgPGZlR2F1c3NpYW5CbHVyIGlk
PSJmZUdhdXNzaWFuQmx1cjM2NDIiIHN0ZERldmlhdGlvbj0iMi4xNDUyNTE3IiBpbmtzY2FwZTpj
b2xsZWN0PSJhbHdheXMiLz4KICAgIDwvZmlsdGVyPgogICAgPGZpbHRlciBoZWlnaHQ9IjEuMTM1
MTk5IiB5PSItMC4wNjc1OTk0MzEiIHdpZHRoPSIxLjYzMDkyNzEiIHg9Ii0wLjMxNTQ2MzU3IiBp
ZD0iZmlsdGVyMzcyMCI+CiAgICAgIDxmZUdhdXNzaWFuQmx1ciBpZD0iZmVHYXVzc2lhbkJsdXIz
NzIyIiBzdGREZXZpYXRpb249IjAuNTYzMzI4NTYiIGlua3NjYXBlOmNvbGxlY3Q9ImFsd2F5cyIv
PgogICAgPC9maWx0ZXI+CiAgICA8ZmlsdGVyIGlkPSJmaWx0ZXI0MzE4Ij4KICAgICAgPGZlR2F1
c3NpYW5CbHVyIGlkPSJmZUdhdXNzaWFuQmx1cjQzMjAiIHN0ZERldmlhdGlvbj0iMS4yMDM5MTk1
IiBpbmtzY2FwZTpjb2xsZWN0PSJhbHdheXMiLz4KICAgIDwvZmlsdGVyPgogICAgPGZpbHRlciBo
ZWlnaHQ9IjEuMjQ2MDYzMiIgeT0iLTAuMTIzMDMxNTkiIHdpZHRoPSIxLjU4NDg1OTQiIHg9Ii0w
LjI5MjQyOTcyIiBpZD0iZmlsdGVyNDM0MiI+CiAgICAgIDxmZUdhdXNzaWFuQmx1ciBpZD0iZmVH
YXVzc2lhbkJsdXI0MzQ0IiBzdGREZXZpYXRpb249IjEuNDYxODkxMyIgaW5rc2NhcGU6Y29sbGVj
dD0iYWx3YXlzIi8+CiAgICA8L2ZpbHRlcj4KICAgIDxyYWRpYWxHcmFkaWVudCBjeD0iLTI3My41
MDYwMSIgY3k9Ijc2Mi40NDgxOCIgcj0iMjEuOTkxOTg5IiBmeD0iLTI3My41MDYwMSIgZnk9Ijc2
Mi40NDgxOCIgaWQ9InJhZGlhbEdyYWRpZW50Njg2MSIgeGxpbms6aHJlZj0iI2xpbmVhckdyYWRp
ZW50Njg1NSIgZ3JhZGllbnRVbml0cz0idXNlclNwYWNlT25Vc2UiIGdyYWRpZW50VHJhbnNmb3Jt
PSJtYXRyaXgoMSwgMCwgMCwgMC41NjMzOCwgMCwgMzMyLjkpIi8+CiAgICA8cmFkaWFsR3JhZGll
bnQgY3g9Ii0xNzMuMTQ4MjEiIGN5PSI4MjMuNzc3OTUiIHI9IjExLjQ2MDYxNCIgZng9Ii0xNzMu
MTQ4MjEiIGZ5PSI4MjMuNzc3OTUiIGlkPSJyYWRpYWxHcmFkaWVudDY5MDAiIHhsaW5rOmhyZWY9
IiNsaW5lYXJHcmFkaWVudDY4OTQiIGdyYWRpZW50VW5pdHM9InVzZXJTcGFjZU9uVXNlIiBncmFk
aWVudFRyYW5zZm9ybT0ibWF0cml4KDAuNDY0ODkyLCAtMS44Mjc4OGUtMDYsIDMuNTk3OTNlLTA2
LCAwLjkxNTA3OCwgLTkyLjY1NTksIDY5Ljk1NykiLz4KICAgIDxsaW5lYXJHcmFkaWVudCB4MT0i
LTIwNy4zODgyIiB5MT0iODk2LjUzMzU3IiB4Mj0iLTE2Ni4wMzgyMSIgeTI9Ijg5Ni41MzM1NyIg
aWQ9ImxpbmVhckdyYWRpZW50NjkwOCIgeGxpbms6aHJlZj0iI2xpbmVhckdyYWRpZW50NjkwMiIg
Z3JhZGllbnRVbml0cz0idXNlclNwYWNlT25Vc2UiLz4KICAgIDxyYWRpYWxHcmFkaWVudCBjeD0i
MTQ5LjA2OTI5IiBjeT0iNDQuNzExMDI1IiByPSIyNi40NzgwMTQiIGZ4PSIxNDkuMDY5MjkiIGZ5
PSI0NC43MTEwMjUiIGlkPSJyYWRpYWxHcmFkaWVudDMzMDEiIHhsaW5rOmhyZWY9IiNsaW5lYXJH
cmFkaWVudDMyOTUiIGdyYWRpZW50VW5pdHM9InVzZXJTcGFjZU9uVXNlIiBncmFkaWVudFRyYW5z
Zm9ybT0ibWF0cml4KDAuOTc5MzM5LCAtMC4wMDA3Njg4NTMsIDAuMDAwODIxNjQ5LCAxLjA0NjU4
LCAyLjk5MDI1LCAtMi4wNzc0NSkiLz4KICA8L2RlZnM+CiAgPGcgdHJhbnNmb3JtPSJ0cmFuc2xh
dGUoMzQxLjk2LCAtNzM1LjcyMSkiIGlkPSJsYXllcjMiIHN0eWxlPSJkaXNwbGF5OiBpbmxpbmU7
Ii8+CiAgPGcgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoMjIxLjc1NCwgLTc5NS40MTQpIiBpZD0ibGF5
ZXIyIiBzdHlsZT0iZGlzcGxheTogaW5saW5lOyI+CiAgICA8cGF0aCBkPSJNIC0zMjIuNDQ1OTQs
OTk1LjY4NzIgQyAtMzIxLjIwNjk1LDk5NS42ODcyIC0yNzguNzcxNyw5OTEuNjYwNSAtMjc4Ljc3
MTcsOTkxLjY2MDUgQyAtMjc4Ljc3MTcsOTkxLjY2MDUgLTI5MS4xNjE1Niw5ODQuMjI2NTggLTI5
NS4xODgyNiw5ODQuMjI2NTggQyAtMjk5LjIxNDk2LDk4NC4yMjY1OCAtMzA5Ljc0NjM0LDk4NC41
MzYzMyAtMzE0LjM5MjUzLDk4Ny4zMjQwNSBDIC0zMTkuMDM4NzMsOTkwLjExMTc2IC0zMjMuMDY1
NDMsOTk1LjM3NzQ1IC0zMjIuNDQ1OTQsOTk1LjY4NzIgeiIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUo
MTIwLjIwNiwgNTkuNjkyNikiIGlkPSJwYXRoNjg5MCIgc3R5bGU9Im9wYWNpdHk6IDE7IGZpbGw6
IHJnYigwLCAwLCAwKTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IGV2ZW5vZGQ7IHN0cm9r
ZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDAuNTsgc3Ryb2tlLWxpbmVjYXA6IHJvdW5k
OyBzdHJva2UtbGluZWpvaW46IHJvdW5kOyBzdHJva2UtbWl0ZXJsaW1pdDogNDsgc3Ryb2tlLWRh
c2hhcnJheTogbm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8cGF0aCBkPSJNIC0zMjIu
NzU1NjgsODQyLjM2Mjc2IEMgLTMyMi43NTU2OCw4NDIuMzYyNzYgLTMyNy4wOTIxMyw4NDMuNjAx
NzUgLTMzMS40Mjg1OCw4NDQuMjIxMjQgQyAtMzM1Ljc2NTAzLDg0NC44NDA3MyAtMzM3LjYyMzUx
LDg0Ni4wNzk3MiAtMzM3LjYyMzUxLDg0Ni4wNzk3MiBMIC0zNDEuOTU5OTUsODcwLjg1OTQzIEMg
LTM0MC40NjYyLDg2Ny4wMDI3MSAtMzI4LjIxNTQyLDg1Mi4wNTc2OSAtMzI0LjMwNDQyLDg1NS4z
NzIxMSBDIC0zMjEuMjA2OTUsODU3Ljg1MDA4IC0zMDIuMzEyNDIsODY2LjUyMjk4IC0zMDAuNDUz
OTUsODY5LjYyMDQ0IEMgLTI5OC41OTU0Nyw4NzIuNzE3OSAtMjczLjgxNTc2LDg4NS43MjcyNSAt
MjczLjgxNTc2LDg4NS43MjcyNSBMIC0yMzcuMjY1NjksODc1LjgxNTM3IEwgLTIzMS42OTAyNiw4
NjQuNjY0NSBDIC0yMzEuNjkwMjYsODY0LjY2NDUgLTIyMi4zOTc4Nyw4NjYuNTIyOTggLTIxOC4w
NjE0Miw4NTcuMjMwNTkgQyAtMjEzLjcyNDk3LDg0Ny45MzgyIC0yMTEuODY2NDksODM4LjAyNjMx
IC0yMTEuODY2NDksODM4LjAyNjMxIEMgLTIxMS44NjY0OSw4MzguMDI2MzEgLTIwNS42NzE1Nyw4
MzYuMTY3ODQgLTIxMy4xMDU0OCw4MzYuMTY3ODQgQyAtMjIwLjUzOTM5LDgzNi4xNjc4NCAtMzIy
Ljc1NTY4LDg0Mi4zNjI3NiAtMzIyLjc1NTY4LDg0Mi4zNjI3NiB6IiB0cmFuc2Zvcm09InRyYW5z
bGF0ZSgxMjAuMjA2LCA1OS42OTI2KSIgaWQ9InBhdGg2ODYzIiBzdHlsZT0ib3BhY2l0eTogMTsg
ZmlsbDogcmdiKDEzMiwgNjQsIDMwKTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IGV2ZW5v
ZGQ7IHN0cm9rZTogbm9uZTsgc3Ryb2tlLXdpZHRoOiAwLjU7IHN0cm9rZS1saW5lY2FwOiByb3Vu
ZDsgc3Ryb2tlLWxpbmVqb2luOiByb3VuZDsgc3Ryb2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1k
YXNoYXJyYXk6IG5vbmU7IHN0cm9rZS1vcGFjaXR5OiAxOyIvPgogICAgPGcgdHJhbnNmb3JtPSJt
YXRyaXgoMC44MzY5MDcsIC0wLjE2ODkwNiwgMC4xNDA0ODQsIDAuOTQ4MDY4LCAtMjMuMjY2OSwg
MTAzLjkyMykiIGlkPSJnNDA5MCIvPgogICAgPHBhdGggZD0iTSAtMTc3LjQ2NzAzLDEwMjguODA1
OCBDIC0xNzQuOTE1MzgsMTAxMC4wNjU2IC0xNzguNDk1ODYsOTkxLjMyNTQ4IC0xNzQuMzkwNzcs
OTcyLjU4NTMgTCAtMTgyLjk1ODM2LDk1My42MzY2NSBMIC0xODcuNjkwOTQsOTg3Ljk1ODgyIEwg
LTIwMi44MTQ0OSw5ODMuMDkzMTcgQyAtMjAyLjQ0MTIzLDk2NS4yNDAyMiAtMTk5LjE0NDA3LDk0
MS41MjAxOCAtMTg4LjIyNzMxLDkzNC41MzExNCBMIC0xNzEuNDIxNiw5MzcuMjc1OTEgTCAtMTQ1
LjYyMDQyLDg5My4wMjgxNyBMIC0xMjMuNzE3NjMsOTMxLjM4MDEgTCAtMTA5LjE1MTIxLDkzOC45
NDEyIEMgLTEwNS41OTQ4Miw5NDEuMTUzNzIgLTExMy44MTE0Miw5NDguMDg5MzggLTEwNC43MzI4
Myw5NTYuMDU1MTEgQyAtMTAxLjUzMjMsOTU4Ljg2MzMxIC05NS4wODg5MSw5NTQuOTc1MzEgLTgx
LjczMzQyLDk1MC4wNTQ5OSBMIC03Ny42ODE2NCw5NjQuNjc5NjkgQyAtODQuNzQxOTMsOTY4Ljg2
MzgyIC0xMDEuMjI3MTcsOTc4Ljg1MDggLTEwOC4xODcsOTc4LjUzMzYzIEMgLTExMS4yMTQ2NSw5
NzguMzk1NjMgLTExNC40NzUyNCw5NzUuNDI3MDYgLTExNS44NTU4OSw5NzQuMzgxNjEgQyAtMTQ5
LjE3NzI1LDk0OS4xNTAxOSAtMTI3Ljc4OTI2LDk2OC4yMDI1MiAtMTI0Ljc1NzEyLDk4NS4yMzA0
MyBDIC0xMjkuMzEyNzksOTc3LjkxOTM0IC0xMjAuMzYzMTQsOTU2LjAyMTAxIC0xMTguOTc5NjIs
OTUxLjM4OTAxIEMgLTExNi43MjAyOCw5NDMuODI0ODEgLTEwNy42MDMyMSw5MzkuOTMxMDEgLTEx
MC4yODMzLDk0NC4yMjk0MSBMIC0xMjUuMDAwNjEsMTAyMy43NjA5IEwgLTE0NS4wODAyLDk4NC4w
ODg4OCBMIC0xNzcuNDY3MDMsMTAyOC44MDU4IHoiIGlkPSJwYXRoNjQzMSIgc3R5bGU9ImZpbGw6
IHJnYigyNTUsIDE3OSwgMTI4KTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IG5vbnplcm87
IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDAuOTYxNDE2OyBzdHJva2UtbGlu
ZWNhcDogcm91bmQ7IHN0cm9rZS1saW5lam9pbjogcm91bmQ7IHN0cm9rZS1taXRlcmxpbWl0OiA0
OyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTgzLjE3MzQ5MywxMDUyLjUz
MzYgQyAtODAuMzE1MjEzLDEwNjQuMjAxNSAtOTguNDY2MjM2LDEwNTkuMTU1OCAtMTE4LjQ5OTg4
LDEwNTcuOTc1MSBDIC0xMjguMzQ3ODksMTA2MC41NzkzIC0xMzkuMTI2MjMsMTA2Mi4zMDYxIC0x
MzkuMTI0MDgsMTA1NC45NzEyIEMgLTEzOS4xMjE2NywxMDQ2Ljc3OTIgLTEzMy45NTc0MiwxMDM5
LjMyODQgLTExNS43MTQ0LDEwMzkuMzI4NCBDIC05Ny40NzEzNywxMDM5LjMyODQgLTg3LjM2ODQ3
NiwxMDQyLjk2ODIgLTgzLjE3MzQ5MywxMDUyLjUzMzYgeiIgaWQ9InBhdGg2NDMzIiBzdHlsZT0i
ZmlsbDogcmdiKDgwLCA0NSwgMjIpOyBmaWxsLW9wYWNpdHk6IDE7IGZpbGwtcnVsZTogbm9uemVy
bzsgc3Ryb2tlOiByZ2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC43MzU3MTU7IHN0cm9rZS1t
aXRlcmxpbWl0OiA0OyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTE1OC41
MzczMywxMDU0LjAzODMgQyAtMTU4LjUzNzMzLDEwNjAuNTU2NCAtMTU2LjgyMDk4LDEwNjEuNDc4
NiAtMTcyLjgwMDEyLDEwNjEuNDc4NiBDIC0xODguNzc5MjYsMTA2MS40Nzg2IC0xOTUuNDk4MDQs
MTA2MS41MTM4IC0xOTUuNDk4MDQsMTA1NC45OTU3IEMgLTE5NS40OTgwNCwxMDQ4LjQ3NzYgLTE5
MC45NzI1NSwxMDQyLjU0OTMgLTE3NC45OTM0MSwxMDQyLjU0OTMgQyAtMTU5LjAxNDI3LDEwNDIu
NTQ5MyAtMTU4LjUzNzMzLDEwNDcuNTIwMiAtMTU4LjUzNzMzLDEwNTQuMDM4MyB6IiBpZD0icGF0
aDY0NTEiIHN0eWxlPSJmaWxsOiByZ2IoODAsIDQ1LCAyMik7IGZpbGwtb3BhY2l0eTogMTsgZmls
bC1ydWxlOiBub256ZXJvOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRoOiAwLjYx
NDE5MTsgc3Ryb2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1vcGFjaXR5OiAxOyIvPgogICAgPHBh
dGggZD0iTSAtMTY1Ljk3NzY3LDEwNDMuMDI4OSBDIC0xNjUuOTc3NjcsMTA0NC40MDQgLTE2OS43
MjY4NSwxMDQ1LjUyIC0xNzQuMzQ2MzcsMTA0NS41MiBDIC0xNzguOTY1ODgsMTA0NS41MiAtMTgy
LjcxNTA1LDEwNDQuNDA0IC0xODIuNzE1MDUsMTA0My4wMjg5IEMgLTE4MS41MDMwMywxMDQyLjY5
MzUgLTE2OC41NTU0NSwxMDQzLjA2NjggLTE2NS45Nzc2NywxMDQzLjAyODkgeiIgaWQ9InBhdGg2
NDU1IiBzdHlsZT0iZmlsbDogcmdiKDIxMiwgODUsIDApOyBmaWxsLW9wYWNpdHk6IDE7IGZpbGwt
cnVsZTogbm9uemVybzsgc3Ryb2tlOiByZ2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC42MTQx
OTE7IHN0cm9rZS1saW5lY2FwOiByb3VuZDsgc3Ryb2tlLWxpbmVqb2luOiByb3VuZDsgc3Ryb2tl
LW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1vcGFjaXR5OiAxOyIvPgogICAgPHBhdGggZD0iTSAtMTg1
LjU2ODYxLDEwNDIuNjI5MiBDIC0xODQuOTU4NTIsMTAzNS40MDYxIC0xNjUuMTM5NzIsMTA0OC45
MzMgLTE2NS4yMTU5OSwxMDQyLjYyOTIgQyAtMTY1LjgzNjA1LDEwMzUuNTQ5NiAtMTg1LjMzOTk2
LDEwNDguNjg4NCAtMTg1LjU2ODYxLDEwNDIuNjI5MiB6IiBpZD0icGF0aDY0NTciIHN0eWxlPSJm
aWxsOiBub25lOyBmaWxsLXJ1bGU6IGV2ZW5vZGQ7IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJv
a2Utd2lkdGg6IDEuMjI4Mzg7IHN0cm9rZS1saW5lY2FwOiBidXR0OyBzdHJva2UtbGluZWpvaW46
IG1pdGVyOyBzdHJva2UtbWl0ZXJsaW1pdDogNDsgc3Ryb2tlLWRhc2hhcnJheTogbm9uZTsgc3Ry
b2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8cGF0aCBkPSJNIC0xNjUuNjM1MDEsOTI4Ljg0MjY3IEwg
LTExOC44MjQ2LDkyNy40ODUzNCBDIC0xMDQuMjY2NDIsOTM1LjgzMzQxIC0xMTIuMzQ4ODMsOTI5
LjkzMDI2IC05OC4yNTMzOCw5NTYuMDIyMzcgQyAtMTAzLjQ5MzI5LDk1Ny45MzQ3NCAtMTA2Ljk4
Njc5LDk2NS41ODE1NyAtMTEyLjIyNjcxLDk2NC42OTQzNCBMIC0xMTQuODA0NDMsOTUxLjg1MjMx
IEwgLTEwMy4zMDQ0OSwxMDQyLjYwNyBDIC0xMjIuNTA1ODksMTA1My4xODU4IC0xNDQuNjA2MjYs
MTA3MC44MDcyIC0xNjIuODY2NTQsMTA1MC4zNzkyIEMgLTE3My4xODkxNSwxMDM4LjgzMTIgLTE5
Ni45ODE4OCwxMDQ1LjY2MTYgLTIwMS4wMDQ2NCwxMDU0LjE1NzQgQyAtMjAyLjA0MDc1LDEwNTEu
MTI1MSAtMjAwLjU5ODc2LDEwNDAuODA0OCAtMTk4LjA5NTQ4LDEwMjcuOTYxIEMgLTE5Mi43Mjgw
MSwxMDAwLjQyMTcgLTE4Mi40ODEyOSw5NjEuMjgxMjYgLTE4MS4zMjE4Myw5NTcuNTA4NDkgTCAt
MTg1LjM5Mzg4LDk3My4wMzg4NCBDIC0xOTguMTExNTgsOTczLjk3Nzg1IC0yMDAuMTE2MDMsOTY2
Ljc0MDg3IC0yMDYuODQxOTYsOTYzLjk4OTk3IEMgLTE5MS42MzQ1MSw5MzUuMTI2NjMgLTE4OC4y
MzA3OSw5MjQuNTM1NjIgLTE2NS42MzUwMSw5MjguODQyNjcgeiIgaWQ9InBhdGg2NDYxIiBzdHls
ZT0iZmlsbDogcmdiKDEzMiwgNjQsIDMwKTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IG5v
bnplcm87IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDAuOTE1ODU1OyBzdHJv
a2UtbGluZWNhcDogcm91bmQ7IHN0cm9rZS1saW5lam9pbjogcm91bmQ7IHN0cm9rZS1taXRlcmxp
bWl0OiA0OyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gMTE0Ljc2ODQ1LDYw
NC42Nzc2NCBMIDEyOC43ODU5Nyw2MDQuNjc3NjQgQyA5NC45MDA1MDUsNjE5LjU2MTgxIDEwMC41
Miw2NDkuMzE0NzEgMTA4LjIxNDk0LDY0Ny45NzEwMiBDIDEwMC4xMzEyNiw2NDcuOTAwODggOTYu
NzQ0ODQ1LDY0NC4zNDY3NSA5MS45ODk5NzUsNjQyLjM0OTczIEMgOTAuODc4ODc1LDYyNS4zODg5
MyA5Ny4zNTM4OTUsNjA3Ljc4NDE4IDExNC43Njg0NSw2MDQuNjc3NjQgeiIgdHJhbnNmb3JtPSJt
YXRyaXgoMC45MTkyNzIsIDAuMDkzODY0NSwgLTAuMDk1NTg4OCwgMC45MDI2ODksIC0yMjQuODM2
LCAzNzQuMTc5KSIgaWQ9InBhdGg2NDY1IiBzdHlsZT0ib3BhY2l0eTogMC41MTA1MjY7IGZpbGw6
IHJnYigyMjYsIDEyOCwgMTIyKTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IG5vbnplcm87
IHN0cm9rZTogcmdiKDE5MiwgNTIsIDUyKTsgc3Ryb2tlLXdpZHRoOiAxOyBzdHJva2UtbGluZWNh
cDogcm91bmQ7IHN0cm9rZS1saW5lam9pbjogcm91bmQ7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBz
dHJva2Utb3BhY2l0eTogMTsgZmlsdGVyOiB1cmwoI2ZpbHRlcjQzMTgpOyIvPgogICAgPHBhdGgg
ZD0iTSAtMjAyLjY4MjA1LDgzNS41Mjc2NyBDIC0yMDkuNjAxMjcsODcwLjczNDkgLTIwOC4wMDAz
Miw4OTAuMzQ3MTcgLTE5OS45MDAwNyw5MDcuMDIzMzkgQyAtMTkxLjQ2NTQyLDkyNC4zODgwNiAt
MTQ2LjYwMDg5LDkzNC4yNDM2MiAtMTM4LjE4ODI2LDkzNC44OTg1MSBDIC0xMjkuOTA4NjcsOTMz
Ljk2MTg3IC0xMDUuNDUwOTQsOTE3LjEwNDY4IC0xMDMuODY5NzgsOTAyLjI0ODYgTCAtOTcuNjgy
MjM1LDg0NC4xMTI1MSBDIC0xMjUuNzU1Miw3ODQuNzg5MyAtMTY1LjMwODY4LDc3Ny45NjMzIC0y
MDIuNjgyMDUsODM1LjUyNzY3IHoiIGlkPSJwYXRoNjQ3MSIgc3R5bGU9ImZpbGw6IHJnYigyNTUs
IDE3OSwgMTI4KTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ry
b2tlLXdpZHRoOiAwLjc2NjMyMnB4OyBzdHJva2UtbGluZWNhcDogYnV0dDsgc3Ryb2tlLWxpbmVq
b2luOiBtaXRlcjsgc3Ryb2tlLW9wYWNpdHk6IDE7IGRpc3BsYXk6IGlubGluZTsiLz4KICAgIDxw
YXRoIGQ9Ik0gLTEyNS40MjI3LDg1OS42MjAwNyBDIC0xMjQuODAzMiw4NjYuODQ3NDkgLTEyNC4x
ODM3MSw4NzQuMDc0OSAtMTIzLjU2NDIyLDg4MS4zMDIzMiBDIC0xMTguNTA1MDMsODgzLjI2NDA1
IC0xMTMuNDQ1ODQsODg1LjIyNTc4IC0xMDguMzg2NjUsODg3LjE4NzUxIEMgLTEwMS4yMDY0Myw4
NzkuOTE3MjkgLTEwMy4zNDc4Nyw4NzIuMTk4OTIgLTEwNC4wNTAyLDg2Mi43MTc1NCBDIC0xMDYu
NTI4MTcsODYwLjg1OTA2IC0xMDkuMDA2MTQsODU5LjAwMDU4IC0xMTEuNDg0MTEsODU3LjE0MjEg
QyAtMTE2LjY5MTMzLDg1NC45NTgwNSAtMTIwLjM2MjA2LDg1OC40NDU0MSAtMTI1LjA5Mjk4LDg1
OS42NjYwNyBMIC0xMjUuMjgxNjYsODU5LjY2MDMyIEwgLTEyNS40MjI2OSw4NTkuNjIwMDkgTCAt
MTI1LjQyMjcsODU5LjYyMDA3IHoiIGlkPSJwYXRoNjgxMiIgc3R5bGU9Im9wYWNpdHk6IDE7IGZp
bGw6IHJnYigyNTUsIDI1NSwgMjU1KTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IGV2ZW5v
ZGQ7IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDAuNTsgc3Ryb2tlLWxpbmVj
YXA6IHJvdW5kOyBzdHJva2UtbGluZWpvaW46IHJvdW5kOyBzdHJva2UtbWl0ZXJsaW1pdDogNDsg
c3Ryb2tlLWRhc2hhcnJheTogbm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8cGF0aCBk
PSJNIC0xNzYuODQwNiw4NjYuMTI0NzcgQyAtMTc3Ljg2ODMxLDg4MC43ODg1IC0xNzkuOTAzOTcs
ODc5LjQ3Nzk0IC0xNzAuMDI2MTgsODg5LjM1NTczIEMgLTE2Mi41OTIyNiw4ODcuMzk0IC0xNTUu
MTU4MzQsODg1LjQzMjI3IC0xNDcuNzI0NDIsODgzLjQ3MDUzIEMgLTE0Ni44OTg0Myw4NzYuMDM2
NjQgLTE0Ni4wNzI0NCw4NjguNjAyNzQgLTE0NS4yNDY0Niw4NjEuMTY4ODQgQyAtMTYzLjA2NzM2
LDg1Ni45MTI0MyAtMTYzLjM2ODQ3LDg1My4wMDk5NCAtMTc2Ljg0MDYsODY2LjEyNDc3IHoiIGlk
PSJwYXRoNjgxMCIgc3R5bGU9Im9wYWNpdHk6IDE7IGZpbGw6IHJnYigyNTUsIDI1NSwgMjU1KTsg
ZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IGV2ZW5vZGQ7IHN0cm9rZTogcmdiKDAsIDAsIDAp
OyBzdHJva2Utd2lkdGg6IDAuNTsgc3Ryb2tlLWxpbmVjYXA6IHJvdW5kOyBzdHJva2UtbGluZWpv
aW46IHJvdW5kOyBzdHJva2UtbWl0ZXJsaW1pdDogNDsgc3Ryb2tlLWRhc2hhcnJheTogbm9uZTsg
c3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8cGF0aCBkPSJNIDE2MS44NDQ2NSw2MTIuNjI2MzQg
QyAxNzkuMDQ1MTgsNjE5LjIwNTI4IDEzOC41NTA5Myw2MTQuNDIxMTUgMTIyLjQyNjIzLDYwMC4w
OTM3OSBDIDEwMy4wNTc2Nyw1ODIuODg0MTMgMTAzLjYyNjM3LDU1My45MDg3IDExNy4yMTc4LDUz
NS42OTIzNyBDIDExMi43MDc4LDUzMS4wMzkyNiAxMDguMjkzNDksNTg3LjIzMDc5IDE2MS44NDQ2
NSw2MTIuNjI2MzQgeiIgdHJhbnNmb3JtPSJtYXRyaXgoMS4wNTEwNSwgMCwgMCwgMC45MDc1NTYs
IC0zMTIuMTAyLCAzNzQuOTQ0KSIgaWQ9InBhdGg2NDczIiBzdHlsZT0iZmlsbDogcmdiKDI1NSwg
MjA0LCAxNzApOyBmaWxsLW9wYWNpdHk6IDE7IGZpbGwtcnVsZTogbm9uemVybzsgc3Ryb2tlOiBu
b25lOyBzdHJva2Utd2lkdGg6IDE7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2Utb3BhY2l0
eTogMTsgZmlsdGVyOiB1cmwoI2ZpbHRlcjM2NDApOyIvPgogICAgPHBhdGggZD0iTSAzMjcuMjg5
NDMsNjk1Ljc3ODMyIEEgODQuODUyODE0LDkxLjkyMzg4MiAwIDEgMSAxNTcuNTgzOCw2OTUuNzc4
MzIgQSA4NC44NTI4MTQsOTEuOTIzODgyIDAgMSAxIDMyNy4yODk0Myw2OTUuNzc4MzIgeiIgdHJh
bnNmb3JtPSJtYXRyaXgoMC4xMTMzMywgLTAuMDI0ODQ0LCAwLjAzMjc1NTUsIDAuMTQwMTkzLCAt
MjEwLjgzNiwgNzgwLjAyNikiIGlkPSJwYXRoNjQ3OSIgc3R5bGU9Im9wYWNpdHk6IDAuNzQyMTA1
OyBmaWxsOiByZ2IoMTAsIDg1LCAxMCk7IGZpbGwtb3BhY2l0eTogMTsgc3Ryb2tlOiByZ2IoMCwg
MCwgMCk7IHN0cm9rZS13aWR0aDogOS41OyBzdHJva2UtbWl0ZXJsaW1pdDogNDsgc3Ryb2tlLWRh
c2hhcnJheTogbm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDE7IGZpbHRlcjogdXJsKCNmaWx0ZXIzNjYz
KTsiLz4KICAgIDxwYXRoIGQ9Ik0gMzI3LjI4OTQzLDY5NS43NzgzMiBBIDg0Ljg1MjgxNCw5MS45
MjM4ODIgMCAxIDEgMTU3LjU4MzgsNjk1Ljc3ODMyIEEgODQuODUyODE0LDkxLjkyMzg4MiAwIDEg
MSAzMjcuMjg5NDMsNjk1Ljc3ODMyIHoiIHRyYW5zZm9ybT0ibWF0cml4KDAuMDkyNTkwMywgLTAu
MDIwMjk3NCwgMC4wMTcxNDM1LCAwLjA3MzM3NCwgLTE5Ni40NDcsIDgyMC41MzQpIiBpZD0icGF0
aDY0ODEiIHN0eWxlPSJvcGFjaXR5OiAwLjc0MjEwNTsgZmlsbDogcmdiKDAsIDAsIDApOyBmaWxs
LW9wYWNpdHk6IDE7IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDkuNTsgc3Ry
b2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1kYXNoYXJyYXk6IG5vbmU7IHN0cm9rZS1vcGFjaXR5
OiAxOyBmaWx0ZXI6IHVybCgjZmlsdGVyMzY2Myk7Ii8+CiAgICA8cGF0aCBkPSJNIDMyNy4yODk0
Myw2OTUuNzc4MzIgQSA4NC44NTI4MTQsOTEuOTIzODgyIDAgMSAxIDE1Ny41ODM4LDY5NS43Nzgz
MiBBIDg0Ljg1MjgxNCw5MS45MjM4ODIgMCAxIDEgMzI3LjI4OTQzLDY5NS43NzgzMiB6IiB0cmFu
c2Zvcm09Im1hdHJpeCgwLjA1NDYzODEsIC0wLjAxMTk3NzYsIDAuMDE0OTM3LCAwLjA2MzkzMDMs
IC0xODQuMDc3LCA4MzAuNTM1KSIgaWQ9InBhdGg2NDgzIiBzdHlsZT0ib3BhY2l0eTogMC44NjMx
NTg7IGZpbGw6IHJnYigwLCAwLCAwKTsgZmlsbC1vcGFjaXR5OiAxOyBzdHJva2U6IHJnYigwLCAw
LCAwKTsgc3Ryb2tlLXdpZHRoOiA5LjU7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFz
aGFycmF5OiBub25lOyBzdHJva2Utb3BhY2l0eTogMTsgZmlsdGVyOiB1cmwoI2ZpbHRlcjM2NjMp
OyIvPgogICAgPHBhdGggZD0iTSAtMTY1Ljc5NzQ3LDg2NC40MTQ2NSBDIC0xNjcuNzMzMzcsODY2
Ljk5NTg3IC0xNjcuMjIzMjIsODcxLjEyODY1IC0xNjguMzU1MTIsODcwLjA2NTA3IEMgLTE2OS40
ODcwMSw4NjkuMDAxNDkgLTE2OS43NTg0OSw4NjUuNDM5MTQgLTE2Ny44MjI1OSw4NjIuODU3OTMg
QyAtMTY1Ljg4NjY4LDg2MC4yNzY3MSAtMTYzLjQwODI1LDg1OS41OTUyNCAtMTYyLjI3NjM1LDg2
MC42NTg4MiBDIC0xNjEuMTQ0NDUsODYxLjcyMjQgLTE2My44NjE1Nyw4NjEuODMzNDUgLTE2NS43
OTc0Nyw4NjQuNDE0NjUgeiIgaWQ9InBhdGg2NDg1IiBzdHlsZT0ib3BhY2l0eTogMC44NjMxNTg7
IGZpbGw6IHJnYigyNTUsIDI1NSwgMjU1KTsgZmlsbC1vcGFjaXR5OiAxOyBzdHJva2U6IHJnYigy
NTUsIDI0OSwgMjQ5KTsgc3Ryb2tlLXdpZHRoOiAxLjEyMTIyOyBzdHJva2UtbWl0ZXJsaW1pdDog
NDsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8cGF0aCBkPSJNIC0xNTIuNjYzOTYsODc2LjM0
Mjk5IEMgLTE1Mi4yNjMwMiw4NzUuMDU1MDEgLTE1Mi45MDg5LDg3My42OTkyMiAtMTUyLjM4NTcs
ODczLjg1MjMyIEMgLTE1MS44NjI0OSw4NzQuMDA1NDIgLTE1MS4zNjU3OSw4NzUuMjA3MDEgLTE1
MS43NjY3Miw4NzYuNDk0OTkgQyAtMTUyLjE2NzY2LDg3Ny43ODI5NyAtMTUyLjk3NTQsODc4LjUw
OCAtMTUzLjQ5ODYxLDg3OC4zNTQ4OSBDIC0xNTQuMDIxODEsODc4LjIwMTc5IC0xNTMuMDY0OSw4
NzcuNjMwOTYgLTE1Mi42NjM5Niw4NzYuMzQyOTkgeiIgaWQ9InBhdGg2NDg3IiBzdHlsZT0ib3Bh
Y2l0eTogMC44NjMxNTg7IGZpbGw6IHJnYigyNTUsIDI1NSwgMjU1KTsgZmlsbC1vcGFjaXR5OiAx
OyBzdHJva2U6IHJnYigyNTUsIDI0OSwgMjQ5KTsgc3Ryb2tlLXdpZHRoOiAwLjQzMDc5NDsgc3Ry
b2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1vcGFjaXR5OiAxOyIvPgogICAgPHBhdGggZD0iTSAt
MTQ2LjMwNTg1LDg4My4xNTcwOSBMIC0xNzAuMzg2NzcsODg4LjQzNjA1IEwgLTE3NC40NzQ3NCw4
ODUuMTY2NjkgTCAtMTY4Ljg4NDc2LDg5MC40ODcwNyBMIC0xNDYuMzA1ODUsODgzLjE1NzA5IHoi
IGlkPSJwYXRoNjQ5MSIgc3R5bGU9ImZpbGw6IHJnYig0MywgMCwgMCk7IGZpbGwtcnVsZTogZXZl
bm9kZDsgc3Ryb2tlOiByZ2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC4xMzMxNTlweDsgc3Ry
b2tlLWxpbmVjYXA6IGJ1dHQ7IHN0cm9rZS1saW5lam9pbjogbWl0ZXI7IHN0cm9rZS1vcGFjaXR5
OiAxOyIvPgogICAgPHBhdGggZD0iTSAtMTc0Ljc5NjA3LDg1Ny45NTg3NiBDIC0xNjUuNzE0MTgs
ODUzLjI3MTc2IC0xNTUuODExNyw4NTQuNDg5NzkgLTE0OS42NjA0OCw4NTguNDAzNCIgaWQ9InBh
dGg2NTAxIiBzdHlsZT0iZmlsbDogbm9uZTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IHJn
YigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRoOiAwLjY5Nzg3NDsgc3Ryb2tlLWxpbmVjYXA6IGJ1dHQ7
IHN0cm9rZS1saW5lam9pbjogbWl0ZXI7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFz
aGFycmF5OiBub25lOyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gMzI3LjI4
OTQzLDY5NS43NzgzMiBBIDg0Ljg1MjgxNCw5MS45MjM4ODIgMCAxIDEgMTU3LjU4MzgsNjk1Ljc3
ODMyIEEgODQuODUyODE0LDkxLjkyMzg4MiAwIDEgMSAzMjcuMjg5NDMsNjk1Ljc3ODMyIHoiIHRy
YW5zZm9ybT0ibWF0cml4KC0wLjA3NjczODMsIC0wLjAyNDAyMDksIC0wLjAyMjE3OTQsIDAuMTM1
NTQ4LCAtODAuNDc2OCwgNzgxLjIzNSkiIGlkPSJwYXRoNjUwNSIgc3R5bGU9Im9wYWNpdHk6IDAu
NzQyMTA1OyBmaWxsOiByZ2IoMTAsIDg1LCAxMCk7IGZpbGwtb3BhY2l0eTogMTsgc3Ryb2tlOiBy
Z2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogOS41OyBzdHJva2UtbWl0ZXJsaW1pdDogNDsgc3Ry
b2tlLWRhc2hhcnJheTogbm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDE7IGZpbHRlcjogdXJsKCNmaWx0
ZXIzNjYzKTsiLz4KICAgIDxwYXRoIGQ9Ik0gMzI3LjI4OTQzLDY5NS43NzgzMiBBIDg0Ljg1Mjgx
NCw5MS45MjM4ODIgMCAxIDEgMTU3LjU4MzgsNjk1Ljc3ODMyIEEgODQuODUyODE0LDkxLjkyMzg4
MiAwIDEgMSAzMjcuMjg5NDMsNjk1Ljc3ODMyIHoiIHRyYW5zZm9ybT0ibWF0cml4KC0wLjA2MjY5
NDYsIC0wLjAxOTYyNDksIC0wLjAxMTYwODIsIDAuMDcwOTQyOSwgLTkwLjIyLCA4MjAuNDAxKSIg
aWQ9InBhdGg2NTA3IiBzdHlsZT0ib3BhY2l0eTogMC43NDIxMDU7IGZpbGw6IHJnYigwLCAwLCAw
KTsgZmlsbC1vcGFjaXR5OiAxOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRoOiA5
LjU7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFzaGFycmF5OiBub25lOyBzdHJva2Ut
b3BhY2l0eTogMTsgZmlsdGVyOiB1cmwoI2ZpbHRlcjM2NjMpOyIvPgogICAgPHBhdGggZD0iTSAz
MjcuMjg5NDMsNjk1Ljc3ODMyIEEgODQuODUyODE0LDkxLjkyMzg4MiAwIDEgMSAxNTcuNTgzOCw2
OTUuNzc4MzIgQSA4NC44NTI4MTQsOTEuOTIzODgyIDAgMSAxIDMyNy4yODk0Myw2OTUuNzc4MzIg
eiIgdHJhbnNmb3JtPSJtYXRyaXgoLTAuMDM2OTk2NSwgLTAuMDExNTgwOCwgLTAuMDEwMTE0Miwg
MC4wNjE4MTIxLCAtOTguNTk2LCA4MzAuMDcxKSIgaWQ9InBhdGg2NTA5IiBzdHlsZT0ib3BhY2l0
eTogMC44NjMxNTg7IGZpbGw6IHJnYigwLCAwLCAwKTsgZmlsbC1vcGFjaXR5OiAxOyBzdHJva2U6
IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRoOiA5LjU7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBz
dHJva2UtZGFzaGFycmF5OiBub25lOyBzdHJva2Utb3BhY2l0eTogMTsgZmlsdGVyOiB1cmwoI2Zp
bHRlcjM2NjMpOyIvPgogICAgPHBhdGggZD0iTSAtMTEwLjk3MzI1LDg2Mi44Mjc3OSBDIC0xMDku
NjYyNDEsODY1LjMyMzQ5IC0xMTAuMDA3ODUsODY5LjMxOTM0IC0xMDkuMjQxNDIsODY4LjI5MSBD
IC0xMDguNDc0OTksODY3LjI2MjY2IC0xMDguMjkxMTcsODYzLjgxODM0IC0xMDkuNjAyMDEsODYx
LjMyMjY1IEMgLTExMC45MTI4NCw4NTguODI2OTYgLTExMi41OTEwNCw4NTguMTY4MDYgLTExMy4z
NTc0Nyw4NTkuMTk2NDEgQyAtMTE0LjEyMzksODYwLjIyNDc1IC0xMTIuMjg0MDgsODYwLjMzMjEx
IC0xMTAuOTczMjUsODYyLjgyNzc5IHoiIGlkPSJwYXRoNjUxMSIgc3R5bGU9Im9wYWNpdHk6IDAu
ODYzMTU4OyBmaWxsOiByZ2IoMjU1LCAyNTUsIDI1NSk7IGZpbGwtb3BhY2l0eTogMTsgc3Ryb2tl
OiByZ2IoMjU1LCAyNDksIDI0OSk7IHN0cm9rZS13aWR0aDogMC45MDcyMDk7IHN0cm9rZS1taXRl
cmxpbWl0OiA0OyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTExOS44NjYy
LDg3NC4zNjA5MSBDIC0xMjAuMTM3NjksODczLjExNTYgLTExOS43MDAzNSw4NzEuODA0NzQgLTEy
MC4wNTQ2Miw4NzEuOTUyNzYgQyAtMTIwLjQwODg5LDg3Mi4xMDA4IC0xMjAuNzQ1MjIsODczLjI2
MjU3IC0xMjAuNDczNzQsODc0LjUwNzg4IEMgLTEyMC4yMDIyNSw4NzUuNzUzMTggLTExOS42NTUz
Miw4NzYuNDU0MTkgLTExOS4zMDEwNSw4NzYuMzA2MTUgQyAtMTE4Ljk0Njc3LDg3Ni4xNTgxMiAt
MTE5LjU5NDcyLDg3NS42MDYyIC0xMTkuODY2Miw4NzQuMzYwOTEgeiIgaWQ9InBhdGg2NTEzIiBz
dHlsZT0ib3BhY2l0eTogMC44NjMxNTg7IGZpbGw6IHJnYigyNTUsIDI1NSwgMjU1KTsgZmlsbC1v
cGFjaXR5OiAxOyBzdHJva2U6IHJnYigyNTUsIDI0OSwgMjQ5KTsgc3Ryb2tlLXdpZHRoOiAwLjM0
ODU2Nzsgc3Ryb2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1vcGFjaXR5OiAxOyIvPgogICAgPHBh
dGggZD0iTSAtMTI0LjE3MTQsODgwLjk0OTI0IEwgLTEwNy44NjU3NSw4ODYuMDUzMjkgTCAtMTA1
LjA5NzcsODgyLjg5MjI2IEwgLTEwOC44ODI3OSw4ODguMDM2MzUgTCAtMTI0LjE3MTQsODgwLjk0
OTI0IHoiIGlkPSJwYXRoNjUxNyIgc3R5bGU9ImZpbGw6IHJnYig0MywgMCwgMCk7IGZpbGwtcnVs
ZTogZXZlbm9kZDsgc3Ryb2tlOiByZ2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC4xMDc3NDNw
eDsgc3Ryb2tlLWxpbmVjYXA6IGJ1dHQ7IHN0cm9rZS1saW5lam9pbjogbWl0ZXI7IHN0cm9rZS1v
cGFjaXR5OiAxOyIvPgogICAgPHBhdGggZD0iTSAtMTA3LjQ1ODExLDg1Ny41NTYwOSBDIC0xMTMu
NjA3NjMsODUzLjAyNDM5IC0xMjAuMzEyNzksODU0LjIwMjA2IC0xMjQuNDc3OSw4NTcuOTg1OTki
IGlkPSJwYXRoNjUyNyIgc3R5bGU9ImZpbGw6IG5vbmU7IGZpbGwtcnVsZTogZXZlbm9kZDsgc3Ry
b2tlOiByZ2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC41NjQ2Njg7IHN0cm9rZS1saW5lY2Fw
OiBidXR0OyBzdHJva2UtbGluZWpvaW46IG1pdGVyOyBzdHJva2UtbWl0ZXJsaW1pdDogNDsgc3Ry
b2tlLWRhc2hhcnJheTogbm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8cGF0aCBkPSJN
IC0xOTkuNjQ0MDYsODkzLjY0MDc2IEMgLTIwMy4wMjk4NCw4OTMuODIxMzUgLTIwNC44NzU5Miw4
OTIuNTc1OTIgLTIwNy4zODY2Niw4OTAuMjYzMzQgQyAtMjE0LjMzMTQ3LDg3OC43MDU4NyAtMjEy
LjA4NTM1LDg2OS44NDI3MSAtMjA0LjExOTg5LDg2NS43OTExNCBNIC0yMDEuMzgxNzUsODcxLjYy
NjYxIEMgLTIwNi43NzU1NSw4NzQuMDY2NDggLTIwNy42Nzg5Miw4ODUuMzA3OTggLTIwMC45MDE3
MSw4ODYuMDkyMTQgQyAtMTk5LjAwNzY4LDg4Ni4zMTEyOSAtMTk3LjY2OTE0LDg4Ny41NDQwMyAt
MTk3LjA4MDksODg1Ljc1NjA4IEMgLTE5Ni4yNDYwMiw4ODMuMjE4NDYgLTIwMC42Nzg4Niw4ODMu
OTM2NDEgLTIwMS4yNTAwNiw4ODEuMzY2NzMiIGlkPSJwYXRoNjUzMSIgc3R5bGU9ImZpbGw6IHJn
YigyNTUsIDE3OSwgMTI4KTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IG5vbnplcm87IHN0
cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDAuNzgwNzA4OyBzdHJva2UtbGluZWNh
cDogcm91bmQ7IHN0cm9rZS1saW5lam9pbjogcm91bmQ7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBz
dHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTE5My4xNzc4Myw4NjMuMjczNTgg
QyAtMTg2LjcyOTM1LDg0Mi40OTQ4OSAtMTYzLjkxNDMyLDgyMi4xNjUzOCAtMTQ2Ljk1NTMxLDg1
NS4zNzU1MyBDIC0xNjEuNTExNTIsODQzLjA0MjkgLTE4NS4wMzk2OCw4NTMuMzQ3NDkgLTE5My4x
Nzc4Myw4NjMuMjczNTggeiIgaWQ9InBhdGg2NTMzIiBzdHlsZT0ib3BhY2l0eTogMC45MzsgZmls
bDogcmdiKDI1NSwgMjU1LCAyNTUpOyBmaWxsLW9wYWNpdHk6IDE7IHN0cm9rZTogcmdiKDAsIDAs
IDApOyBzdHJva2Utd2lkdGg6IDAuNTg1Nzc4OyBzdHJva2UtbWl0ZXJsaW1pdDogNDsgc3Ryb2tl
LWRhc2hhcnJheTogbm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDAuODE4MTgyOyIvPgogICAgPHBhdGgg
ZD0iTSAtMjIyLjM5Nzg3LDc3NC4yMTg1NyBDIC0yMjIuMzk3ODcsNzc0LjIxODU3IC0yMTMuNzI0
OTcsNzc2LjY5NjU0IC0yMTIuNDg1OTksNzgxLjY1MjQ4IEMgLTIxMS4yNDcsNzg2LjYwODQyIC0y
MTAuMDA4MDIsNzgyLjI3MTk3IC0yMTMuMTA1NDgsNzkyLjgwMzM1IEMgLTIxNi4yMDI5NCw4MDMu
MzM0NzIgLTIxMi40ODU5OSw4MDIuNzE1MjMgLTIxNS41ODM0NSw4MDguMjkwNjcgQyAtMjE4LjY4
MDkxLDgxMy44NjYxIC0yMjAuNTM5MzksODE0LjQ4NTU5IC0yMjAuNTM5MzksODE0LjQ4NTU5IEwg
LTIxOS4zMDA0MSw3OTEuNTY0MzYgTCAtMjIyLjM5Nzg3LDc3NC4yMTg1NyB6IiB0cmFuc2Zvcm09
InRyYW5zbGF0ZSgxMjAuMjA2LCA1OS42OTI2KSIgaWQ9InBhdGg2ODE2IiBzdHlsZT0ib3BhY2l0
eTogMTsgZmlsbDogcmdiKDIzMCwgMjMwLCAyMzApOyBmaWxsLW9wYWNpdHk6IDE7IGZpbGwtcnVs
ZTogZXZlbm9kZDsgc3Ryb2tlOiByZ2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC41OyBzdHJv
a2UtbGluZWNhcDogcm91bmQ7IHN0cm9rZS1saW5lam9pbjogcm91bmQ7IHN0cm9rZS1taXRlcmxp
bWl0OiA0OyBzdHJva2UtZGFzaGFycmF5OiBub25lOyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAg
IDxwYXRoIGQ9Ik0gLTk2Ljk0ODY3NCw4NTUuMTg2NTYgQyAtMTAxLjM4MzU5LDg0NC4zNTM1NyAt
MTE3LjE1MTM1LDgyOS4zOTgxOCAtMTMwLjI5NzYsODQ2LjYxMTA3IEMgLTEyMS4zODE3OCw4NTIu
MTAxMDUgLTEwMi4yNzc4Nyw4NDUuNzUxNzIgLTk2Ljk0ODY3NCw4NTUuMTg2NTYgeiIgaWQ9InBh
dGg2NTM1IiBzdHlsZT0ib3BhY2l0eTogMC45MzsgZmlsbDogcmdiKDI1NSwgMjU1LCAyNTUpOyBm
aWxsLW9wYWNpdHk6IDE7IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDAuNDI5
MzE5OyBzdHJva2UtbWl0ZXJsaW1pdDogNDsgc3Ryb2tlLWRhc2hhcnJheTogbm9uZTsgc3Ryb2tl
LW9wYWNpdHk6IDAuODE4MTgyOyIvPgogICAgPHBhdGggZD0iTSAxNjIuMzg2NDgsNTU4Ljc4Mzc5
IEwgMTYxLjg3MzE3LDU3OC4zMjg5NCBMIDE1Ny42NzQ4OSw1NzMuNzExOTcgTCAxNjIuMzg2NDgs
NTU4Ljc4Mzc5IHoiIHRyYW5zZm9ybT0ibWF0cml4KDEuMDc4NjksIC0wLjM2NjAyNywgMC4xNjg0
NjcsIDAuNzU3OTQ2LCAtNDA0LjIyOCwgNTE1LjE1NSkiIGlkPSJwYXRoNjU0MSIgc3R5bGU9ImZp
bGw6IHJnYigyNTUsIDIwNCwgMTcwKTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IG5vbnpl
cm87IHN0cm9rZTogbm9uZTsgc3Ryb2tlLXdpZHRoOiAxOyBzdHJva2UtbWl0ZXJsaW1pdDogNDsg
c3Ryb2tlLW9wYWNpdHk6IDE7IGZpbHRlcjogdXJsKCNmaWx0ZXIzNzIwKTsiLz4KICAgIDxwYXRo
IGQ9Ik0gLTI3NS42NzQyNCw5MzQuNjY3MTcgQyAtMjc5LjY3MTIzLDk0NC4xNzQ2MiAtMzAwLjUz
NDI1LDk1OC4xNzY2IC0zMDcuODg3ODYsOTUwLjE1NDQ5IEMgLTMxMS42MDQ4MSw5NDQuNTc5MDUg
LTMxMi44NDM4LDkzNy43NjQ2MyAtMzEyLjg0MzgsOTM3Ljc2NDYzIEwgLTMxNS4zMjE3Nyw5NTIu
NjMyNDYgQyAtMjk1LjY3OTA5LDk3MC42MDUxIC0yODcuMTk1NjIsOTUwLjE2MjgzIC0yNzQuNDM1
MjUsOTQxLjQ4MTU5IEMgLTI3MC4wOTg4LDkzOS42MjMxMSAtMjczLjE5NjI3LDkzNi41MjU2NSAt
MjczLjE5NjI3LDkzNi41MjU2NSBMIC0yNzUuNjc0MjQsOTM0LjY2NzE3IHoiIHRyYW5zZm9ybT0i
dHJhbnNsYXRlKDEyMC4yMDYsIDU5LjY5MjYpIiBpZD0icGF0aDY4MjQiIHN0eWxlPSJvcGFjaXR5
OiAxOyBmaWxsOiByZ2IoMjUwLCAyMDIsIDg4KTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6
IGV2ZW5vZGQ7IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDAuNTsgc3Ryb2tl
LWxpbmVjYXA6IHJvdW5kOyBzdHJva2UtbGluZWpvaW46IHJvdW5kOyBzdHJva2UtbWl0ZXJsaW1p
dDogNDsgc3Ryb2tlLWRhc2hhcnJheTogbm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8
cGF0aCBkPSJNIC0yMDEuNzkwMzksOTgxLjEwMDk2IEMgLTIwMy45MDQ5Niw5ODMuMTExOTEgLTIw
NS4zMzk2MSw5ODYuNTI0MDcgLTIwNS44MzY0MSw5OTAuNjEzNjkgQyAtMjA2Ljc1MDA0LDk5NS4w
MDEyOCAtMjA2Ljg5MDcxLDEwMDAuMzQ3MiAtMjA0LjIwODA5LDEwMDIuNzg5NSBDIC0yMDIuODk4
MTksMTAwMy45ODIxIC0yMDEuODEwODUsMTAwNS4zODI0IC0yMDAuMjY5MzgsMTAwNy41NjU0IEMg
LTE5OS41NTc2MSwxMDA4LjU3MzQgLTE5Ny4wOTY3NywxMDA4Ljg0MDggLTE5NC45ODU3OCwxMDA4
LjIyOTIgQyAtMTk0LjMxNzgsMTAwOC4wMzU2IC0xOTMuNDEwMTIsMTAwOC4yNCAtMTkyLjY0NDE4
LDEwMDguMTEzMyBDIC0xOTEuNDM2ODUsMTAwNy45MTM0IC0xOTAuOTcwMDQsMTAwNy4xMTIzIC0x
OTAuMTM3NjQsMTAwNi44NjgxIEMgLTE4Mi44NDU1OCwxMDA0LjcyODggLTE4OC4zNjQxLDEwMDEu
NjM3NCAtMTg5LjU5MDkzLDk5OS40NjgyOCBDIC0xOTEuOTgxMjgsOTk1LjI0MjAxIC0xOTAuODQ0
MzQsOTkwLjQ1Mjk4IC0xODcuMjY2ODQsOTk1Ljc0NjMyIEMgLTE4Ni40Mzg5Myw5OTYuOTcxMyAt
MTg3LjgyMzc1LDk5Ny4wNzY4MSAtMTg4LjA1NjM0LDk5OS40Njc5NyBDIC0xODguMTA0MDcsOTk5
Ljk1ODY1IC0xODYuNDA0MjYsMTAwMy40NTY5IC0xODQuOTk5MTQsMTAwMS43MTUzIEMgLTE4MC4x
NzcwMSw5OTUuNzM4MjggLTE4NS4wODEwMiw5OTAuMDc2NDMgLTE4Ny43NDUzMSw5ODIuNzYwNjki
IGlkPSJwYXRoNjU0NSIgc3R5bGU9ImZpbGw6IHJnYigyNTUsIDE3OSwgMTI4KTsgZmlsbC1vcGFj
aXR5OiAxOyBmaWxsLXJ1bGU6IG5vbnplcm87IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Ut
d2lkdGg6IDEuMDQ1NTc7IHN0cm9rZS1saW5lY2FwOiByb3VuZDsgc3Ryb2tlLWxpbmVqb2luOiBy
b3VuZDsgc3Ryb2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1kYXNoYXJyYXk6IG5vbmU7IHN0cm9r
ZS1vcGFjaXR5OiAxOyIvPgogICAgPHBhdGggZD0iTSAtMTk5Ljk4NTcsOTk2LjQ0MTM2IEwgLTE5
OC4yMjc5NywxMDAyLjU4ODUgTCAtMTk0Ljc5NTI2LDEwMDguMDQzNSIgaWQ9InBhdGg2NTQ3IiBz
dHlsZT0iZmlsbDogbm9uZTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IHJnYigwLCAwLCAw
KTsgc3Ryb2tlLXdpZHRoOiAxLjA3ODcycHg7IHN0cm9rZS1saW5lY2FwOiBidXR0OyBzdHJva2Ut
bGluZWpvaW46IG1pdGVyOyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTIw
My43NjMwNCw5OTUuODk0NDQgTCAtMjAyLjY1NzAxLDEwMDIuODE5MiBMIC0yMDEuMjM0NTQsMTAw
Ni4wNDcyIiBpZD0icGF0aDY1NDkiIHN0eWxlPSJmaWxsOiBub25lOyBmaWxsLXJ1bGU6IGV2ZW5v
ZGQ7IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDEuMDc4NzJweDsgc3Ryb2tl
LWxpbmVjYXA6IGJ1dHQ7IHN0cm9rZS1saW5lam9pbjogbWl0ZXI7IHN0cm9rZS1vcGFjaXR5OiAx
OyIvPgogICAgPHBhdGggZD0iTSAtMTk1LjQxNjkzLDk5Ny4xMzYxIEwgLTE5My4xODczNSwxMDAy
LjMxMzUgTCAtMTkwLjE5MDE1LDEwMDYuNjgyIEwgLTE5MC4xOTAxNSwxMDA2LjY4MiIgaWQ9InBh
dGg2NTUxIiBzdHlsZT0iZmlsbDogbm9uZTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IHJn
YigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRoOiAxLjA3ODcycHg7IHN0cm9rZS1saW5lY2FwOiBidXR0
OyBzdHJva2UtbGluZWpvaW46IG1pdGVyOyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRo
IGQ9Ik0gLTEwMS40ODU0OSw5NTYuNzU2MDIgQyAtMTA0LjE5Njk2LDk1OC40NjA0NCAtMTA2LjI1
MzEzLDk2NC4wODE3NiAtMTA3LjQzOTc2LDk2OC4wMzIxOSIgaWQ9InBhdGg2NTUzIiBzdHlsZT0i
ZmlsbDogbm9uZTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ry
b2tlLXdpZHRoOiAwLjkxNTg1NXB4OyBzdHJva2UtbGluZWNhcDogYnV0dDsgc3Ryb2tlLWxpbmVq
b2luOiBtaXRlcjsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8cGF0aCBkPSJNIC0xOTMuMjgx
NzEsODg0LjQ4ODI2IEwgLTE5Mi4wNDI3Myw4NTMuNTEzNjMgQyAtMTkyLjA0MjczLDg1My41MTM2
MyAtMTg5LjU2NDc2LDg1NC4xMzMxMiAtMTg4LjMyNTc3LDg1MC40MTYxNyBDIC0xODcuMDg2Nzks
ODQ2LjY5OTIxIC0xODMuOTg5MzIsODQ0LjIyMTI0IC0xODguMzI1NzcsODQyLjM2Mjc2IEMgLTE5
Mi42NjIyMiw4NDAuNTA0MjggLTE5Ni4zNzkxOCw4NDIuOTgyMjYgLTE5Ni4zNzkxOCw4MzMuNjg5
ODcgQyAtMTk2LjM3OTE4LDgyNC4zOTc0OCAtMTkzLjI4MTcxLDgxMC4xNDkxNCAtMTkzLjI4MTcx
LDgxMC4xNDkxNCBDIC0xOTMuMjgxNzEsODEwLjE0OTE0IC0xNzUuOTM1OTIsNzk2LjUyMDMgLTE3
MC45Nzk5OCw4MTAuMTQ5MTQgQyAtMTY2LjAyNDA0LDgyMy43Nzc5OCAtMTYzLjU0NjA3LDgzNC4z
MDkzNiAtMTcwLjk3OTk4LDgzMy42ODk4NyBDIC0xNzguNDEzODksODMzLjA3MDM3IC0xODMuMzY5
ODMsODI5LjM1MzQyIC0xODAuMjcyMzcsODM0LjkyODg1IEMgLTE3Ny4xNzQ5LDg0MC41MDQyOCAt
MTczLjQ1Nzk1LDg0MC41MDQyOCAtMTc0LjY5NjkzLDg0NC44NDA3MyBDIC0xNzUuOTM1OTIsODQ5
LjE3NzE4IC0xODIuMTMwODUsODQ5LjE3NzE4IC0xODMuMzY5ODMsODUyLjg5NDE0IEMgLTE4NC42
MDg4Miw4NTYuNjExMDkgLTE4NS44NDc4LDg2OS42MjA0NCAtMTg1LjIyODMxLDg3My45NTY4OSBD
IC0xODQuNjA4ODIsODc4LjI5MzM0IC0xODUuODQ3OCw4OTUuMDE5NjQgLTE4Ni40NjcyOSw5MDcu
NDA5NDkgQyAtMTg3LjA4Njc5LDkxOS43OTkzNSAtMTg4LjMyNTc3LDkzMi4xODkyIC0xODkuNTY0
NzYsOTQ4LjI5NjAxIEMgLTE5MC44MDM3NCw5NjQuNDAyODIgLTE5My45MDEyMSw5NzUuNTUzNjkg
LTE5Mi4wNDI3Myw5ODEuMTI5MTIgQyAtMTkwLjE4NDI1LDk4Ni43MDQ1NiAtMTgzLjk4OTMyLDk5
MS42NjA1IC0xOTEuNDIzMjQsOTg3LjMyNDA1IEMgLTE5OC44NTcxNSw5ODIuOTg3NiAtMjA4Ljc2
OTAzLDk4MC41MDk2MyAtMjA2LjkxMDU1LDk3NC45MzQxOSBDIC0yMDUuMDUyMDcsOTY5LjM1ODc2
IC0yMDIuNTc0MSw5NjMuMTYzODMgLTIwMC43MTU2Myw5NTguMjA3ODkgQyAtMTk4Ljg1NzE1LDk1
My4yNTE5NSAtMTk1Ljc1OTY4LDk0Ny42NzY1MiAtMTk3LjYxODE2LDkzOS4wMDM2MiBDIC0xOTku
NDc2NjQsOTMwLjMzMDcyIC0yMDEuMzM1MTIsOTI4LjQ3MjI0IC0yMDAuMDk2MTMsOTI0LjEzNTgg
QyAtMTk4Ljg1NzE1LDkxOS43OTkzNSAtMTk4LjIzNzY1LDkyNy4yMzMyNiAtMTk2LjM3OTE4LDkx
Mi4zNjU0MyBDIC0xOTQuNTIwNyw4OTcuNDk3NjEgLTE5NC41MjA3LDg5NC40MDAxNSAtMTk0LjUy
MDcsODk0LjQwMDE1IEwgLTE5My4yODE3MSw4ODQuNDg4MjYgeiIgdHJhbnNmb3JtPSJ0cmFuc2xh
dGUoMTIwLjIwNiwgNTkuNjkyNikiIGlkPSJwYXRoNjgzMiIgc3R5bGU9Im9wYWNpdHk6IDE7IGZp
bGw6IHVybCgjbGluZWFyR3JhZGllbnQ2OTA4KSByZ2IoMCwgMCwgMCk7IGZpbGwtb3BhY2l0eTog
MTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRo
OiAwLjU7IHN0cm9rZS1saW5lY2FwOiByb3VuZDsgc3Ryb2tlLWxpbmVqb2luOiByb3VuZDsgc3Ry
b2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1kYXNoYXJyYXk6IG5vbmU7IHN0cm9rZS1vcGFjaXR5
OiAxOyIvPgogICAgPHBhdGggZD0iTSAtODguNDgyMDE2LDk1Mi4wNjMwNiBDIC04Ny40MDcyOTYs
OTQ5LjI5MTM5IC04MS4yNzk5NjYsOTQ1LjY4MDU4IC03OC4xOTU3NjQsOTQ0LjYyMTA2IEMgLTc1
Ljg2MjQwOCw5NDMuODE5NDggLTY2LjgwNjAzMiw5NDYuNzU3NCAtNjYuNzk3OTUzLDk0Ny41OTQy
NiBDIC02Ni43ODQ1OTksOTQ4Ljk3NjE5IC02OC4xMTQ5MDQsOTUwLjczNjg1IC03MC45ODEwMTUs
OTQ5Ljc0OTIzIEMgLTc3LjY3NzY3LDk0OS40MjU5NCAtNzEuMzkyODAzLDk1Mi41NjQ0MSAtNjcu
MTA5NTkzLDk1MS4wOTg3MiBDIC02NS44NjY5MzQsOTUwLjc3MTI4IC01My44MjkyMDUsOTQ5Ljk0
MTM0IC02MC43NjU5MjgsOTUzLjY5NjM2IEMgLTU3LjA1NDcwNyw5NTMuNDQ1OCAtNTIuODMyMDc5
LDk1NS4xNDgxOCAtNTguNDgwNTI5LDk1Ny42MTc2OSBDIC01NC4xOTI3OTQsOTU4LjIwMTI4IC01
OC40NDY1LDk2MC44NjE0OSAtNTkuODE4OTUsOTYyLjE2OTIxIEMgLTU2LjI4MDMzLDk2NS44MTkw
NCAtNjkuMjEwNjgxLDk2OS4zNTg5OCAtNzYuMTAyMDAzLDk2OC43MDYwOSBDIC03OS43ODY5NzIs
OTY4LjAwMjc2IC04Mi43MjA0NTIsOTY2LjQ3NTY2IC04NC44NjA3NTYsOTY0LjUxNjM5IiBpZD0i
cGF0aDY1NTkiIHN0eWxlPSJmaWxsOiByZ2IoMjU1LCAxNzksIDEyOCk7IGZpbGwtb3BhY2l0eTog
MTsgZmlsbC1ydWxlOiBub256ZXJvOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRo
OiAxLjAwNTczOyBzdHJva2UtbGluZWNhcDogcm91bmQ7IHN0cm9rZS1saW5lam9pbjogcm91bmQ7
IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFzaGFycmF5OiBub25lOyBzdHJva2Utb3Bh
Y2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTU4Ljk5NjMzOCw5NjEuNjM5ODMgQyAtNjIuODYy
NjQ5LDk2MS45NDc4NSAtNjguNDM0MzE0LDk2NS4xNzM3IC03Mi4yNjcyMzYsOTYzLjQ3ODM1IEwg
LTcyLjI2NzIzNiw5NjMuNDc4MzUgTCAtNzIuMjY3MjM2LDk2My40NzgzNSIgaWQ9InBhdGg2NTYx
IiBzdHlsZT0iZmlsbDogbm9uZTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IHJnYigwLCAw
LCAwKTsgc3Ryb2tlLXdpZHRoOiAwLjY2MjQ2M3B4OyBzdHJva2UtbGluZWNhcDogYnV0dDsgc3Ry
b2tlLWxpbmVqb2luOiBtaXRlcjsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8cGF0aCBkPSJN
IC01Ny4xNTQzODEsOTU3LjEyNzUyIEMgLTYxLjg2ODE2NSw5NTguMjA2OSAtNjguNjMyNTk3LDk1
OS40OTMwMSAtNzAuNjAzMzczLDk1OC40NTUwMSIgaWQ9InBhdGg2NTYzIiBzdHlsZT0iZmlsbDog
bm9uZTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdp
ZHRoOiAwLjY2MjQ2M3B4OyBzdHJva2UtbGluZWNhcDogYnV0dDsgc3Ryb2tlLWxpbmVqb2luOiBt
aXRlcjsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8cGF0aCBkPSJNIC02MC41OTI4NTcsOTUz
LjkwNTIzIEMgLTY5LjA5OTI2Myw5NTQuNzU4MTMgLTY4Ljc5MzgxNiw5NTUuNjg0NDYgLTcyLjA2
MTE4Miw5NTQuNjU1ODQiIGlkPSJwYXRoNjU2NSIgc3R5bGU9ImZpbGw6IG5vbmU7IGZpbGwtcnVs
ZTogZXZlbm9kZDsgc3Ryb2tlOiByZ2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC42NjI0NjNw
eDsgc3Ryb2tlLWxpbmVjYXA6IGJ1dHQ7IHN0cm9rZS1saW5lam9pbjogbWl0ZXI7IHN0cm9rZS1v
cGFjaXR5OiAxOyIvPgogICAgPHBhdGggZD0iTSAtNzMuMTEzMDExLDk1MC4xMTQwNyBDIC03Ni4w
ODE5NDgsOTUzLjkyNTQ1IC03OS4wNTA4OTQsOTU0LjQ4MDYgLTgyLjAxOTg0LDk1NC42NTE4MiIg
aWQ9InBhdGg2NTY3IiBzdHlsZT0iZmlsbDogbm9uZTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJv
a2U6IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRoOiAwLjkxNTg1NXB4OyBzdHJva2UtbGluZWNh
cDogYnV0dDsgc3Ryb2tlLWxpbmVqb2luOiBtaXRlcjsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAg
ICA8cGF0aCBkPSJNIC03NC4yNDE1NjgsOTU1LjQ0NjkzIEwgLTc1LjE5MDE5Nyw5NjMuNDQ2MjEi
IGlkPSJwYXRoNjU2OSIgc3R5bGU9ImZpbGw6IG5vbmU7IGZpbGwtcnVsZTogZXZlbm9kZDsgc3Ry
b2tlOiByZ2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC45MTU4NTVweDsgc3Ryb2tlLWxpbmVj
YXA6IGJ1dHQ7IHN0cm9rZS1saW5lam9pbjogbWl0ZXI7IHN0cm9rZS1vcGFjaXR5OiAxOyIvPgog
ICAgPHBhdGggZD0iTSAtMzEzLjQ2MzMsNzcxLjc0MDYgQyAtMzEzLjQ2MzMsNzcxLjc0MDYgLTMx
OS4wMzg3Myw3NzkuNzk0IC0zMTUuMzIxNzcsNzg2LjYwODQyIEMgLTMxMS42MDQ4Miw3OTMuNDIy
ODQgLTMxNy4xODAyNSw3OTYuNTIwMyAtMzE1Ljk0MTI2LDgwMi43MTUyMyBDIC0zMTQuNzAyMjgs
ODA4LjkxMDE2IC0zMTIuODQzOCw4MTUuNzI0NTggLTMxNS45NDEyNiw4MTIuMDA3NjIgQyAtMzE5
LjAzODczLDgwOC4yOTA2NyAtMzE5LjY1ODIyLDgwMy42NDQ0NyAtMzIzLjM3NTE3LDgwNS41MDI5
NSBDIC0zMjcuMDkyMTMsODA3LjM2MTQzIC0zMjkuODc5ODUsODA5LjgzOTQgLTMyOS44Nzk4NSw4
MDkuODM5NCBDIC0zMjkuODc5ODUsODA5LjgzOTQgLTMzMS40Mjg1OCw4MTkuNDQxNTMgLTMzMy45
MDY1NSw4MTEuMzg4MTMgQyAtMzM2LjM4NDUzLDgwMy4zMzQ3MiAtMzM5LjQ4MTk4LDgwMi4wOTU3
NCAtMzM3LjYyMzUxLDc4OS43MDU4OSBDIC0zMzUuNzY1MDMsNzc3LjMxNjAzIC0zMzEuNDI4NTgs
Nzc2LjA3NzA1IC0zMjcuMDkyMTMsNzcyLjk3OTU4IEMgLTMyMi43NTU2OCw3NjkuODgyMTIgLTMy
NC42MTQxNiw3NTQuMzk0OCAtMzIwLjI3NzcxLDc1Ny40OTIyNyBDIC0zMTUuOTQxMjYsNzYwLjU4
OTczIC0zMTIuODQzOCw3NTguMTExNzYgLTMxMC4zNjU4Myw3NTguNzMxMjUgQyAtMzA3Ljg4Nzg2
LDc1OS4zNTA3NCAtMzA3Ljg4Nzg2LDc2NC4zMDY2OSAtMzA3Ljg4Nzg2LDc2NC4zMDY2OSBMIC0z
MTMuNDYzMyw3NzEuNzQwNiB6IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgxMjAuMjA2LCA1OS42OTI2
KSIgaWQ9InBhdGg2ODE0IiBzdHlsZT0ib3BhY2l0eTogMTsgZmlsbDogcmdiKDIzMCwgMjMwLCAy
MzApOyBmaWxsLW9wYWNpdHk6IDE7IGZpbGwtcnVsZTogZXZlbm9kZDsgc3Ryb2tlOiByZ2IoMCwg
MCwgMCk7IHN0cm9rZS13aWR0aDogMC41OyBzdHJva2UtbGluZWNhcDogcm91bmQ7IHN0cm9rZS1s
aW5lam9pbjogcm91bmQ7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFzaGFycmF5OiBu
b25lOyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTMzNy4wMDQwMSw4NDUu
NDYwMjMgQyAtMzM3LjAwNDAxLDg0NS40NjAyMyAtMzE5LjAzODczLDg1OS43MDg1NiAtMzA1LjQw
OTg5LDg2Ni41MjI5OCBDIC0yOTEuNzgxMDUsODczLjMzNzQgLTI2OS40NzkzMSw4ODIuMDEwMjkg
LTI2Ni4zODE4NSw4ODUuMTA3NzYgQyAtMjYzLjI4NDM5LDg4OC4yMDUyMiAtMjY0LjUyMzM3LDk0
MC4yNDI2IC0yNjQuNTIzMzcsOTQwLjI0MjYgTCAtMjU2LjQ2OTk3LDkzOS42MjMxMSBDIC0yNTYu
NDY5OTcsOTM5LjYyMzExIC0yNTguOTQ3OTQsODk1LjYzOTEzIC0yNTYuNDY5OTcsODg4LjgyNDcx
IEMgLTI1My45OTIsODgyLjAxMDI5IC0yNTMuMzcyNSw4ODAuNzcxMzEgLTI0Ny43OTcwNyw4Nzcu
MDU0MzUgQyAtMjQyLjIyMTYzLDg3My4zMzc0IC0yNDEuNjAyMTQsODcwLjIzOTkzIC0yMzEuNjkw
MjYsODY3LjE0MjQ3IEMgLTIxMS44NTU4NSw4NjMuMTAxNDEgLTIxMi4xMDM5MSw4NTQuMDE5MSAt
MjEwLjAwODAyLDgzNi4xNjc4NCBMIC0yMTUuNTgzNDUsODM2Ljc4NzMzIEMgLTIxOC41Mjc3OSw4
NTcuODE0OTggLTIxNi43NjkyMyw4NTcuMzc3ODkgLTIzNC4xNjgyMyw4NjQuMDQ1MDEgQyAtMjQw
Ljk4MjY1LDg2Ny4xNDI0NyAtMjUxLjUxNDAyLDg3Mi43MTc5IC0yNTUuODUwNDcsODc4LjI5MzM0
IEMgLTI2MC4xODY5Miw4ODMuODY4NzcgLTI2My4yODQzOSw4ODIuMDEwMjkgLTI2NS43NjIzNiw4
NzkuNTMyMzIgQyAtMjY4LjI0MDMzLDg3Ny4wNTQzNSAtMjk3LjM1NjQ4LDg2NS45MDM0OCAtMzA2
LjY0ODg3LDg2MS41NjcwNCBDIC0zMTUuOTQxMjYsODU3LjIzMDU5IC0zMzAuMTg5Niw4NDQuMjIx
MjQgLTMzMC4xODk2LDg0NC4yMjEyNCBMIC0zMzcuMDA0MDEsODQ1LjQ2MDIzIHoiIHRyYW5zZm9y
bT0idHJhbnNsYXRlKDEyMC4yMDYsIDU5LjY5MjYpIiBpZD0icGF0aDY4MjAiIHN0eWxlPSJvcGFj
aXR5OiAxOyBmaWxsOiByZ2IoMjUwLCAyMDIsIDg4KTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1
bGU6IGV2ZW5vZGQ7IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDAuNTsgc3Ry
b2tlLWxpbmVjYXA6IHJvdW5kOyBzdHJva2UtbGluZWpvaW46IHJvdW5kOyBzdHJva2UtbWl0ZXJs
aW1pdDogNDsgc3Ryb2tlLWRhc2hhcnJheTogbm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAg
ICA8cGF0aCBkPSJNIC0yNDkuNjU1NTUsOTMzLjQyODE5IEMgLTI0My43MzMzNiw5NDEuMTE2IC0y
MzUuOTgyNSw5NDcuMjc4NzIgLTIyNi4xMTQ4Myw5NDguMjk2MDEgTCAtMjI1LjQ5NTMzLDk1Mi4w
MTI5NyBDIC0yMzAuODUyOTksOTU0LjQzNzM4IC0yNDQuOTc0NjgsOTQzLjc4OTIxIC0yNDkuNjU1
NTUsOTM4LjM4NDEzIEMgLTI1MS41MTQwMiw5MzUuOTA2MTYgLTI1MS41MTQwMiw5MzQuMDQ3Njgg
LTI0OS42NTU1NSw5MzMuNDI4MTkgeiIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoMTIwLjIwNiwgNTku
NjkyNikiIGlkPSJwYXRoNjgyNiIgc3R5bGU9Im9wYWNpdHk6IDE7IGZpbGw6IHJnYigyNTAsIDIw
MiwgODgpOyBmaWxsLW9wYWNpdHk6IDE7IGZpbGwtcnVsZTogZXZlbm9kZDsgc3Ryb2tlOiByZ2Io
MCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC41OyBzdHJva2UtbGluZWNhcDogcm91bmQ7IHN0cm9r
ZS1saW5lam9pbjogcm91bmQ7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFzaGFycmF5
OiBub25lOyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gMjEyLjg4MTcxLDYx
Ny40MjgxNyBMIDIyMC4xNjgxMSw2MTUuMDk3NDEgQyAyMjkuNjYxMDEsNjIyLjI3MTA2IDIyMy42
NjYwNSw2NjQuODgwMTMgMjI4LjI0Mzc3LDY3NS44NzI1MyBMIDIwNy40MTU2Miw2ODAuODAwODgg
TCAyMTIuODgxNzEsNjE3LjQyODE3IHoiIHRyYW5zZm9ybT0ibWF0cml4KDEuNTE0NjksIDAuMjI4
NDM2LCAtMC4yMzA2ODcsIDEuNDk5OTEsIC0zNTYuOTUsIC0yNC4zNCkiIGlkPSJwYXRoNjQ2NyIg
c3R5bGU9Im9wYWNpdHk6IDAuNDIxMDUzOyBmaWxsOiByZ2IoMjMzLCAxNzUsIDE3NSk7IGZpbGwt
b3BhY2l0eTogMTsgZmlsbC1ydWxlOiBub256ZXJvOyBzdHJva2U6IHJnYigxOTIsIDUyLCA2MSk7
IHN0cm9rZS13aWR0aDogMC42NzA2NDU7IHN0cm9rZS1saW5lY2FwOiByb3VuZDsgc3Ryb2tlLWxp
bmVqb2luOiByb3VuZDsgc3Ryb2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1vcGFjaXR5OiAxOyBm
aWx0ZXI6IHVybCgjZmlsdGVyNDM0Mik7Ii8+CiAgICA8cGF0aCBkPSJNIC0zMzcuMDA0MDEsODQ1
LjQ2MDIzIEMgLTMzNy4wMDQwMSw4NDUuNDYwMjMgLTMxOS4wMzg3Myw4NTkuNzA4NTYgLTMwNS40
MDk4OSw4NjYuNTIyOTggQyAtMjkxLjc4MTA1LDg3My4zMzc0IC0yNjkuNDc5MzEsODgyLjAxMDI5
IC0yNjYuMzgxODUsODg1LjEwNzc2IEMgLTI2My4yODQzOSw4ODguMjA1MjIgLTI2NC41MjMzNyw5
NDAuMjQyNiAtMjY0LjUyMzM3LDk0MC4yNDI2IEwgLTI1Ni40Njk5Nyw5MzkuNjIzMTEgQyAtMjU2
LjQ2OTk3LDkzOS42MjMxMSAtMjU4Ljk0Nzk0LDg5NS42MzkxMyAtMjU2LjQ2OTk3LDg4OC44MjQ3
MSBDIC0yNTMuOTkyLDg4Mi4wMTAyOSAtMjUzLjM3MjUsODgwLjc3MTMxIC0yNDcuNzk3MDcsODc3
LjA1NDM1IEMgLTI0Mi4yMjE2Myw4NzMuMzM3NCAtMjQxLjYwMjE0LDg3MC4yMzk5MyAtMjMxLjY5
MDI2LDg2Ny4xNDI0NyBDIC0yMTEuODU1ODUsODYzLjEwMTQxIC0yMTIuMTAzOTEsODU0LjAxOTEg
LTIxMC4wMDgwMiw4MzYuMTY3ODQgTCAtMjE1LjU4MzQ1LDgzNi43ODczMyBDIC0yMTguNTI3Nzks
ODU3LjgxNDk4IC0yMTYuNzY5MjMsODU3LjM3Nzg5IC0yMzQuMTY4MjMsODY0LjA0NTAxIEMgLTI0
MC45ODI2NSw4NjcuMTQyNDcgLTI1MS41MTQwMiw4NzIuNzE3OSAtMjU1Ljg1MDQ3LDg3OC4yOTMz
NCBDIC0yNjAuMTg2OTIsODgzLjg2ODc3IC0yNjMuMjg0MzksODgyLjAxMDI5IC0yNjUuNzYyMzYs
ODc5LjUzMjMyIEMgLTI2OC4yNDAzMyw4NzcuMDU0MzUgLTI5Ny4zNTY0OCw4NjUuOTAzNDggLTMw
Ni42NDg4Nyw4NjEuNTY3MDQgQyAtMzE1Ljk0MTI2LDg1Ny4yMzA1OSAtMzMwLjE4OTYsODQ0LjIy
MTI0IC0zMzAuMTg5Niw4NDQuMjIxMjQgTCAtMzM3LjAwNDAxLDg0NS40NjAyMyB6IiB0cmFuc2Zv
cm09InRyYW5zbGF0ZSgxMjAuMjA2LCA1OS42OTI2KSIgaWQ9InBhdGg0NzQxIiBzdHlsZT0ib3Bh
Y2l0eTogMTsgZmlsbDogdXJsKCNXYXZ5KSByZ2IoMCwgMCwgMCk7IGZpbGwtb3BhY2l0eTogMTsg
ZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRoOiAw
LjU7IHN0cm9rZS1saW5lY2FwOiByb3VuZDsgc3Ryb2tlLWxpbmVqb2luOiByb3VuZDsgc3Ryb2tl
LW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1kYXNoYXJyYXk6IG5vbmU7IHN0cm9rZS1vcGFjaXR5OiAx
OyIvPgogICAgPHBhdGggZD0iTSAtMzI3LjcxMTYyLDkwMi40NTM1NSBMIC0zMzAuODA5MDksOTEw
LjUwNjk2IEMgLTMzMC44MDkwOSw5MTAuNTA2OTYgLTMyMy45OTQ2Nyw5MTYuMDgyMzkgLTMxOC40
MTkyMyw5MTYuNzAxODggQyAtMzEwLjEzNDY2LDkxNi44MzUwOCAtMzA1LjEzMjA1LDkyNC4wMjE0
NiAtMzA0LjE3MDksOTIxLjAzODMzIEMgLTMwMi45MzE5Miw5MTUuNDYyOSAtMjk5LjgzNDQ1LDkx
My42MDQ0MiAtMzAyLjkzMTkyLDkxMy42MDQ0MiBDIC0zMDYuMDI5MzgsOTEzLjYwNDQyIC0zMDAu
MTQ0Miw5MTQuODQzNDEgLTMwOS43NDYzNCw5MTIuOTg0OTMgQyAtMzE5LjI2NjI1LDkxMS4xNDIz
NyAtMzE3LjE4MDI1LDkwOC42NDg0OCAtMzIyLjc1NTY4LDkwNi4xNzA1MSBDIC0zMjguMzMxMTIs
OTAzLjY5MjU0IC0zMjkuNTcwMSw5MDIuNDUzNTUgLTMyNy43MTE2Miw5MDIuNDUzNTUgeiIgdHJh
bnNmb3JtPSJ0cmFuc2xhdGUoMTIwLjIwNiwgNTkuNjkyNikiIGlkPSJwYXRoNjgyOCIgc3R5bGU9
Im9wYWNpdHk6IDE7IGZpbGw6IHJnYigyNTAsIDIwMiwgODgpOyBmaWxsLW9wYWNpdHk6IDE7IGZp
bGwtcnVsZTogZXZlbm9kZDsgc3Ryb2tlOiByZ2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMTsg
c3Ryb2tlLWxpbmVjYXA6IHJvdW5kOyBzdHJva2UtbGluZWpvaW46IHJvdW5kOyBzdHJva2UtbWl0
ZXJsaW1pdDogNDsgc3Ryb2tlLWRhc2hhcnJheTogbm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+
CiAgICA8cGF0aCBkPSJNIC0yMTYuMjAyOTQsODk1LjAxOTY0IEMgLTIxNi4yMDI5NCw4OTUuMDE5
NjQgLTIxOC4wNjE0Miw4OTEuOTIyMTggLTIyMS43NzgzOCw4OTMuNzgwNjUgQyAtMjI1LjQ5NTMz
LDg5NS42MzkxMyAtMjI2LjczNDMyLDg5OS4zNTYwOSAtMjI4LjU5MjgsOTAxLjgzNDA2IEMgLTIz
MC40NTEyNyw5MDQuMzEyMDMgLTIzMi45MjkyNCw5MDYuMTcwNTEgLTIzMi45MjkyNCw5MDYuMTcw
NTEgTCAtMjMxLjA3MDc3LDkxNi4wODIzOSBDIC0yMzEuMDcwNzcsOTE2LjA4MjM5IC0yMzEuMDcw
NzcsOTE1LjQ2MjkgLTIyNC44NzU4NCw5MDguNjQ4NDggQyAtMjE4LjY4MDkxLDkwMS44MzQwNiAt
MjIxLjE1ODg4LDkwNi43OSAtMjE3LjQ0MTkzLDkwMS4yMTQ1NyBDIC0yMTMuNzI0OTcsODk1LjYz
OTEzIC0yMTMuNzI0OTcsODk1LjAxOTY0IC0yMTYuMjAyOTQsODk1LjAxOTY0IHoiIHRyYW5zZm9y
bT0idHJhbnNsYXRlKDEyMC4yMDYsIDU5LjY5MjYpIiBpZD0icGF0aDY4MzAiIHN0eWxlPSJvcGFj
aXR5OiAxOyBmaWxsOiByZ2IoMjUwLCAyMDIsIDg4KTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1
bGU6IGV2ZW5vZGQ7IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDE7IHN0cm9r
ZS1saW5lY2FwOiByb3VuZDsgc3Ryb2tlLWxpbmVqb2luOiByb3VuZDsgc3Ryb2tlLW1pdGVybGlt
aXQ6IDQ7IHN0cm9rZS1kYXNoYXJyYXk6IG5vbmU7IHN0cm9rZS1vcGFjaXR5OiAxOyIvPgogICAg
PHBhdGggZD0iTSAtMjg3LjQ0NDYsODU2LjYxMTA5IEMgLTI4Ny40NDQ2LDg1Ni42MTEwOSAtMjkx
Ljc4MTA1LDg2MC45NDc1NCAtMjg5LjMwMzA4LDg2NS45MDM0OCBDIC0yODYuODI1MTEsODcwLjg1
OTQzIC0yODIuNDg4NjYsODc5LjUzMjMyIC0yODIuNDg4NjYsODc5LjUzMjMyIEwgLTI3Ny41MzI3
Miw4NzAuODU5NDMgTCAtMjcwLjcxODMsODgwLjE1MTgyIEwgLTI2Ny4wMDEzNCw4NzUuMTk1ODcg
TCAtMjYxLjQyNTkxLDg4Ni4zNDY3NCBMIC0yNTUuODUwNDcsODgwLjE1MTgyIEwgLTI1Mi43NTMw
MSw4ODIuNjI5NzkgTCAtMjQ2LjU1ODA4LDg3MS40Nzg5MiBMIC0yNDAuOTgyNjUsODcyLjA5ODQx
IEwgLTIzNy4yNjU2OSw4NjYuNTIyOTggTCAtMjMyLjMwOTc1LDg2Ny43NjE5NiBMIC0yMzAuNDUx
MjcsODYwLjk0NzU0IEwgLTIyOC41OTI4LDg1MS42NTUxNSBMIC0yMzEuNjkwMjYsODQyLjk4MjI2
IEMgLTIzMS42OTAyNiw4NDIuOTgyMjYgLTI1MC44OTQ1Myw4NDguNTU3NjkgLTI2Mi42NjQ4OSw4
NDkuMTc3MTggQyAtMjc0LjQzNTI1LDg0OS43OTY2NyAtMjc5LjM5MTE5LDg0Ny45MzgyIC0yODIu
NDg4NjYsODUxLjAzNTY2IEMgLTI4NS41ODYxMiw4NTQuMTMzMTIgLTI4OC4wNjQwOSw4NTYuNjEx
MDkgLTI4Ny40NDQ2LDg1Ni42MTEwOSB6IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgxMjAuMjA2LCA1
OS42OTI2KSIgaWQ9InBhdGg2ODM1IiBzdHlsZT0ib3BhY2l0eTogMTsgZmlsbDogcmdiKDIzMCwg
MjMwLCAyMzApOyBmaWxsLW9wYWNpdHk6IDE7IGZpbGwtcnVsZTogZXZlbm9kZDsgc3Ryb2tlOiBy
Z2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC41OyBzdHJva2UtbGluZWNhcDogcm91bmQ7IHN0
cm9rZS1saW5lam9pbjogcm91bmQ7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFzaGFy
cmF5OiBub25lOyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxnIHRyYW5zZm9ybT0ibWF0cml4
KDAuMjEwNjA0LCAwLCAwLCAwLjMxNjk2NCwgLTIwNC42NDcsIDgwMC45MzIpIiBpZD0iZzY4MDQi
PgogICAgICA8cGF0aCBkPSJNIDMxNy41MDAwMSwzNzUuMjE5MzQgQyAyNjIuMDAwMzMsMzczLjQ4
MzYzIDIxOC45OTgyLDM5NS43NzA5MSAyMTcuMTQyODgsMzU5Ljg2MjIgQyAyMTUuMjQyNywzMjMu
MDg1MzUgMjU3LjU4NjMzLDMzNy4zNzU2OCAzMTQuNjQyODYsMzI4LjA3NjQ5IEMgMzczLjc4Mzgz
LDMxOC40Mzc1NyA0MjMuNTY2MjgsMjcxLjQ3NzMyIDQzMS40Mjc0NSwzMzUuODA4NTcgQyA0Mzku
Mjg4NjIsNDAwLjEzOTgyIDM1Mi42MDA1MSwzNzIuNTI2NTggMzE3LjUwMDAxLDM3NS4yMTkzNCB6
IiBpZD0icGF0aDY4MDYiIHN0eWxlPSJmaWxsOiByZ2IoMjU1LCAyNTUsIDI1NSk7IGZpbGwtb3Bh
Y2l0eTogMTsgZmlsbC1ydWxlOiBub256ZXJvOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ryb2tl
LXdpZHRoOiAzLjg3MDQ1OyBzdHJva2UtbGluZWNhcDogYnV0dDsgc3Ryb2tlLWxpbmVqb2luOiBt
aXRlcjsgc3Ryb2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1kYXNoYXJyYXk6IG5vbmU7IHN0cm9r
ZS1vcGFjaXR5OiAxOyIvPgogICAgICA8cGF0aCBkPSJNIDQzMS40Mjg1NywzMzUuOTMzNjIgQyA0
MjcuMTIyOTcsMzM3LjQ4Njg0IDQxNy41OTI0NywzMzYuMzgzNSA0MDkuNTMwMDUsMzM4LjM2MDUz
IEMgNDAxLjk1MzYzLDM0MC4yMTg0IDM5MC44NDA2NiwzNDYuNjQyNjMgMzg2LjUxNzE0LDM0Ni40
NjY1NyBDIDM3Ny40OTI1NywzNDYuMDk5MDYgMzcwLjc1MzU3LDM0My4xOTAwNyAzNjIuODk1Mjgs
MzQ0LjcxMDI2IEMgMzQ3LjUzNzc3LDM0Ny42ODExOCAzNDEuMTg4MTMsMzU3LjY2MDIxIDMyOS4y
ODU3MSwzNTEuNjQ3OSIgaWQ9InBhdGg2ODA4IiBzdHlsZT0iZmlsbDogbm9uZTsgZmlsbC1ydWxl
OiBldmVub2RkOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRoOiAzLjg3MDQ1OyBz
dHJva2UtbGluZWNhcDogYnV0dDsgc3Ryb2tlLWxpbmVqb2luOiBtaXRlcjsgc3Ryb2tlLW1pdGVy
bGltaXQ6IDQ7IHN0cm9rZS1kYXNoYXJyYXk6IG5vbmU7IHN0cm9rZS1vcGFjaXR5OiAxOyIvPgog
ICAgPC9nPgogICAgPHBhdGggZD0iTSAtMjY1LjE0Mjg2LDgzNi43ODczMyBDIC0yNzcuNTMyNzIs
ODQwLjUwNDI4IC0yODIuNDg4NjYsODM2LjE2Nzg0IC0yODQuMzQ3MTQsODQ2LjA3OTcyIEMgLTI4
Ni4yMDU2MSw4NTUuOTkxNiAtMzAxLjY5MjkzLDg3MC44NTk0MyAtMjg5LjkyMjU3LDg2NC42NjQ1
IEMgLTI3OC4xNTIyMSw4NTguNDY5NTcgLTI3OC43NzE3LDg1Ny4yMzA1OSAtMjc0LjQzNTI1LDg1
Mi44OTQxNCBDIC0yNzAuMDk4OCw4NDguNTU3NjkgLTI3MS4zMzc3OSw4NTEuMDM1NjYgLTI2NS43
NjIzNiw4NTEuMDM1NjYgQyAtMjYwLjE4NjkyLDg1MS4wMzU2NiAtMjYyLjI2NjQ2LDg0Ny42NjA0
OCAtMjU4Ljk0Nzk0LDg0Ny4zMTg3IEMgLTI1MS44MDQ1NSw4NDYuNTgzIC0yNTMuNjM4MjgsODQy
LjUwNDI1IC0yNTEuODIzNzcsODQ0LjIyMTI1IEMgLTI0OC41Nzg5OCw4NDcuMjkxNjcgLTI0Ni41
NTgwOCw4NDcuMzE4NyAtMjQ0LjY5OTYxLDg0NC44NDA3MyBDIC0yNDIuODQxMTMsODQyLjM2Mjc2
IC0yNDQuNjk5NjEsODM2LjE2Nzg0IC0yMzkuMTI0MTcsODQxLjEyMzc4IEMgLTIzMy41NDg3NCw4
NDYuMDc5NzIgLTIxNi44MjI0NCw4NjcuNzYxOTYgLTIyMS43NzgzOCw4NTUuOTkxNiBDIC0yMjYu
NzM0MzIsODQ0LjIyMTI0IC0yMzIuOTI5MjQsODM1LjU0ODM0IC0yMzUuNDA3MjIsODM0LjMwOTM2
IEMgLTIzNy44ODUxOSw4MzMuMDcwMzcgLTIzNy44ODUxOSw4MjkuMzUzNDIgLTI0Ni41NTgwOCw4
MzMuNjg5ODcgQyAtMjU1LjIzMDk4LDgzOC4wMjYzMSAtMjUzLjk5Miw4NDEuNzQzMjcgLTI1OC45
NDc5NCw4MzguNjQ1ODEgQyAtMjYzLjkwMzg4LDgzNS41NDgzNCAtMjY1LjE0Mjg2LDgzNS41NDgz
NCAtMjY1LjE0Mjg2LDgzNi43ODczMyB6IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgxMjAuMjA2LCA1
OS42OTI2KSIgaWQ9InBhdGg2ODE4IiBzdHlsZT0ib3BhY2l0eTogMTsgZmlsbDogcmdiKDIzMCwg
MjMwLCAyMzApOyBmaWxsLW9wYWNpdHk6IDE7IGZpbGwtcnVsZTogZXZlbm9kZDsgc3Ryb2tlOiBy
Z2IoMCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMC41OyBzdHJva2UtbGluZWNhcDogcm91bmQ7IHN0
cm9rZS1saW5lam9pbjogcm91bmQ7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFzaGFy
cmF5OiBub25lOyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTI1MS41MTQw
Miw3NjIuNDQ4MTggQSAyMS45OTE5ODksMTIuMzg5ODUzIDAgMSAxIC0yOTUuNDk4LDc2Mi40NDgx
OCBBIDIxLjk5MTk4OSwxMi4zODk4NTMgMCAxIDEgLTI1MS41MTQwMiw3NjIuNDQ4MTggeiIgdHJh
bnNmb3JtPSJtYXRyaXgoMS4zMDk4NiwgMCwgMCwgMS40NzUsIDIwMS44NTcsIC0zMDguMzU1KSIg
aWQ9InBhdGg2ODM3IiBzdHlsZT0ib3BhY2l0eTogMTsgZmlsbDogdXJsKCNyYWRpYWxHcmFkaWVu
dDY4NjEpIHJnYigwLCAwLCAwKTsgZmlsbC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IGV2ZW5vZGQ7
IHN0cm9rZTogbm9uZTsgc3Ryb2tlLXdpZHRoOiAwLjU7IHN0cm9rZS1saW5lY2FwOiByb3VuZDsg
c3Ryb2tlLWxpbmVqb2luOiByb3VuZDsgc3Ryb2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1kYXNo
YXJyYXk6IG5vbmU7IHN0cm9rZS1vcGFjaXR5OiAxOyIvPgogICAgPHBhdGggZD0iTSAtMTkyLjk3
MTk3LDg5MC4wNjM3IEwgLTE5Ni4wNjk0Myw5MTAuODE2NyBMIC0xODYuNDY3MjksOTExLjEyNjQ1
IEwgLTE4NC45MTg1Niw4ODkuNzUzOTUgTCAtMTkxLjExMzQ5LDg5MC45OTI5NCBMIC0xOTIuOTcx
OTcsODkwLjA2MzcgeiIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoMTIwLjIwNiwgNTkuNjkyNikiIGlk
PSJwYXRoNjg2NSIgc3R5bGU9Im9wYWNpdHk6IDE7IGZpbGw6IHJnYigxMjMsIDY2LCA0OSk7IGZp
bGwtb3BhY2l0eTogMTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IG5vbmU7IHN0cm9rZS13
aWR0aDogMC41OyBzdHJva2UtbGluZWNhcDogcm91bmQ7IHN0cm9rZS1saW5lam9pbjogcm91bmQ7
IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFzaGFycmF5OiBub25lOyBzdHJva2Utb3Bh
Y2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTE2MS42ODc1OSw4MjMuNzc3OTUgQSAxMS40NjA2
MTQsMTQuODY3ODI1IDAgMSAxIC0xODQuNjA4ODIsODIzLjc3Nzk1IEEgMTEuNDYwNjE0LDE0Ljg2
NzgyNSAwIDEgMSAtMTYxLjY4NzU5LDgyMy43Nzc5NSB6IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgx
MTUuMjUsIDUwLjQwMDIpIiBpZD0icGF0aDY4NjgiIHN0eWxlPSJvcGFjaXR5OiAxOyBmaWxsOiB1
cmwoI3JhZGlhbEdyYWRpZW50NjkwMCkgcmdiKDAsIDAsIDApOyBmaWxsLW9wYWNpdHk6IDE7IGZp
bGwtcnVsZTogZXZlbm9kZDsgc3Ryb2tlOiBub25lOyBzdHJva2Utd2lkdGg6IDAuNTsgc3Ryb2tl
LWxpbmVjYXA6IHJvdW5kOyBzdHJva2UtbGluZWpvaW46IHJvdW5kOyBzdHJva2UtbWl0ZXJsaW1p
dDogNDsgc3Ryb2tlLWRhc2hhcnJheTogbm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8
cGF0aCBkPSJNIC0zMjIuNzU1NjgsOTg0LjIyNjU4IEwgLTMyNC42MTQxNiw5OTUuMzc3NDUgQyAt
MzI0LjYxNDE2LDk5NS4zNzc0NSAtMzIyLjEzNjE5LDk5NS45OTY5NSAtMzIwLjI3NzcxLDk5My41
MTg5NyBDIC0zMTEuODIxMzgsOTg2LjgwODYxIC0yOTguMDE3NSw5ODUuNjE2NjcgLTI4OC4wNjQw
OSw5ODkuMTgyNTMgQyAtMjc3LjI0MjA0LDk5OC40MTY0MSAtMjY1Ljk2MzI4LDEwMDQuNTYwNSAt
MjUxLjUxNDAyLDk5OS4wOTQ0MSBDIC0yMzkuODEzNjEsOTk0LjI4NjE1IC0yMjguOTQxMjcsOTg4
LjE0NDIgLTIxOC4wNjE0Miw5ODEuNzQ4NjEgTCAtMjE5LjkxOTksOTc2LjE3MzE4IEMgLTIxOS45
MTk5LDk3Ni4xNzMxOCAtMjI2LjExNDgzLDk3Ny40MTIxNyAtMjMwLjQ1MTI3LDk3OS44OTAxNCBD
IC0yMzguNDE5MzMsOTg2LjQ0MTk2IC0yNTkuMjcxNDcsOTk5LjU5OTQgLTI2OC44NTk4Miw5OTMu
NTE4OTcgQyAtMjc4LjQyMTU4LDk4MS41NjAyNCAtMzAwLjcyOTk3LDk3NC42ODQ0NSAtMzE1LjMy
MTc3LDk3OC42NTExNSBDIC0zMjEuNTE2Nyw5ODEuNzQ4NjEgLTMyNC42MTQxNiw5ODQuODQ2MDgg
LTMyMi43NTU2OCw5ODQuMjI2NTggeiIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoMTIwLjIwNiwgNTku
NjkyNikiIGlkPSJwYXRoNjg4OCIgc3R5bGU9Im9wYWNpdHk6IDE7IGZpbGw6IHJnYigyNTAsIDIw
MiwgODgpOyBmaWxsLW9wYWNpdHk6IDE7IGZpbGwtcnVsZTogZXZlbm9kZDsgc3Ryb2tlOiByZ2Io
MCwgMCwgMCk7IHN0cm9rZS13aWR0aDogMTsgc3Ryb2tlLWxpbmVjYXA6IHJvdW5kOyBzdHJva2Ut
bGluZWpvaW46IHJvdW5kOyBzdHJva2UtbWl0ZXJsaW1pdDogNDsgc3Ryb2tlLWRhc2hhcnJheTog
bm9uZTsgc3Ryb2tlLW9wYWNpdHk6IDE7Ii8+CiAgICA8cGF0aCBkPSJNIDE0OS45MDk5OCw3NC44
MjkwNDYgTCAxNDQuMjcyNzksNTUuNDExOTYgTCAxMjYuNTM1MTMsNjYuMTk3ODkxIEwgMTM4LjIw
MTI5LDQ5LjY4NDIwNCBMIDExOC43MDkyNiw0Mi41NDEyNTggTCAxMzguODkzOTEsNDEuMzY2MTEz
IEwgMTMyLjMyNTQsMjEuNjczMDc1IEwgMTQ1LjgyOTA5LDM2LjcyMTM4IEwgMTU3LjEzMDM0LDE5
LjMwNzUwOSBMIDE1My43ODQ1MSwzOS4yNDc1ODMgTCAxNzQuNDQ1NDMsMzcuMjI1ODc5IEwgMTU2
Ljc2OTU3LDQ3LjA0MjQ0IEwgMTcxLjIzMjA3LDYxLjkzNTI4OCBMIDE1Mi41MzY0NSw1NC4yMzYy
NjUgTCAxNDkuOTA5OTgsNzQuODI5MDQ2IHoiIHRyYW5zZm9ybT0ibWF0cml4KDEuNTM3NjcsIDAs
IDAsIDEuNTM3NjcsIC0yODUuMDI2LCA4MDIuODE4KSIgaWQ9InBhdGgyNTIxIiBzdHlsZT0ib3Bh
Y2l0eTogMTsgZmlsbDogdXJsKCNyYWRpYWxHcmFkaWVudDMzMDEpIHJnYigwLCAwLCAwKTsgZmls
bC1vcGFjaXR5OiAxOyBmaWxsLXJ1bGU6IGV2ZW5vZGQ7IHN0cm9rZTogbm9uZTsgc3Ryb2tlLXdp
ZHRoOiA2OyBzdHJva2UtbGluZWNhcDogcm91bmQ7IHN0cm9rZS1saW5lam9pbjogcm91bmQ7IHN0
cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFzaGFycmF5OiBub25lOyBzdHJva2Utb3BhY2l0
eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTE1NC4zMDY2Nyw4ODguODIzMzEgQyAtMTYzLjM4ODU2
LDg5My41MTAzMSAtMTczLjI5MTA0LDg5Mi4yOTIyOCAtMTc5LjQ0MjI2LDg4OC4zNzg2NyIgaWQ9
InBhdGg1MDA1IiBzdHlsZT0iZmlsbDogbm9uZTsgZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6
IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRoOiAwLjY5Nzg3NDsgc3Ryb2tlLWxpbmVjYXA6IGJ1
dHQ7IHN0cm9rZS1saW5lam9pbjogbWl0ZXI7IHN0cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2Ut
ZGFzaGFycmF5OiBub25lOyBzdHJva2Utb3BhY2l0eTogMTsiLz4KICAgIDxwYXRoIGQ9Ik0gLTEw
My40MzE0MSw4ODYuOTU3NjUgQyAtMTA5LjU4MDkzLDg5MS40ODkzNSAtMTE2LjI4NjA5LDg5MC4z
MTE2OCAtMTIwLjQ1MTIsODg2LjUyNzc1IiBpZD0icGF0aDUwMDciIHN0eWxlPSJmaWxsOiBub25l
OyBmaWxsLXJ1bGU6IGV2ZW5vZGQ7IHN0cm9rZTogcmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6
IDAuNTY0NjY4OyBzdHJva2UtbGluZWNhcDogYnV0dDsgc3Ryb2tlLWxpbmVqb2luOiBtaXRlcjsg
c3Ryb2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1kYXNoYXJyYXk6IG5vbmU7IHN0cm9rZS1vcGFj
aXR5OiAxOyIvPgogICAgPHBhdGggZD0iTSAtMTU0LjYxNjQyLDg5MS42MTEwMyBDIC0xNjMuNjk4
MzEsODk2LjI5ODAzIC0xNzMuNjAwNzksODk1LjA4IC0xNzkuNzUyMDEsODkxLjE2NjM5IiBpZD0i
cGF0aDUwMDkiIHN0eWxlPSJmaWxsOiBub25lOyBmaWxsLXJ1bGU6IGV2ZW5vZGQ7IHN0cm9rZTog
cmdiKDAsIDAsIDApOyBzdHJva2Utd2lkdGg6IDAuNjk3ODc0OyBzdHJva2UtbGluZWNhcDogYnV0
dDsgc3Ryb2tlLWxpbmVqb2luOiBtaXRlcjsgc3Ryb2tlLW1pdGVybGltaXQ6IDQ7IHN0cm9rZS1k
YXNoYXJyYXk6IG5vbmU7IHN0cm9rZS1vcGFjaXR5OiAxOyIvPgogICAgPHBhdGggZD0iTSAtMTAz
LjQzMTQxLDg4OS4xMjU4NyBDIC0xMDkuNTgwOTMsODkzLjY1NzU3IC0xMTYuMjg2MDksODkyLjQ3
OTkgLTEyMC40NTEyLDg4OC42OTU5NyIgaWQ9InBhdGg1MDExIiBzdHlsZT0iZmlsbDogbm9uZTsg
ZmlsbC1ydWxlOiBldmVub2RkOyBzdHJva2U6IHJnYigwLCAwLCAwKTsgc3Ryb2tlLXdpZHRoOiAw
LjU2NDY2ODsgc3Ryb2tlLWxpbmVjYXA6IGJ1dHQ7IHN0cm9rZS1saW5lam9pbjogbWl0ZXI7IHN0
cm9rZS1taXRlcmxpbWl0OiA0OyBzdHJva2UtZGFzaGFycmF5OiBub25lOyBzdHJva2Utb3BhY2l0
eTogMTsiLz4KICA8L2c+Cjwvc3ZnPg=='''

TRASH_BIN_ICON = '''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'''

# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
# @brief Extension class
# =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
class Extension (object):

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Initialize extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def __init__ (self, mediator):
    self.id = 'report-wizard'
    self.name = 'Report Wizard'
    self.author = 'Eduardo Aguiar'
    self.version = '0.1.0'
    self.description = 'Report programming environment'
    self.mediator = mediator.new_client_mediator ()
    self.icon_data = ICON_DATA

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Start extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def start (self):
    self.mediator.call ('toolbox.add', self.id, self.icon_data, 'Report\nWizard', self.on_activate)

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief Stop extension
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def stop (self):
    self.mediator.call ('ui.working-area.del', self.id)
    self.mediator.clear ()

  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  # @brief event: on_activate
  # =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=
  def on_activate (self, item_id):
    working_area = self.mediator.call ('ui.working-area.get', self.id)

    if not working_area:
      widget = Widget (self.mediator)

      working_area = self.mediator.call ('ui.working-area.new', self.id)
      working_area.set_default_size (800, 600)
      working_area.set_title (self.name)
      working_area.set_icon (self.icon_data)
      working_area.set_widget (widget)

    working_area.show ()
