/*
 * @(#)Symbol.java               1.0             14 October 1999
 *
 * This work is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This work is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * Copyright (c) 1999 Ericsson Telecom. All rights reserved.
 * Copyright (c) 2002 Per Cederberg. All rights reserved.
 */

package net.percederberg.mib.symbol;

import java.util.Vector;
import net.percederberg.mib.type.Type;

/**
 * The base class for all symbols in the MIB file. Each symbol is
 * typically identified by it's name, which must be unique within
 * each MIB file.<p>
 *
 * All symbols are also present in a tree structure of symbols, each
 * symbol having references to both it's parent and all it's children.
 * This symbol tree should not be changed once it's created, and is
 * intended to be created on the basis of the object identifiers.<p>
 *
 * The object identifiers are only present in some symbols, but an
 * interface is provided for accessing it for all symbols (in order
 * to avoid some casting and case splitting). For symbols not having
 * object identifiers, an error should be generated runtime.
 *
 * @version  1.0
 * @author   Per Cederberg, per@percederberg.net
 */
public abstract class Symbol extends Object {

    /**
     * The symbol name.
     */
    protected String name = "";

    /**
     * The symbol type.
     */
    protected Type type = null;

    /**
     * The object identifier number.
     */
    protected int id = -1;

    /**
     * The parent symbol.
     */
    protected Symbol parent = null;

    /**
     * The child symbols.
     */
    protected Vector children = null;

    /**
     * Checks if this symbol represents a data field in the model.
     *
     * @return true if this symbol represents a data field, or
     *         false otherwise
     */
    public boolean isField() {
        return false;
    }

    /**
     * Checks if this symbol represents a top data field in the
     * hierarchy. Typically data fields contained within arrays and
     * composed types will return false for this method.
     *
     * @return true if this symbol represents a top data field, or
     *         false otherwise
     */
    public boolean isTopField() {
        return false;
    }

    /**
     * Returns the child at a given position.
     *
     * @param   index     the child position, 0 <= index < children()
     *
     * @return  the child at the position, or null if not present
     */
    public Symbol childAt(int index) {
        if (index < 0 || this.children() <= index) {
            return null;
        } else {
            return (Symbol) this.children.elementAt(index);
        }
    }

    /**
     * Returns the number of children to this symbol.
     *
     * @return  the number of children
     */
    public int children() {
        if (this.children == null) {
            return 0;
        } else {
            return this.children.size();
        }
    }

    /**
     * Returns the symbol name.
     *
     * @return the symbol name
     */
    public String getName() {
        return this.name;
    }

    /**
     * Returns the object identifier string for this object. Typically
     * this method traverses the parent nodes for retrieving their
     * object identifiers. Note that if some symbol in the symbol
     * hierachy doesn't support object identifiers, a runtime
     * exception is thrown.
     *
     * @return  a string containing the numeric object identifier
     */
    public String getOID() {
        String base = "";
        if (this.parent != null) {
            base = this.parent.getOID();
        }
        if (id >= 0) {
            return base + "." + this.id;
        } else {
            return base;
        }
    }

    /**
     * Returns the parent symbol.
     *
     * @return  the parent symbol, or null for no parent.
     */
    public Symbol getParent() {
        return this.parent;
    }

    /**
     * Returns the type of this symbol.
     *
     * @return the type of the symbol, or null for unknown type
     */
    public Type getType() {
        return type;
    }

    /**
     * Sets this symbols object identifier. To be able to retrieve the
     * full object identifier, all ancestor nodes must also have been
     * assigned object identifiers. For symbols not supporting object
     * identifiers, this method may be overridden to throw a runtime
     * exception.
     *
     * @param   id       the object identifier number, 0 <= id
     */
    public void setOID(int id) {
        this.id = id;
    }

    /**
     * Sets the symbol parent. Also registers this symbol as a child
     * to the new parent. The method guarantees consistency by
     * removing the reference in case of parent change.
     *
     * @param   parent    the new symbol parent
     */
    public void setParent(Symbol parent) {
        if (this.parent == parent) {
            // Do nothing
            return;
        }
        if (this.parent != null) {
            this.parent.removeChild(this);
        }
        this.parent = parent;
        if (parent != null) {
            parent.addChild(this);
        }
    }

    /**
     * Sets the type of this symbol.
     *
     * @param   type     the symbol type information
     */
    public void setType(Type type) {
        this.type = type;
    }

    /**
     * Checks for equality with an object. Returns true for symbol
     * objects with the same name.
     *
     * @param   obj   any object
     *
     * @return  true if the objects are equal, or
     *          false otherwise
     */
    public boolean equals(Object obj) {
        if (obj instanceof Symbol) {
            Symbol  sym = (Symbol) obj;
            return this.name.equals(sym.name);
        } else {
            return false;
        }
    }

    /**
     * Returns a description of the symbol.
     *
     * @return  a symbol description (the symbol name)
     */
    public String toString() {
        return this.name;
    }

    /**
     * Adds a child to this symbol. For symbols not accepting child
     * nodes this method can be overridden to throw a runtime
     * exception.
     *
     * @param   child       a child symbol
     */
    protected void addChild(Symbol child) {
        if (children == null) {
            children = new Vector();
        }
        children.addElement(child);
    }

    /**
     * Removes a child from this symbol.
     *
     * @param   child       a child symbol
     */
    protected void removeChild(Symbol child) {
        if (children == null) {
            return;
        }
        children.removeElement(child);
    }

    /**
     * Searches for a child with a given name. The current node can be
     * returned if it has the name searched for.
     *
     * @param   name     the child name
     *
     * @return  the child, or null if not found
     */
    protected Symbol findChild(String name) {
        if (getName().equals(name)) {
            return this;
        } else {
            Symbol  sym;
            for (int i = 0; i < children(); i++) {
                sym = childAt(i).findChild(name);
                if (sym != null) {
                    return sym;
                }
            }
            return null;
        }
    }
}
