/*
	This file is part of jrisk.
	
	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 15-feb-2005 by Alessio Treglia
 * Copyright  2005 Alessio Treglia
 */

import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.Iterator;

/**
 * Contiene una mappatura del tabellone di gioco.
 * Gestisce i collegamenti fra i territori e i
 * giocatori che li occupano.
 * 
 * @author Alessio Treglia
 * @version 1.0.0
 */
public class Tabellone {
    /** Mappa del gioco */
    private Map territorio2giocatore;
    
    /**
     * Inizializza il tabellone
     */
    public Tabellone() {
        territorio2giocatore = new HashMap();
    }
    
    /**
     * Aggiunge un territorio alla mappa mondiale,
     * impostando il riferimento al giocatore che
     * lo occupa.
     * 
     * @param t <code>Territorio</code> da aggiungere
     * @param g <code>Giocatore</code> occupante del territorio
     */
    public void aggiungi(Territorio t, Giocatore g) {
        territorio2giocatore.put(t.getNome(), g);
    }
    /**
     * Rimuove un territorio dal tabellone.
     * 
     * @param nomeTerritorio territorio da rimuovere
     */
    public void rimuovi(String nomeTerritorio) {
        territorio2giocatore.remove(nomeTerritorio);
    }
    
    /**
     * Restituisce un riferimento al giocatore occupante
     * di un territorio il cui nome sia uguale alla stringa
     * nomeTerritorio.
     * 
     * @param nomeTerritorio nome del territorio
     * @return il riferimento al <code>Giocatore</code>
     * occupante del territorio
     */
    public Giocatore getGiocatoreAt(String nomeTerritorio) {
        return (Giocatore) territorio2giocatore.get(nomeTerritorio);
    }
    /**
     * Cerca un giocatore a partire dal suo colore. Se lo
     * trova restituisce un riferimento ad esso, altrimenti
     * rstituisce un riferimento nullo.
     * 
     * @param coloreGiocatore <code>String</code> contenente
     * il colore del <code>Giocatore</code> da cercare
     * @return un riferimento al <code>Giocatore</code>,
     * <code>null</code> in caso contrario
     */    
    public Giocatore getGiocatore(String coloreGiocatore) {
        Set territori = territorio2giocatore.keySet();
        Iterator it = territori.iterator();
        String territorio;
        Giocatore g;
        
        while(it.hasNext()) {
            territorio = (String) it.next();
            g = getGiocatoreAt(territorio);
            if(g.getColore().equals(coloreGiocatore))
                return g;
        }
        return null;
    }
    
    /**
     * Quando un giocatore conquista un territorio viene 
     * posizionato su di esso e viene aggiornata la mappa dei
     * giocatori sui territori.
     * 
     * @param t <code>Territorio</code> da aggiornare
     * @param occupante nuovo occupante del territorio
     * @return la mappa aggiornata dei territori
     */
    public Map aggiornaTerritorio(Territorio t, Giocatore occupante) {
        Giocatore temp;									// Riferimento a un oggetto temporaneo
        /* Calcola il proprietario del territorio
         * e lo memorizza in temp         */
        temp = (Giocatore) territorio2giocatore.get(t.getNome());
        /* il vecchio proprietario lascia il territorio */
        temp.rimuoviTerritorio(t.getNome());
        /* il nuovo proprietario prende il territorio */
        occupante.aggiungiTerritorio(t);
        /* aggiorna la mappa */
        territorio2giocatore.put(t.getNome(), occupante);
        
        return territorio2giocatore;
    }
    /**
     * Descrizione stringa del tabellone
     * 
     * @return una descrizione in formato <code>String</code>
     * dell'oggetto
     */
    public String toString() {
        StringBuffer buffer = new StringBuffer();
        Set keys = territorio2giocatore.keySet();
        Iterator it = keys.iterator();
        String t;
        Giocatore g;
        
        while(it.hasNext()) {
            t = (String) it.next();
            buffer.append(t + "  controllato dal giocatore ");
            g = (Giocatore) territorio2giocatore.get(t);
            buffer.append(g.getColore());
            buffer.append(" con " + g.getTerritorio(t).getArmate()
                    + " armata/e" + "\n");
        }
        
        
        return buffer.toString();
    }
    /**
     * Trova e restituisce il giocatore che possiede un dato territorio
     * 
     * @param t <code>Territorio</code> posseduto dal giocatore ricercato
     * @return un rifeirmento al <code>Giocatore</code> che occupa
     * il <code>Territorio</code>
     */
    public Giocatore trova(Territorio t) {
        String nomeTerritorio = t.getNome();
        return (Giocatore) territorio2giocatore.get(nomeTerritorio);
    }
    /**
     * Imposta come confinante due territori che abbiano come nomi
     * i parametri arg0 e arg1. Cerca all'interno della mappa i
     * giocatori occupanti di tali territori e dunque li imposta
     * come confinanti.
     * 
     * @param arg0 territorio che confina con arg1
     * @param arg1 territorio che confina con arg0
     */
    public void setConfinanti(String arg0, String arg1) {
        Territorio tuno, tdue;
        Set territoriDue;
        Giocatore uno = (Giocatore)territorio2giocatore.get(arg0);
        Giocatore due = (Giocatore)territorio2giocatore.get(arg1);
        tuno = uno.getTerritorio(arg0);
        tdue = due.getTerritorio(arg1);
        tuno.setConfinante(tdue);
        tdue.setConfinante(tuno);
    }
    /**
     * Imposta un territorio come confinante a tutti i territori
     * contenuti in una collezione. Si appoggia e sovraccarica
     * il metodo setConfinanti(String, String).
     * 
     * @param arg0 nome del territorio confinante
     * @param arg1 collezione di <code>String</code>, contenente
     * i nomi di tutti i territori confinanti
     */
    public void setConfinantiLista(String arg0, Set arg1) {
        Iterator it;							// Iteratore per la lista arg1 di nomi dei territori
        String nome;							// Nome corrente in lista
        
        it = arg1.iterator();					// Inizializza l'iteratore della lista
        /*
         * Scorri la lista di territori arg1
         */
        //System.out.println(arg1.size() + " DEBUG");
        while(it.hasNext()) {
            nome = (String )it.next();			// prendi il nome corrente
            setConfinanti(arg0, nome);			// settali come confinanti
        }
    }
}
