/*
	This file is part of jrisk.
	
	jrisk is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	
	jrisk is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
	
	You should have received a copy of the GNU General Public License
	along with jrisk; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
 * Created on 19-feb-2005 by Alessio Treglia
 * Copyright  2005 Alessio Treglia
 */

import java.util.Set;
import java.util.Iterator;
import java.io.IOException;

/**
 *  un comando per la stampa di informazioni.
 * Quando un giocatore vuole stampare delle informazioni
 * lancia un comando di stampa per il tipo di
 * informazioni desiderate.
 * 
 * @author Alessio Treglia
 * @version 1.0.0
 */
public class ComandoStampa extends Comando {
    /** Contiene il nome del comando */
    private final String nomeComando = "stampa";
    /** Contiene il tipo di informazioni desiderate */
    private String info;
    
    /**
     * Costruisce un comando che stampa le informazioni
     * richieste nel parametro.
     * 
     * @param what informazioni desiderate
     */
    public ComandoStampa(String what) {
        this.info = what;
    }
    
    /* (non-Javadoc)
     * @see Comando#getNomeComando()
     */
    public String getNomeComando() { return this.nomeComando; }
    
    /* (non-Javadoc)
     * @see Comando#esegui(Giocatore, Tabellone)
     */
    public boolean esegui(Giocatore giocatore, Tabellone planisfero)
    								throws IOException {
        /* Cosa vuole stampare il giocatore? */
        if(info == null)
            new ComandoNonValido().esegui(giocatore,planisfero);
        	else
        	    if(info.equals("tabellone"))
        	        stampaTabellone(giocatore,planisfero);
        	    else
        	        if(planisfero.getGiocatore(info)!=null)
        	            stampaGiocatore(giocatore,planisfero);
        	        else
        	            if(ElencoTerritoriContinenti.territorioValido(info))
        	                stampaTerritorio(giocatore, planisfero);
        	            else
        	                new ComandoNonValido().esegui(giocatore, planisfero);
        return false;
    }
    /**
     * Stampa il tabellone.
     * 
     * @param g <code>Giocatore</code> che ha richiesto la stampa delle informazioni
     * @param t riferimento al <code>Tabellone</code> di gioco
     */
    private void stampaTabellone(Giocatore g, Tabellone t) {
        g.schermo.stampaln(t.toString());
    }
    /**
     * Stampa delle informazioni su un dato territorio del tabellone.
     * Mostra da chi  occupato e con quante armate.
     *  
     * @param g <code>Giocatore</code> che ha richiesto la stampa delle informazioni
     * @param t riferimento al <code>Tabellone</code> di gioco
     */
    private void stampaTerritorio(Giocatore g, Tabellone t) {
        Territorio territorio, confinante;
        Set confinanti;
        Giocatore temp;
        Iterator it;
        
        /* Chiede al tabellone un riferimento al giocatore
         * che occupa un territorio con tale nome
         */
        temp = t.getGiocatoreAt(info);
        /* Chiede al giocatore che occupa il territorio
         * un riferimento allo stesso
         */
        territorio = temp.getTerritorio(info);
        /* Stampa sullo schermo del chiamante
         * le informazioni desiderate
         */
        g.schermo.stampa(territorio.toString());
        g.schermo.stampa(" e' occupato dal giocatore " +
                temp.getColore());
        g.schermo.stampaln(" con " + territorio.getArmate() +
                " armate");
        g.schermo.stampa("Confina con");
        
        /* Stampa i nomi dei territori confinanti */
        confinanti = territorio.getTerritoriConfinanti();
        it = confinanti.iterator();
        
        while(it.hasNext()) {
            confinante = (Territorio) it.next();
            /* Stampa il nome del territorio confinante */
            g.schermo.stampa(" " + confinante.getNome());
        }
        g.schermo.stampaln();						// Ritorno a capo
    }
    /**
     * Stampa le informazioni sul giocatore.
     * 
     * @param g <code>Giocatore</code> del quale stampare le informazioni
     * @param t <code>Tabellone</code> di gioco
     */
    private void stampaGiocatore(Giocatore g, Tabellone t) {
        Giocatore temp = t.getGiocatore(info);
        g.schermo.stampaln(temp);
        g.schermo.stampa(temp.getColore());
        if(temp.numeroCarte()==0)
            g.schermo.stampa(" non possiede carte, ");
        else
            if(temp.numeroCarte()==1)
                g.schermo.stampa(" possiede 1 carta, ");
            else
                g.schermo.stampa(temp.numeroCarte() + " carte, ");
        g.schermo.stampa("occupa " + temp.getTerritori().size() + " territori e ");
        if(!temp.getPuoPescare())
            g.schermo.stampaln("al momento non puo' pescare la carta.");
        else
            g.schermo.stampaln("puo' pescare la carta.");
    }
}
