;;;; "video.scm" video processing of spectra -*-scheme-*-
;;; Copyright (C) 2006 Aubrey Jaffer

;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation, either version 3 of the License, or
;;; (at your option) any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

;; http://swiss.csail.mit.edu/~jaffer/FreeSnell

(require 'array)
(require 'subarray)
(require 'array-for-each)

(define (dbize x)
  (if (or (not (real? x)) (negative? x)) (slib:error 'dbize x))
  (* 20 (real-log10 x)))

(define (dbize-columns! data)
  (define ys (subarray data
		       #f
		       (list 1 (+ -1 (cadr (array-dimensions data))))))
  (array-map! ys dbize ys)
  data)

;;; Finds smallest index of element not greater than VALUE in
;;; monotonically increasing ARRAY; or #f if no element of ARRAY is
;;; less than VALUE.
(define (greatest-lower-bound-index xs value)
  (define (glb-search left right)
    (if (< right left) #f
	(let ((mid (+ left (quotient (- right left) 2))))
	  (cond ((> value (array-ref xs mid))
		 (max (or (glb-search (+ 1 mid) right) mid) mid))
		((< value (array-ref xs mid))
		 (glb-search left (+ -1 mid)))
		(else mid)))))
  ;;(print 'xs xs)
  (glb-search 0 (+ -1 (car (array-dimensions xs)))))

;;; Finds largest index of element not less than VALUE in
;;; monotonically increasing ARRAY; or #f if no element of ARRAY is
;;; greater than VALUE.
(define (least-upper-bound-index xs value)
  (define (lub-search left right)
    (if (< right left) #f
	(let ((mid (+ left (quotient (- right left) 2))))
	  (cond ((> value (array-ref xs mid))
		 (lub-search (+ 1 mid) right))
		((< value (array-ref xs mid))
		 (min (or (lub-search left (+ -1 mid)) mid) mid))
		(else mid)))))
  (lub-search 0 (+ -1 (car (array-dimensions xs)))))

;;; The integration of a Gaussian function with the arrays containing
;;; x and y coordinates is normalized to the integration of the
;;; Gaussian function with constant 1.  Near the extremes this derives
;;; values with as much information as is contained in the array.
(define (trapezoid-integrate-point-array-with-Gaussian xs ys idx sigma)
  (define /-2*sigma^2 (/ -.5 (* sigma sigma)))
  (define margin (* 2 sigma))
  (define ivl (array-ref xs idx))
  (let ((ubx (or (least-upper-bound-index xs (+ ivl margin))
		 (+ -1 (car (array-dimensions ys)))))
	(lbx (or (greatest-lower-bound-index xs (- ivl margin)) 0)))
    ;;(print 'lbx lbx 'ubx ubx)
    (let loop ((jdx lbx)
	       (lx (array-ref xs lbx))
	       (lgsn 0)
	       (lprd 0)
	       (wgt 0)
	       (int 0))
      (if (> jdx ubx)
	  (cond ((zero? wgt)
		 (slib:warn 'trapezoid-integrate-point-array-with-Gaussian
			    'null 'Gaussian int))
		(else (/ int wgt)))
	  (let* ((x (- (array-ref xs jdx) ivl))
		 (gsn (exp (* x x /-2*sigma^2))))
	    (define dlt (- (array-ref xs jdx) lx))
	    (define prd (* gsn (array-ref ys jdx)))
	    (loop (+ 1 jdx)
		  x
		  gsn
		  prd
		  (+ (* dlt (+ gsn lgsn)) wgt)
		  (+ (* dlt (+ prd lprd)) int)))))))

(define (convolve-point-arrays-with-Gaussian! newys xs ys sigma)
  (do ((idx (+ -1 (car (array-dimensions ys))) (+ -1 idx)))
      ((negative? idx) newys)
    (array-set! newys
		(trapezoid-integrate-point-array-with-Gaussian
		 xs ys idx sigma)
		idx)))

(define (smooth-columns data sigma)
  (define dims (array-dimensions data))
  (define nra (apply make-array data (array-dimensions data)))
  (let ((xrow (subarray data #f 0)))
    (serial-array:copy! (subarray nra #f 0) xrow)
    (do ((idx (+ -1 (cadr dims)) (+ -1 idx))
	 (srcrow (subarray data #f (+ -1 (cadr dims)))
		 (subarray data #f (+ -1 idx)))
	 (dstrow (subarray nra #f (+ -1 (cadr dims)))
		 (subarray nra #f (+ -1 idx))))
	((< idx 1) nra)
      (convolve-point-arrays-with-Gaussian! dstrow xrow srcrow sigma))))

;;(print (convolve-point-arrays-with-Gaussian! (make-vector 5) '#(1 2 3 4 5) '#(0 0 1 0 0) .75))
