"""evaltest.py - test the results of evaluating formulas/programs"""

import unittest, types
import testmod
import lazyarray, program, eval, objects, relref
from parsing import parse

def exo(x):
	"""Convert python object x to ExObject"""
	if type(x) is types.StringType:
		return objects.ExString(x)
	elif type(x) in (types.IntType, types.FloatType):
		return objects.ExNum(x)
	elif isinstance(x, lazyarray.LazyArray):
		return objects.ExArray(x)
	assert False, "Unsupported type"

def get_la():
	"""Return an example lazy array"""
	subla = lazyarray.LazyArray(4)
	subla.set_items_precomputed(((exo('test'), exo(3.0)),
								 (exo('foo'), exo(2.0)),
								 (exo(32), exo('hello'))))

	la = lazyarray.LazyArray(8)
	la.set_items_precomputed(((exo(1), None), (exo(2), exo('bar')),
							  (exo('None'), exo(5)), (exo('seven'), exo(7.0)),
							  (exo('twelvehalf'), exo(12.5)),
							  (exo('sub'), exo(subla))))
	return la

def compareResults(result, t, verbose = False):
	"""True if value in result match tuple/scalar t"""
	result_tuple = testmod.result_to_tuple(result)
	#print "## result and type:", result_tuple, type(result_tuple)
	#print "## wanted:", t
	return result_tuple == t


class EvalTest(unittest.TestCase):
	"""Test the eval and eval_node functions"""
	eval_pairs = (('32', 32.0),
				  ('123.2', 123.2),
				  ('-23', -23),
				  ('seven', 7.0),
				  ('twelvehalf', 12.5),
				  ('seven+32', 39.0),
				  ('10+twelvehalf*seven', 10+12.5*7.0),
				  ('twelvehalf^seven^2', 12.5**7**2),
				  ('sub:test', 3.0),
				  ('sub:test+2*10', 23.0),
				  ('sub(30+2)', 'hello'),
				  ('array(5)', lazyarray.LazyArray(5)),
				  ('"hello"', 'hello'),
				  ('"hello\\\\there"', 'hello\\there'),
				  ('"foobar".upper', 'FOOBAR'),
				  ('"foobar".upper.lower.blank?', False),
				  ('2 < 3', True),
				  ('-23 > 48', False),
				  ('23 >= 23', True),
				  ('23 <= 22', False),
				  ('23 = 23', True),
				  ('23 != 23', False),
				  ('3<2 and 2 < 3', False),
				  ('3<2 or 22 <= 23', True))

	def testEval(self):
		"""Test the eval function"""
		las = eval.default_las + get_la()
		for expr_str, desired_result in self.eval_pairs:
			#print expr_str
			ast = parse.parse_string(expr_str)
			#print ast
			result = eval.eval(las, ast, relref.RelRefInfo()).obj
			if desired_result != result:
				print ("In formula\n%s\nWanted result: %s, got: %s" %
					   (expr_str, desired_result, result))
				assert False

	prog_pairs = {'empty': ("""
Reference,Name,Formula
"()",, 3
""", 3),
         'onelevel': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0,)", a, 5
"(1,)", b, 7
""", (5,7)),
         'twolevel': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0,)", a, "array(1)"
"(0,0)", c, 2
"(1,)", b, 7
""", ((2,), 7)),
         'a+b': ("""
Reference,Name,Formula
"()",, "array(3)"
"(0,)", a, 5
"(1,)", b, 7
"(2,)", c, "a+b"    
""", (5, 7, 12)),
         'f:a+f:b': ("""
Reference,Name,Formula
"()",, "array(3)"
"(0,)", f, array(2)
"(0,0)", a, 3
"(0,1)", b, f:a
"(1,)", c, 7
"(2,)", d, "f:a+f:b" 
""", ((3, 3), 7, 6)),
         'f:a rel': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0,)", f, array(2)
"(0,0)", a, 3
"(0,1)", b, a
"(1,)", a, "f:a+f:b"
""", ((3, 3), 6)),
         'empty': ("""
Reference,Name,Formula
"()",, "array(2)"
""", (None, None)),
		 'strings': ('''
Reference,Name,Formula
(),,array(2)
(0),,"""hello"""
(1),,"""world"""
''', ('hello', 'world')),
         'arraycopy': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0,)", f, "array(2)"
"(0,0)", a, 3
"(0,1)", b, 4
"(1,)", g, f
""", ((3,4), (3,4))),
         'arraycopy2': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0,)", f, array(2)
"(0,0)", a, 3
"(0,1)", b, 4
"(1,)", g, f
"(1,1)", b, 7
""", ((3,4), (3,7))),
         'arraycopy3': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0,)", f, g
"(0,1)", b, 7
"(1,)", g, array(2)
"(1,0)", a, 3
"(1,1)", b, 4
""", ((3,7), (3,4))),
         'arraycopydeep': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0,)", f, array(2)
"(0,0)", a, array(1)
"(0,0,0)",, 3
"(0,1)", b, 4
"(1,)", g, f
"(1,0,0)", a, 7
""", (((3,),4), ((7,),4))),
		 'relref: prev': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0,)", a, 5
"(1,)", b, prev+1
""", (5,6)),
		 'relref: me.prev2': ("""
Reference,Name,Formula
"()",, "array(3)"
"(0,)", a, 5
"(2,)", b, me.prev(2)+1
""", (5, None, 6)),
		 'relref: up.next': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0,)", a, "array(2)"
"(0,0)", aa, 5
"(0,1)", ab, "me.up.next(1)+1"
"(1,)", b, 17
""", ((5, 18), 17)),
		 'index': ("""
Reference,Name,Formula
"()",, "array(3)"
"(0,)", a, 5
"(1,)", b, me.index
"(2,)", c, a.index+18
""", (5, 2, 19)),
		 'name': ("""
Reference,Name,Formula
"()",, "array(3)"
"(0,)", a, me.name
"(1,)", b, me.name
"(2,)", 4, me.name+8
""", ('a', 'b', 12)),
		 'array_op1': ("""
Reference,Name,Formula
"()",, "array(3)"
"(0,)", ar1, "array(3)"
"(0,0)", a, 1
"(0,1)", b, 2
"(0,2)", c, 3
"(1,)", ar2, "array(3)"
"(1,0)", a, 5
"(1,1)", b, 10
"(1,2)", c, 12
"(2,)",, ar1+ar2
""", ((1,2,3), (5, 10, 12), (6, 12, 15))),
		 'array_op2': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0,)", ar1, "array(3)"
"(0,0)", a, 1
"(0,1)", b, 2
"(0,2)", c, 3
"(1,)",, "ar1*2"
""", ((1,2,3), (2, 4, 6))),
		 'array_op3': ("""
Reference,Name,Formula
"()",, "array(3)"
"(0,)", ar1, "array(3)"
"(0,0)", a, 1
"(0,1)", b, 2
"(0,2)", c, 3
"(1,)",, "ar1/2+10"
"(2,)",, "ar1.len*2"
""", ((1,2,3), (10.5, 11.0, 11.5), 6)),
		 'array_op4': ("""
Reference,Name,Formula
"()",, "array(2)"
"(0)", ar1, "array(2)"
"(0,0)", a, 1
"(0,1)", b, 2
"(1,)", foo, "array(ar1)"
""", ((1,2), ((None,), (None, None)))),
		 'array_col': ("""
Reference,Name,Formula
"()",, "array(1)"
"(0)", ar1, "array(3)"
"(0,C)",, 6
""", ((6,6,6),)),
		 'array_relative1': ("""
Reference,Name,Formula
"()",, "array(1)"
"(0)", ar1, "array(4)"
"(0,C)",, "prev"
"(0,0)",, 26
""", ((26, 26, 26, 26),)),
		 'array_relative2': ("""
Reference,Name,Formula
"()",, "array(1)"
"(0)",, "array(6)"
"(0,C)",, "2*prev+1"
"(0,0)",, 1
"(0,4)",, -23
""", ((1, 3, 7, 15, -23, -45),)),
		 'array_fibs': ("""
Reference,Name,Formula
"()",, "array(1)"
"(0)",, "array(6)"
"(0,C)",, "me.prev(1)+me.prev(2)"
"(0,0)",, 1
"(0,1)",, 1
""", ((1, 1, 2, 3, 5, 8),)),
		 'array_col_index': ("""
Reference,Name,Formula
"()",, "array(1)"
"(0)",, "array(3)"
"(0,C)",, "me.index"
""", ((1, 2, 3),)),
		 'array_col_names': ("""
Reference,Name,Formula
"()",, "array(1)"
"(0)",, "array(2)"
"(0,C)",, "me.name"
"(0,0)", hello,
"(0,1)", 4,
""", (('hello', 4),)),
		 'bug:prev': ("""
Reference,Name,Formula
"()",, "array(3)"
"(0)", b, 4
"(1)",, "array(b)"
"(1,0)",, 1
"(1,1)",, prev+2
""", (4, (1, 3, None, None), None)),
		 'index1': ("""
Reference,Name,Formula
(),,array(4)
(0),a,array(3)
"(0,0)",,5
"(0,1)",,10
"(0,2)",,15
(1),,a[1]
(2),,a[2]
(3),,a[3]
""", ((5, 10, 15), 5, 10, 15)),
		 'index2': ("""
Reference,Name,Formula
(),,array(4)
(0),a,array(3)
"(0,0)",,5
"(0,1)",,10
"(0,2)",,15
(1),,a[-1]
(2),,a[-2]
(3),,a[-3]
""", ((5, 10, 15), 15, 10, 5)),
		 'index3': ("""
Reference,Name,Formula
(),,array(3)
(0),a,array(3)
"(0,0)",,5
"(0,1)",,10
"(0,2)",,15
(1),b,array(4)
"(1,0)",,1
"(1,1)",,3
"(1,2)",,-3
"(1,3)",,2
(2),,a[b]
""", ((5, 10, 15), (1, 3, -3, 2), (5, 15, 5, 10))),
		 'where1': ("""
Reference,Name,Formula
(),,array(2)
(0),a,array(3)
"(0,0)",,5
"(0,1)",,10
"(0,2)",,15
(1),,"a where a < 11"
""", ((5, 10, 15), (5, 10))),
		 'array.distinct': ("""
Reference,Name,Formula
(),,array(2)
(0),a,array(5)
"(0,0)",,5
"(0,1)",,15
"(0,2)",,5
"(0,3)",,10
"(0,4)",,10
(1),,a.distinct
""", ((5, 15, 5, 10, 10), (5, 15, 10))),
		 'array.sort': ("""
Reference,Name,Formula
(),,array(2)
(0),a,array(5)
"(0,0)",,5
"(0,1)",,15
"(0,2)",,5
"(0,3)",,10
"(0,4)",,10
(1),,a.sort
""", ((5, 15, 5, 10, 10), (5, 5, 10, 10, 15))),
		 'array.sortOn': ("""
Reference,Name,Formula
(),,array(3)
(0),a,array(4)
"(0,0)",,5
"(0,1)",,10
"(0,2)",,15
"(0,3)",,20
(1),b,array(4)
"(1,0)",,3
"(1,1)",,1
"(1,2)",,2
"(1,3)",,4
(2),,a.sortOn(b)
""", ((5, 10, 15, 20), (3, 1, 2, 4), (10, 15, 5, 20))),
		 'array.reverse': ("""
Reference,Name,Formula
(),,array(2)
(0),a,array(5)
"(0,0)",,5
"(0,1)",,15
"(0,2)",,5
"(0,3)",,10
"(0,4)",,10
(1),,a.reverse
""", ((5, 15, 5, 10, 10), (10, 10, 5, 15, 5))),
		 'arithmetic seq1': ("""
Reference,Name,Formula
(),,array(1)
(0),,1..10
""", (tuple(range(1,11)),)),
		 'arithmetic seq2': ("""
Reference,Name,Formula
(),,array(1)
(0),,-1..-5
""", ((-1, -2, -3, -4, -5),)),
		 'arithmetic seq3': ("""
Reference,Name,Formula
(),,array(1)
(0),,"1, 3 .. 8"
""", ((1, 3, 5, 7),)),
		 'table basic1': ("""
Reference,Name,Formula
(),,array(1)
(0),,"table(2,2)"
"(0, T:R)",,array(2)
""", (testmod.Table(rownames = (None, None),
					colnames = (None, None),
					values = ((None, None), (None, None))),)),
		 'table basic2': ("""
Reference,Name,Formula
(),,array(1)
(0),,"table(2,2)"
"(0, T:C)",name,1..2
"(0, T:2,0)",hello,
""", (testmod.Table(rownames = (None, 'hello'),
					colnames = (1, 2),
					values = ((None, None), (None, None))),)),
		 'table rowrec name1': ("""
Reference,Name,Formula
(),,array(1)
(0),,"table(4,1)"
"(0, T:R)",rowname,prev(rowname)+1
"(0, T:1,0)",1,
""", (testmod.Table(rownames = (1, 2, 3, 4),
					colnames = (None,),
					values = ((None,), (None,), (None,), (None,))),)),
		 'table rowrec name fibs': ("""
Reference,Name,Formula
(),,array(1)
(0),,"table(5,1)"
"(0, T:R)",row,"me.prev(row, 1)+me.prev(row, 2)"
"(0, T:C)",col,
"(0, T:1,0)",1,
"(0, T:2,0)",2,
""", (testmod.Table(rownames = (1, 2, 3, 5, 8),
					colnames = (None,),
					values = ((None,), (None,), (None,), (None,), (None,))),)),
		 'table colrec name1': ("""
Reference,Name,Formula
(),,array(1)
(0),,"table(1,4)"
"(0, T:C)",colname,prev(colname)+1
"(0, T:0,1)",1,
""", (testmod.Table(rownames = (None,),
					colnames = (1, 2, 3, 4),
					values = ((None, None, None, None),)),)),
		 'table colrec name fibs': ("""
Reference,Name,Formula
(),,array(1)
(0),,"table(1,5)"
"(0, T:R)",row,
"(0, T:C)",col,"me.prev(col, 1)+me.prev(col, 2)"
"(0, T:0, 1)",1,
"(0, T:0, 2)",2,
""", (testmod.Table(rownames = (None,),
					colnames = (1, 2, 3, 5, 8),
					values = ((None, None, None, None, None),)),)),
		 'table value0.5': ('''
Reference,Name,Formula
(),,array(1)
(0),,"table(2,1)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:1,1)",,1
"(0, T:2,1)",,
''', (testmod.Table(rownames = (None, None),
					colnames = (None,),
					values = ((1,), (None,))),)),
		 'table value1': ('''
Reference,Name,Formula
(),,array(1)
(0),,"table(2,2)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:1,1)",,1
"(0, T:1,2)",,"""hello"""
"(0, T:2,2)",,4
''', (testmod.Table(rownames = (None, None),
					colnames = (None, None),
					values = ((1,'hello'), (None, 4))),)),			
		 'table absref1': ("""
Reference,Name,Formula
(),,array(3)
(0),a,"table(2,2)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:1,0)",rowname1,
"(0, T:2,0)",rowname2,
"(0, T:0,1)",colname1,
"(0, T:0,2)",colname2,
"(0, T:1,1)",,1
"(0, T:1,2)",,2
"(0, T:2,1)",,3
"(0, T:2,2)",,4
(1),,a:rowname1:colname1+5
(2),,a:colname2:rowname2*2
""", (testmod.Table(rownames = ('rowname1', 'rowname2'),
					colnames = ('colname1', 'colname2'),
					values = ((1,2), (3, 4))),
	  6, 8)),
		 'table absref2': ("""
Reference,Name,Formula
(),,array(1)
(0),a,"table(2,2)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:1,0)",rowname1,
"(0, T:2,0)",rowname2,
"(0, T:0,1)",colname1,
"(0, T:0,2)",colname2,
"(0, T:1,1)",,1
"(0, T:1,2)",,2
"(0, T:2,1)",,a:rowname1:colname1+5
"(0, T:2,2)",,a:rowname1:colname2*10
""", (testmod.Table(rownames = ('rowname1', 'rowname2'),
					colnames = ('colname1', 'colname2'),
					values = ((1,2), (6, 20))),)),
		 'table relref cell1': ("""
Reference,Name,Formula
(),,array(1)
(0),a,"table(3,3)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:1,0)",rowname1,
"(0, T:2,0)",rowname2,
"(0, T:3,0)",rowname3,
"(0, T:0,1)",colname1,
"(0, T:0,2)",colname2,
"(0, T:0,3)",colname3,
"(0, T:1,1)",,1
"(0, T:1,2)",,prev(col)+1
"(0, T:1,3)",,10
"(0, T:2,1)",,row
"(0, T:2,2)",,col
"(0, T:2,3)",,prev(row)+1
"(0, T:3,1)",,row
"(0, T:3,2)",,col
"(0, T:3,3)",,col
""", (testmod.Table(rownames = ('rowname1', 'rowname2', 'rowname3'),
					colnames = ('colname1', 'colname2', 'colname3'),
					values = ((1,2,10), ('rowname2', 'colname2', 11),
						      ('rowname3', 'colname2', 'colname3'))),)),
		 'table relref cell2': ("""
Reference,Name,Formula
(),,array(1)
(0),a,"table(3,3)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:1,0)",rowname1,
"(0, T:2,0)",rowname2,
"(0, T:3,0)",rowname3,
"(0, T:0,1)",colname1,
"(0, T:0,2)",colname2,
"(0, T:0,3)",colname3,
"(0, T:1,1)",,1
"(0, T:1,2)",,rowname2+1
"(0, T:1,3)",,10
"(0, T:2,1)",,rowname3-1
"(0, T:2,2)",,7
"(0, T:2,3)",,colname2*2
"(0, T:3,1)",,23
"(0, T:3,2)",,rowname2*2+1
"(0, T:3,3)",,2+a:rowname3:colname2
""", (testmod.Table(rownames = ('rowname1', 'rowname2', 'rowname3'),
					colnames = ('colname1', 'colname2', 'colname3'),
					values = ((1,8,10), (22, 7, 14), (23, 15, 17))),)),
		 'table relref cell3': ("""
Reference,Name,Formula
(),,array(1)
(0),a,"table(3,3)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:1,0)",rowname1,
"(0, T:2,0)",rowname2,
"(0, T:3,0)",rowname3,
"(0, T:0,1)",colname1,
"(0, T:0,2)",colname2,
"(0, T:0,3)",colname3,
"(0, T:1,1)",,1
"(0, T:1,2)",,5
"(0, T:1,3)",,10
"(0, T:2,1)",,"rowname1.next(col, 1)+1"
"(0, T:2,2)",,"rowname1.prev(col, 1)+1"
"(0, T:2,3)",,"colname1.prev(row, 1)*3"
"(0, T:3,1)",,"colname3.prev(row, 2)+5"
"(0, T:3,2)",,"rowname2*2+1"
"(0, T:3,3)",,30
""", (testmod.Table(rownames = ('rowname1', 'rowname2', 'rowname3'),
					colnames = ('colname1', 'colname2', 'colname3'),
					values = ((1,5,10), (6, 2, 3), (15, 5, 30))),)),
		 'table col val form1': ('''
Reference,Name,Formula
(),,array(1)
(0),,"table(3,2)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:0,1)",,1..3
"(0, T:0,2)",,10
''', (testmod.Table(rownames = (None, None, None),
					colnames = (None, None),
					values = ((1, 10), (2, 10), (3, 10))),)),
		 'table col val form1.5': ('''
Reference,Name,Formula
(),,array(1)
(0),,"table(3,3)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:0,1)",col1,1..3
"(0, T:0,2)",col2,col1+1
"(0, T:0,3)",col3,1+col1*col2
''', (testmod.Table(rownames = (None, None, None),
					colnames = ('col1', 'col2', 'col3'),
					values = ((1, 2, 3), (2, 3, 7), (3, 4, 13))),)),
		 'table col val form1.7': ('''
Reference,Name,Formula
(),,array(1)
(0),,"table(3,2)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:0,1)",col1,1..3
"(0, T:0,2)",,prev(col)+10
''', (testmod.Table(rownames = (None, None, None),
					colnames = ('col1', None),
					values = ((1, 11), (2, 12), (3, 13))),)),
		 'table col val form2': ('''
Reference,Name,Formula
(),,array(1)
(0),,"table(3,2)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:0,1)",col1,prev(row)+1
"(0, T:0,2)",,col1
"(0, T:1,1)",,10
''', (testmod.Table(rownames = (None, None, None),
					colnames = ('col1', None),
					values = ((10, 10), (11, 11), (12, 12))),)),
		 'table col val form3': ('''
Reference,Name,Formula
(),,array(1)
(0),,"table(4,2)"
"(0, T:R)",row,
"(0, T:C)",col,
"(0, T:0,1)",,col2*prev(row)
"(0, T:0,2)",col2,1..4
"(0, T:1,1)",,1
''', (testmod.Table(rownames = (None, None, None, None),
					colnames = (None, 'col2'),
					values = ((1, 1), (2, 2), (6, 3), (24, 4))),))
	}

	def testEvalProgBasic(self):
		"""Test evaluating a whole program"""
		self.eval_test_progs(('empty', 'onelevel', 'twolevel', 'a+b',
							  'f:a+f:b', 'f:a rel', 'empty', 'arraycopy',
							  'arraycopy2', 'arraycopy3', 'arraycopydeep'))

	def testStrings(self):
		"""Test evaluating a program with strings in it"""
		self.eval_test_progs(('strings',))

	def eval_test_progs(self, test_names):
		"""Go through the test names and run each"""
		for test_name in test_names:
			prog_str, ans = self.prog_pairs[test_name]
			print "Executing test "+test_name
			result = eval.eval_prog(program.Program(prog_str))
			if not compareResults(result, ans, True):
				print "Desired: "+str(ans)
				print "Received: "+str(testmod.result_to_tuple(result))
				assert False, ('Compare failure with test '+test_name)

	def testEvalProgRelRef(self):
		"""Run relative reference tests"""
		self.eval_test_progs(('relref: prev', 'relref: me.prev2',
							  'relref: up.next', 'index', 'name'))

	def testArrayOps(self):
		"""Run tests on array operations"""
		self.eval_test_progs(('array_op1', 'array_op2', 'array_op3',
							  'array_op4', 'where1'))

	def testArrayMethods(self):
		"""Run tests on array methods (distinct, sort, etc.)"""
		self.eval_test_progs(('array.distinct', 'array.sort',
							  'array.sortOn', 'array.reverse'))

	def testColFormulas(self):
		"""Run tests involving column formulas"""
		self.eval_test_progs(('array_col', 'array_relative1', 'array_relative2',
							  'array_fibs', 'array_col_index',
							  'array_col_names'))

	def testPrevBugs(self):
		"""This checks for some bugs found in previous versions"""
		self.eval_test_progs(('bug:prev',))

	def testIndexing(self):
		"""Test indexing into arrays"""
		self.eval_test_progs(('index1', 'index2', 'index3'))

	def testArithSeq(self):
		"""Test the .. operator"""
		self.eval_test_progs(('arithmetic seq1', 'arithmetic seq2',
							  'arithmetic seq3'))
		
	def testTableBasic(self):
		"""Run basic tests on programs that make 2D tables"""
		self.eval_test_progs(('table basic1', 'table basic2'))

	def testTableRecursiveNames(self):
		"""Run basic tests with tables whose row/col names use prev"""
		self.eval_test_progs(('table rowrec name1', 'table rowrec name fibs',
							  'table colrec name1', 'table colrec name fibs'))

	def testTableValues(self):
		"""Test simple table formulas and absolute references"""
		self.eval_test_progs(('table value0.5', 'table value1',
							  'table absref1', 'table absref2'))

	def testTableRelRefCells(self):
		"""Test table internal relative referencing"""
		self.eval_test_progs(('table relref cell1', 'table relref cell2',
							  'table relref cell3'))

	def testTableColForm(self):
		"""Test table column formulas that produce values"""
		self.eval_test_progs(('table col val form2', 'table col val form1.5',
							  'table col val form1.7',
							  'table col val form2', 'table col val form3'))

if __name__ == '__main__': unittest.main()