/*	file.cpp
 * 	Denemo File IO 
 *
 * 	for Denemo, a gtk+ frontend to GNU Lilypond
 * 	(c) Adam Tee, Matthew Hiller 2000-2005
 * 	(c) University of Leeds 2000-2005
 */

#include "calculatepositions.h"
#include "commandfuncs.h"
#include "contexts.h"
#include <denemo/denemo.h>
#include "dialogs.h"
#include "exportabc.h"
#include "exportlilypond.h"
#include "file.h"
#include "frogio.h"
#include "frogdefs.h"
#include "moveviewport.h"
#include "staffops.h"
#include "scoreops.h"
#include "utils.h"
#include "exportxml.h"
#include "exportmidi.h"
#include "importxml.h"
#include "exportcsound.h"
#include "importmidi.h"
#include "lyparserfuncs.h"
/*#include "lyparser.h"*/
#include "prefops.h"
#include "binreloc.h"
#include "view.h"

#include <stdio.h>
#include <string.h>
#include <sys/stat.h>		/* check existance and type of files */
#include <dirent.h>		/* filter and sort filenames */
#include <malloc.h>		/* I use free() */


typedef enum
{ DENEMO_FORMAT = 0,
  DNM_FORMAT,
  MUDELA_FORMAT,
  ABC_FORMAT,
  JTF_FORMAT,
  MIDI_FORMAT,
  JTF_POLY_FORMAT,
  CSOUND_FORMAT
}
FileFormatNames;

/* Keep this up to date ! */

#define FIRST_FORMAT_NAME DENEMO_FORMAT
#define LAST_FORMAT_NAME CSOUND_FORMAT

struct FileFormatData
{
  gchar *filename_mask;
  gchar *description;
  gchar *filename_extension;
};

static struct FileFormatData supported_import_file_formats[] = {
  {"*.denemo", N_("Denemo XML format (*.denemo)"), ".denemo"},
  {"*.dnm", N_("Denemo XML format (*.dnm)"), ".dnm"},
  {"*.ly", N_("Lilypond (formerly Mudela) (*.ly)"), ".ly"},
  {"*.mid", N_("Midi (*.mid)"), ".mid"},
  {"*.midi", N_("Midi (*.midi)"), ".midi"}
};


static struct FileFormatData supported_export_file_formats[] = {
  {"*.denemo", N_("Denemo XML format (*.denemo)"), ".denemo"},
  {"*.dnm", N_("Denemo XML format (*.dnm)"), ".dnm"},
  {"*.ly", N_("Lilypond (formerly Mudela) (*.ly)"), ".ly"},
  {"*.abc", N_("ABC (*.abc)"), ".abc"},
  {"*.jtf", N_("Unnamed file format (*.jtf)"), ".jtf"},
  {"*.mid", N_("Midi (*.mid)"), ".mid"},
  {"*.jtf", N_("Unnamed file format (*.jtf) Poly"), ".jtf"},
  {"*.sco", N_("CSound Score File (*.sco)"), ".sco"}
};

/* Some macros just to shorten lines */
#define FORMAT_MASK(i) supported_export_file_formats[i].filename_mask
#define FORMAT_DESCRIPTION(i) supported_export_file_formats[i].description
#define FORMAT_EXTENSION(i) supported_export_file_formats[i].filename_extension

#define COLUMN_NAME (0)
#define COLUMN_ID (1)

struct callbackdata
{
  struct scoreinfo *si;
  GtkWidget *fs;
  GtkWidget *comboentry;
};

/* directory last used for saving */
static gchar *file_selection_path = NULL;
static gchar *system_template_path = NULL;
static gchar *local_template_path = NULL;
static gchar *default_template_path = NULL;

/* Prototypes for non-exported functions */
static gint guess_file_format (gchar * file_name);

/**
 * Display a message box asking the user to confirm that unsaved 
 * changes will be lost
 * @return TRUE if the OK button was pressed
 */
gboolean
confirmbox (DenemoGUI * gui)
{
  GtkWidget *dialog;
  gboolean r = 0;

  dialog = gtk_message_dialog_new (NULL,
				   (GtkDialogFlags)
				   (GTK_DIALOG_MODAL |
				    GTK_DIALOG_DESTROY_WITH_PARENT),
				   GTK_MESSAGE_QUESTION,
				   GTK_BUTTONS_YES_NO,
				   _("The current score has changes in it"
				     " which you have not saved."));

  gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
					    _("Proceed with the operation "
					      "nonetheless?"));
  gtk_widget_show_all (dialog);
  r = (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES);
  gtk_widget_destroy (dialog);
  return r;
}

/**
 * This function is to be called after a file load of any sort.  
 *
 * @param si pointer to the scoreinfo structure
 */
void
updatescoreinfo (DenemoScore * si)
{
  staffnode *curstaff;
  for (curstaff = si->thescore; curstaff; curstaff = curstaff->next)
    {
      beamsandstemdirswholestaff ((DenemoStaff *) curstaff->data);
      showwhichaccidentalswholestaff ((DenemoStaff *) curstaff->data);
    }
  find_xes_in_all_measures (si);
  find_leftmost_allcontexts (si);

  si->currentstaff = si->thescore;
  si->currentmeasure = firstmeasurenode (si->currentstaff);
  si->currentobject = firstobjnode (si->currentmeasure);
  if (!si->currentobject)
    si->cursor_appending = TRUE;
  else
    si->cursor_appending = FALSE;
  si->leftmeasurenum = si->currentstaffnum = si->currentmeasurenum = 1;
  //si->cursor_x = 0;
  si->haschanged = FALSE;

}


/**
 * Custom function to compare the elements of the History queue
 * it simply wraps up stcmp
 * 
 * @param a pointer to a queue elements
 * @param b pointer to the comparison value
 * @return gint 0 if match -1 or 1 otherwise
 */
static gint
history_compare (gconstpointer a, gconstpointer b)
{
  return (strcmp ((gchar *) a, (gchar *) b));
}


/**
 * Sets the filename in the scoreinfo structure and 
 * for the main window
 */
void
set_si_filename (DenemoGUI * gui, gchar * filename)
{
  gchar *dialog_title = g_strconcat ("Denemo - ", filename, NULL);
  GList ret;
  gtk_window_set_title (GTK_WINDOW (gui->window), dialog_title);
  g_free (dialog_title);
  g_string_assign (gui->si->filename, filename);


  if (!g_queue_find_custom
      (gui->prefs->history, gui->si->filename->str, &history_compare))
    {
#ifdef DEBUG
      g_print ("%s not in history list\n", gui->si->filename->str);
#endif
      if (g_queue_get_length (gui->prefs->history) > MAX_HISTORY)
	{
	  gpointer data = g_queue_pop_head (gui->prefs->history);
	  if (data)
	    g_free (data);
	}

      GtkWidget *item =
	gtk_ui_manager_get_widget (gui->ui_manager,
				   "/MainMenu/FileMenu/OpenRecent/Stub");
      GtkWidget *menu = gtk_widget_get_parent (GTK_WIDGET (item));
      struct cbd data;
      data.gui = gui;
      data.menu = menu;

      addhistorymenuitem (filename, &data);
      g_queue_push_tail (gui->prefs->history, gui->si->filename->str);
    }

}


static void      update_file_selection_path (gchar *file) {
  if(file_selection_path)
    g_free(file_selection_path);
  file_selection_path = g_path_get_dirname(file);
}




/**
 * The function that actually determines the file type and calls the
 *  function that opens the file.  (So many layers of indirection...)  
 *
 * @return success or failure
 */
gint
open_for_real (gchar * filename, DenemoGUI * gui, gboolean template)
{
  gint result = -1;
  if (gui->si->lily_file != NULL)
    abandon_lily_tree (gui);
  if (strcmp (filename + strlen (filename) - 7, ".denemo") == 0)
    result = importXML (filename, gui->si);
  else if (strcmp (filename + strlen (filename) - 4, ".dnm") == 0)
    result = importXML (filename, gui->si);
  else if (strcmp (filename + strlen (filename) - 3, ".ly") == 0)
    result = lyinput (filename, gui);
  else if (strcmp (filename + strlen (filename) - 4, ".mid") == 0 ||
	   strcmp (filename + strlen (filename) - 5, ".midi") == 0)
    result = importMidi (filename, gui);
  else if (strcmp (filename + strlen (filename) - 4, ".jtf") == 0)
    result = froginput (filename, gui->si);
  if (result != -1)
    {
      if(!template) {// not a template
	update_file_selection_path (filename);      
	set_si_filename (gui, filename);
      }
      updatescoreinfo (gui->si);
      set_rightmeasurenum (gui->si);
      set_bottom_staff (gui);
      update_hscrollbar (gui);
      update_vscrollbar (gui);
      gtk_widget_queue_draw (gui->scorearea);
      gtk_signal_emit_by_name (GTK_OBJECT (gui->hadjustment), "changed");
      gtk_signal_emit_by_name (GTK_OBJECT (gui->vadjustment), "changed");
    }
  return result;
}

/**
 * denemo_warning prompts the user to save the work in the denemo  
 * format if not done so.
 * @param si pointer to the denemo score object
 * @param format_id the numeric id of the files format
 * @return none
 */

static void
denemo_warning (DenemoScore * si, gint format_id)
{


  if (format_id != DENEMO_FORMAT && format_id != DNM_FORMAT)
    {
      GtkWidget *dialog;
      dialog = gtk_message_dialog_new (NULL,
				       GTK_DIALOG_DESTROY_WITH_PARENT,
				       GTK_MESSAGE_WARNING,
				       GTK_BUTTONS_YES_NO,
				       "You have made changes to your document that was not saved as denemo file."
				       " I advise you save your work now as a denemo file to easily continue work later. Save as denemo?");
      gtk_dialog_set_default_response (GTK_DIALOG (dialog),
				       GTK_RESPONSE_ACCEPT);
      if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES)
	{

	  gchar **file = g_strsplit (si->filename->str, ".", 0);

	  file[0] = g_strconcat (file[0], ".denemo", NULL);
	  g_print ("file %s\n", file[0]);
	  exportXML (file[0], si, 0, 0);
	  g_strfreev (file);
	}
      gtk_widget_destroy (dialog);
    }


}

/*
	If the filename already has a denemo file name extension use
	it regardless of the value of format_id, otherwise add the 
	file name extension 
*/
static gchar * 
create_filename (const gchar * file_name, gint *format_id)
{
  gint i;

  if (*format_id < 0)
        return (g_strdup (file_name));

  for (i = 0; i < (gint) G_N_ELEMENTS (supported_export_file_formats); i++)
    {
  	if (g_pattern_match_simple (FORMAT_MASK (i), file_name))
    	{
	     *format_id = i;
     	     return (g_strdup (file_name));
    	}
    }
  return (g_strconcat (file_name, FORMAT_EXTENSION (*format_id), NULL));
}

/**
 * File save callback called by fileselsave callback
 * param file_name is full path to file with extension
 */
static void
filesel_save (DenemoGUI * gui, const gchar * file_name, gint format_id, gboolean template)
{
  g_assert (gui != NULL);
  g_assert (file_name != NULL);
  g_assert (format_id >= 0 && format_id <
	    (int) G_N_ELEMENTS (supported_export_file_formats));

  DenemoScore *si = gui->si;
  // Append file extension extension
  gchar *file = NULL;
  gchar *basename = NULL;

  file = create_filename(file_name, &format_id);
#ifdef DEBUG
  g_print("Saving to file %s", file);
#endif
  if(!template)
    update_file_selection_path(file);
  basename = g_path_get_basename (file);
  /* we don't want to save scores under ".denemo",
     ".jtf" and so on. do we? */
  if (basename[0] != '.')
    {
      switch (format_id)
	{
	case DENEMO_FORMAT:
	case DNM_FORMAT:
	  {
	    exportXML (file, si, 0, 0);
	    break;
	  };
	case MUDELA_FORMAT:
	  {
	    exportlilypond (file, si, 0, 0);
	    break;
	  };
	case ABC_FORMAT:
	  {
	    exportabc (file, si, 0, 0);
	    break;
	  };
	case JTF_FORMAT:
	  {
	    filesave (file, si, 0, 0, 0);
	    break;
	  };
	case MIDI_FORMAT:
	  {
	    exportmidi (file, si, 0, 0);
	    break;
	  };
	case JTF_POLY_FORMAT:
	  {
	    filesave (file, si, 0, 0, 1);
	    break;
	  };
	case CSOUND_FORMAT:
	  {
	    exportcsound (file, si, 0, 0);
	    break;
	  };
	default:
	  break;
	};
	
	/*export parts as lilypond files*/
	if(gui->prefs->saveparts)
		export_lilypond_parts(file,si,0,0);
	
	
      si->haschanged = FALSE;
      si->readonly = FALSE;
      if (gui->prefs)
	{
	  g_timeout_add (gui->prefs->autosave_timeout * 1000 * 60,
			 (GSourceFunc) auto_save_document_timeout, si);

	  set_si_filename (gui, file);
	}
    }
  if (gui->statusbar)
    {
      gchar *tmp = NULL;
      tmp = g_strconcat (file_name, " saved", NULL);
      gtk_statusbar_push (GTK_STATUSBAR (gui->statusbar),
			  gui->status_context_id, tmp);
      g_free (tmp);
    }

  g_free(basename);
  g_free(file);
  denemo_warning (gui->si, format_id);
}

/* set local_template_path up */
static void
init_local_path(void) {
      local_template_path = g_build_filename (locatedotdenemo(), "templates", NULL);
      gboolean err = g_mkdir_with_parents(local_template_path, 0770);
      if(err) {
	warningdialog("Could not create .denemo/templates for you personal templates");
	g_free(local_template_path);
	local_template_path = NULL;
	}
}

/*
 * Sets the file_selection_path to the templates directory and 
 * calls file_open to create the file selection dialog
 * LOCAL whether to use the local templates or systemwide ones
 * does nothing if unable to access templates
 */
static void
template_open (DenemoGUI * gui, gboolean local)
{
  if(local) {
    if(local_template_path==NULL) {
      init_local_path();
    }
    default_template_path = local_template_path;   
  } else { 
    if(system_template_path==NULL) {
      system_template_path = g_build_filename (get_data_dir (), "templates", NULL);
      GDir *denemo_path = g_dir_open(system_template_path, 0, NULL);
      if(denemo_path == NULL) {
	warningdialog ("No templates directory in installation");
	system_template_path = NULL;
      }
    }
    default_template_path = system_template_path;
  }
  if(default_template_path) {
    file_open (gui, TRUE);
    gui->si->filename = g_string_new("");
  }
}




/*
 * Open system template file callback function 
 */
void
system_template_open_with_check (GtkAction * action, DenemoGUI * gui) {
  if (gui->si->haschanged)
    {
      if (confirmbox (gui))
	{
	  template_open (gui, FALSE);
	}
    }
  else
    {
      template_open (gui, FALSE);
    }
}
/*
 * Open local template file callback function 
 */
void
local_template_open_with_check (GtkAction * action, DenemoGUI * gui) {
  if (gui->si->haschanged)
    {
      if (confirmbox (gui))
	{
	  template_open (gui, TRUE);
	}
    }
  else
    {
      template_open (gui, TRUE);
    }
}


/**
 * Wrapper function for opening a file
 * checks to see if current score has changed and prompts user to save 
 * otherwise opens the file
 */
void
file_open_with_check (GtkAction * action, DenemoGUI * gui)
{
  if (gui->si->haschanged)
    {
      if (confirmbox (gui))
	{
	  file_open (gui, FALSE);
	}
    }
  else
    {
      file_open (gui, FALSE);
    }
}

static void  set_current_folder(GtkWidget *file_selection, DenemoGUI *gui, gboolean template) {
  gchar *path, *fallback;
  if(template) {
    fallback = path = default_template_path;
  } else {
    fallback = path = file_selection_path;
    GDir *denemo_path = g_dir_open(gui->prefs->denemopath->str, 0, NULL);
    if(denemo_path != NULL)
      {
	g_dir_close(denemo_path);
	fallback = gui->prefs->denemopath->str;
      } 
  }
  if (path != NULL)
    {
      gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (file_selection),
					   path);
    } else {
      if(fallback != NULL)
	gtk_file_chooser_set_current_folder 
	  (GTK_FILE_CHOOSER (file_selection), fallback);
    } 
}
/**
 * File open dialog - opened where appropriate 
 *
 */
void
file_open (DenemoGUI * gui, gboolean template)
{
  GtkWidget *file_selection;
  GtkFileFilter *filter;

  int i;

  file_selection = gtk_file_chooser_dialog_new (_("Open"),
						GTK_WINDOW (gui->window),
						GTK_FILE_CHOOSER_ACTION_OPEN,
						GTK_STOCK_CANCEL,
						GTK_RESPONSE_REJECT,
						GTK_STOCK_OPEN,
						GTK_RESPONSE_ACCEPT, NULL);
  /* Open the last visited directory, if any. */
  set_current_folder(file_selection, gui, template);


  for (i = 0; i < (gint) G_N_ELEMENTS (supported_import_file_formats); i++)
    {
      filter = gtk_file_filter_new ();
      gtk_file_filter_set_name (filter,
				supported_import_file_formats[i].description);
      gtk_file_filter_add_pattern (filter,
				   supported_import_file_formats[i].
				   filename_mask);
      gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (file_selection), filter);
    }
  filter = gtk_file_filter_new ();
  gtk_file_filter_set_name (filter, _("All files"));
  gtk_file_filter_add_pattern (filter, "*");
  gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (file_selection), filter);
  gtk_dialog_set_default_response (GTK_DIALOG (file_selection),
				   GTK_RESPONSE_ACCEPT);
  gtk_widget_show_all (file_selection);
  if (gtk_dialog_run (GTK_DIALOG (file_selection)) == GTK_RESPONSE_ACCEPT)
    {
      gchar *filename =
	gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (file_selection));
      open_for_real (filename, gui, template);
      g_free (filename);
    }
  gtk_widget_destroy (file_selection);

}

/**
 * Wrapper function to save the current file if not already 
 * saved.
 */
void
file_saveaswrapper (GtkAction * action, DenemoGUI * gui)
{
  file_saveas (gui, FALSE);
}

/**
 * Wrapper function to save the current file as template
 */
void
template_save (GtkAction * action, DenemoGUI * gui)
{
  init_local_path();
  default_template_path = local_template_path;
  file_saveas (gui, TRUE);
}





/**
 * Wrapper function for saving an existing file
 *
 */
void
file_savewrapper (GtkAction * action, DenemoGUI * gui)
{
  file_save (NULL, gui);
}

/**
 * Filters the filename based on its extension and calls
 * the relevant export function
 */
void
file_save (GtkWidget * widget, DenemoGUI * gui)
{
  DenemoScore *si = gui->si;
  g_print ("READONLY %d\n", si->readonly);
  if ((si->filename->len == 0)/* || (si->readonly == TRUE)*/)
    /* No filename's been given or is opened from template */
    file_saveas (gui, FALSE);
  else
    switch (guess_file_format (si->filename->str))
      {
      case DENEMO_FORMAT:
      case DNM_FORMAT:
	{
	  exportXML (si->filename->str, si, 0, 0);
	  break;
	};
      case MUDELA_FORMAT:
	{
	  exportlilypond (si->filename->str, si, 0, 0);
	  break;
	};
      case ABC_FORMAT:
	{
	  exportabc (si->filename->str, si, 0, 0);
	  break;
	};
      case JTF_POLY_FORMAT:
	{
	  filesave (si->filename->str, si, 0, 0, 1);
	  break;
	};
      case JTF_FORMAT:
	{
	  filesave (si->filename->str, si, 0, 0, 0);
	  break;
	};
      case MIDI_FORMAT:
	{
	  exportmidi (si->filename->str, si, 0, 0);
	  break;
	};
      case CSOUND_FORMAT:
	{
	  exportcsound (si->filename->str, si, 0, 0);
	  break;
	};

      default:
	{
	  exportXML (si->filename->str, si, 0, 0);
	  break;
	};
      };
   /*Save parts as lilypond files*/   
   if(gui->prefs->saveparts)
	export_lilypond_parts(si->filename->str,si,0,0);
  
  denemo_warning (gui->si, guess_file_format (si->filename->str));
  si->haschanged = FALSE;
}

/**
 * Create file saveas dialog to enable user to save the current file to
 *
 *
 */
void
file_saveas (DenemoGUI * gui, gboolean template)
{
  GtkWidget *file_selection;
  GtkWidget *label;
  GtkWidget *combobox;
  GtkWidget *hbox;
  GtkListStore *list_store;
  GtkTreeIter iter;
  GtkCellRenderer *renderer;


  file_selection = gtk_file_chooser_dialog_new (_("Save As"),
						GTK_WINDOW (gui->window),
						GTK_FILE_CHOOSER_ACTION_SAVE,
						GTK_STOCK_CANCEL,
						GTK_RESPONSE_REJECT,
						GTK_STOCK_SAVE,
						GTK_RESPONSE_ACCEPT, NULL);


  /*set default folder for saving */
  set_current_folder(file_selection, gui, template);

  /* assign title */ 
  if (gui->si->headerinfo->title != NULL)
    { 
      gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (file_selection), 
					 gui->si->headerinfo->title->str);
    }
 // else {
  //	gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (file_selection), "Untitled Document");
  //}


  hbox = gtk_hbox_new (FALSE, 8);
  label = gtk_label_new (_("Format:"));
  gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, TRUE, 0);

  list_store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_INT);
  combobox = gtk_combo_box_new_with_model (GTK_TREE_MODEL (list_store));
  renderer = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (combobox), renderer, TRUE);
  gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (combobox),
				 renderer, "text", COLUMN_NAME);
  gtk_box_pack_start (GTK_BOX (hbox), combobox, TRUE, TRUE, 0);

      int i;
      for (i = 0; i < (int) G_N_ELEMENTS (supported_export_file_formats); i++)
	{
	  gtk_list_store_append (list_store, &iter);
	  gtk_list_store_set (list_store, &iter,
			      COLUMN_NAME,
			      supported_export_file_formats[i].description,
			      COLUMN_ID, i, -1);
	}

  gtk_tree_model_get_iter_first (GTK_TREE_MODEL (list_store), &iter);
  gtk_combo_box_set_active_iter (GTK_COMBO_BOX (combobox), &iter);
  gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (file_selection), hbox);
  gtk_dialog_set_default_response (GTK_DIALOG (file_selection),
				   GTK_RESPONSE_ACCEPT);
  gtk_widget_show_all (file_selection);
  gboolean close = FALSE;
  do
    {
      if (gtk_dialog_run (GTK_DIALOG (file_selection)) == GTK_RESPONSE_ACCEPT)
	{
	  gint format_id = -1;
	  gchar *file_name
	    =
	    gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (file_selection));

          gtk_combo_box_get_active_iter (GTK_COMBO_BOX (combobox), &iter);
          gtk_tree_model_get (GTK_TREE_MODEL (list_store), &iter,
                                  COLUMN_ID, &format_id, -1);


	  if (replace_existing_file_dialog
	      (file_name, GTK_WINDOW (gui->window), format_id))
	    {
	      filesel_save (gui, file_name, format_id, template);
	      close = TRUE;
	    }
	  g_free (file_name);
	}
      else
	{
	  close = TRUE;
	}
    }
  while (!close);

  gtk_widget_destroy (file_selection);

}

/**
 * Wrapper function for creating a new file
 *
 */
void
file_newwrapper (GtkAction * action, DenemoGUI * gui)
{
  if (gui->si->haschanged)
    {
      if (confirmbox (gui))
	{
	  deletescore (NULL, gui);
	}
    }
  else
    {
      deletescore (NULL, gui);
    };
}

/**
 * Delete the given score and create a new one
 *
 */
void
deletescore (GtkWidget * widget, DenemoGUI * gui)
{
  free_score (gui->si);
  newstaff (gui->si, INITIAL, DENEMO_NONE);
  set_rightmeasurenum (gui->si);
  update_hscrollbar (gui);
  update_vscrollbar (gui);
  gtk_widget_queue_draw (gui->scorearea);
}

void
dnm_deletescore (GtkWidget * widget, DenemoGUI * gui){
  deletescore (widget, gui);
}


/**
 * Try to suggest the format of a given file, after its file name extension. A
 * more powerful function could be written to guess the format after the
 * file contents 
 */
gint
guess_file_format (gchar * file_name)
{
  gint name_iterator;
  gboolean format_match;

  name_iterator = FIRST_FORMAT_NAME;
  format_match = FALSE;

  while (!format_match && name_iterator <= LAST_FORMAT_NAME)
    {
      format_match = g_pattern_match_simple (FORMAT_MASK (name_iterator++),
                                             file_name);
    };

  /* In case no match could be found, we just give a 'default' format.
   * Chances are that all formats will be wrong, however ;-) */
  if (!format_match)
    return (DENEMO_FORMAT);
  else
    return (--name_iterator);
};



/**
 * Reloads a lilypond file specified by the .denemo/reloadfile.ly
 * only used when lilypond mode is active
 */
void
reload_lily_file (GtkWidget * button, gpointer data)
{
  // delete me
}

/**
 * Creates dialog to say that the chosen filename already exists
 * and do you want to overwrite it.
 *
 */
gboolean
replace_existing_file_dialog (const gchar * filename,
			      GtkWindow * parent_window, gint format_id)
{

  gchar *file = create_filename (filename, &format_id);
  if (!g_file_test (file, G_FILE_TEST_EXISTS))
    {
      g_free (file);
      return TRUE;
    }

  GtkWidget *dialog = gtk_message_dialog_new (parent_window,
					      (GtkDialogFlags)
					      (GTK_DIALOG_MODAL |
					       GTK_DIALOG_DESTROY_WITH_PARENT),
					      GTK_MESSAGE_QUESTION,
					      GTK_BUTTONS_YES_NO,
					      _
					      ("A file with the name %s already exists."),
					      file);

  gtk_message_dialog_format_secondary_text (GTK_MESSAGE_DIALOG (dialog),
					    _("Do you want to replace it?"));
  gtk_widget_show_all (dialog);
  gboolean r = (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_YES);
  gtk_widget_destroy (dialog);
  g_free (file);
  g_print ("Yes dialog is %d\n", r);
  return r;
}


/**
 * Save parts to individual files
 */
void
file_savepartswrapper (GtkAction * action, DenemoGUI * gui)
{
  if (!gui->si->filename)
    {
      file_saveas (gui, FALSE);
    }

  export_lilypond_parts (gui->si->filename->str, gui->si, 0, 0);
}
