<?php
/*
Copyright (©) 2003-2013 Teus Benschop.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/


class Database_Search
{
  private static $instance;
  private $db;
  private function __construct() {
    $this->db = Database_SQLite::connect ("search");
  }
  public static function getInstance()
  {
    if (empty (self::$instance)) {
      self::$instance = new Database_Search ();
    }
    return self::$instance;
  }


  public function create ()
  {
    // Searching in SQLite could use the FTS3 or FTS4 module.
    // This module is optional. On some shared hosting account, it is not implemented.
    // Therefore it cannot be used.
    // The search database as created below uses a standard table.
    
$sql = <<<'EOD'
CREATE TABLE IF NOT EXISTS bibles (
  bible text,
  book integer,
  chapter integer,
  verse integer,
  usfmraw text,
  usfmlower text,
  plainraw text,
  plainlower text
);
EOD;
    Database_SQLite::exec ($this->db, $sql);

    $sql = "CREATE INDEX IF NOT EXISTS speedup ON bibles (bible, book, chapter)";
    Database_SQLite::exec ($this->db, $sql);
  }


  public function optimize () 
  {
    Database_SQLite::exec ($this->db, "REINDEX bibles;");
    Database_SQLite::exec ($this->db, "VACUUM bibles;");
  }


  // Updates the search fields for a chapter in Bible $name.
  public function updateSearchFields ($name, $book, $chapter)
  {
    $database_bibles = Database_Bibles::getInstance ();
    $database_config_general = Database_Config_General::getInstance ();

    $usfm = $database_bibles->getChapter ($name, $book, $chapter);
    $stylesheet = $database_config_general->getExportStylesheet ();

    // Data to store.
    $usfmraw = array ();
    $usfmlower = array ();
    $plainraw = array ();
    $plainlower = array ();

    // Get the verses in the chapter.
    $verses = Filter_Usfm::getVerseNumbers ($usfm);
    $verses = array_unique ($verses);
    sort ($verses, SORT_NUMERIC);

    // One column contains the raw USFM as it is, and another one the lowercase text.
    foreach ($verses as $verse) {
      $raw = Filter_Usfm::getVerseText ($usfm, $verse);
      $lower = mb_convert_case ($raw, MB_CASE_LOWER);
      $usfmraw [$verse] = $raw;
      $usfmlower [$verse] = $lower;
    }
    
    // Text filter for getting the plain text.
    $filter_text = new Filter_Text ("");
    $filter_text->text_text = new Text_Text ();
    $filter_text->initializeHeadingsAndTextPerVerse ();
    $filter_text->addUsfmCode ($usfm);
    $filter_text->run ($stylesheet);

    // Get the clean verse texts.
    $texts = $filter_text->getVersesText ();
    foreach ($texts as $verse => $text) {
      if (!isset ($plainraw [$verse])) $plainraw [$verse] = "";
      $plainraw [$verse] .= "$text\n";
    }

    // Add any clean headings.
    $headings = $filter_text->verses_headings;
    foreach ($headings as $verse => $heading) {
      if (!isset ($plainraw [$verse])) $plainraw [$verse] = "";
      $plainraw [$verse] .= "$heading\n";
    }
    
    // Create the lower case plain text.
    foreach ($plainraw as $verse => $raw) {
      $plainlower [$verse] = mb_convert_case ($raw, MB_CASE_LOWER);
    }
    
    // Get all possible verses.
    $allverses = array_merge (array_keys ($usfmraw), array_keys ($plainraw));
    $allverses = array_unique ($allverses);
    
    // Store everything.
    Database_SQLite::exec ($this->db, "BEGIN;");
    $name = Database_SQLiteInjection::no ($name);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $query = "DELETE FROM bibles WHERE bible = '$name' AND book = $book AND chapter = $chapter;";
    Database_SQLite::exec ($this->db, $query);
    foreach ($allverses as $verse) {
      @$ur = Database_SQLiteInjection::no ($usfmraw [$verse]);
      @$ul = Database_SQLiteInjection::no ($usfmlower [$verse]);
      @$pr = Database_SQLiteInjection::no ($plainraw [$verse]);
      @$pl = Database_SQLiteInjection::no ($plainlower [$verse]);
      $query = "INSERT INTO bibles (bible, book, chapter, verse, usfmraw, usfmlower, plainraw, plainlower) VALUES ('$name', $book, $chapter, $verse, '$ur', '$ul', '$pr', '$pl');";
      Database_SQLite::exec ($this->db, $query);
    }
    Database_SQLite::exec ($this->db, "COMMIT;");
  }


  /**
  * Searches the text of the Bibles.
  * Returns an array with the bible / book / chapter / verse of matching verses.
  * If $limit is non-NULL, it indicates the starting limit for the selection.
  * $search: Contains the text to search for.
  */
  public function searchText ($search, $limit = NULL)
  {
    $ids = array ();

    $search = mb_convert_case ($search, MB_CASE_LOWER);
    $search = str_replace (",", "", $search);
    $search = Database_SQLiteInjection::no ($search);

    if (is_numeric ($limit)) {
      $limit = Database_SQLiteInjection::no ($limit);
      $query .= "LIMIT $limit, 50";
    } else {
      $limit = "";
    }

    $query = "SELECT rowid FROM bibles WHERE plainlower LIKE '%$search%' $limit;";
    $hits = array ();
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      $hits [] = $row [0];
    }
    
    return $hits;
  }


  public function debug ()
  {
    $result = Database_SQLite::query ($this->db, "SELECT * from bibles;");
    foreach ($result as $row) {
      for ($i = 0; $i < 10; $i++) unset ($row[$i]);
      var_dump ($row);
    }
  }


  // Returns the Bible and passage for a $rowid.
  public function getBiblePassage ($rowid)
  {
    $rowid = Database_SQLiteInjection::no ($rowid);
    $query = "SELECT bible, book, chapter, verse FROM bibles WHERE rowid = $rowid;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      unset ($row [0]);
      unset ($row [1]);
      unset ($row [2]);
      unset ($row [3]);
      return $row;
    }
    return NULL;
  }
  

  // Gets the plain raw text for the $bible and passage given.  
  public function getBibleVerseText ($bible, $book, $chapter, $verse)
  {
    $bible = Database_SQLiteInjection::no ($bible);
    $book = Database_SQLiteInjection::no ($book);
    $chapter = Database_SQLiteInjection::no ($chapter);
    $verse = Database_SQLiteInjection::no ($verse);
    $query = "SELECT plainraw FROM bibles WHERE bible = '$bible' AND book = $book AND chapter = $chapter AND verse = $verse;";
    $result = Database_SQLite::query ($this->db, $query);
    foreach ($result as $row) {
      return $row[0];
    }
    return "";
  }


}


?>
