#ifndef INTEGERGRID_H
#define INTEGERGRID_H

#include <stdio.h>
#include <stdlib.h>


#define GRID_DIMENSION 20

#define LONG _int64

const int vertmap[8][3] = {{0,0,0},{0,0,1},{0,1,0},{0,1,1},{1,0,0},{1,0,1},{1,1,0},{1,1,1}} ;
const int centmap[3][3][3][2] = 
{{{{0,0},{0,1},{1,1}},{{0,2},{0,3},{1,3}},{{2,2},{2,3},{3,3}}},
{{{0,4},{0,5},{1,5}},{{0,6},{0,7},{1,7}},{{2,6},{2,7},{3,7}}},
{{{4,4},{4,5},{5,5}},{{4,6},{4,7},{5,7}},{{6,6},{6,7},{7,7}}}} ;
const int edgemap[12][2] = {{0,4},{1,5},{2,6},{3,7},{0,2},{1,3},{4,6},{5,7},{0,1},{2,3},{4,5},{6,7}};

/**
 * Structure for the projections inheritable from parent
 */
struct InheritableProjections
{
	/// Projections of the triangle vertices
	LONG trigProj[13][2] ;

	/// End points along each axis
	//int cubeEnds[13][2] ;

	/// Error range on each axis
	/// LONG errorProj[13];
};


/**
 * Class for projections of cube / triangle vertices on the separating axes
 */
class Projections
{
public:
	/// Inheritable portion
	InheritableProjections* inherit ;

	/// Projections of the cube vertices
	LONG cubeProj[13][6] ;

public:

	/**
	 * Default constructor
	 */
	Projections( )
	{
	};

	/** 
	 * Construction
	 * from a cube (axes aligned) and triangle
	 */
	Projections( LONG cube[2][3], LONG trig[3][3], LONG error )
	{
		int i, j ;
		inherit = new InheritableProjections ;

		/// Create axes
		LONG axes[13][3] ;

		// Cube faces
		axes[0][0] = 1 ;
		axes[0][1] = 0 ;
		axes[0][2] = 0 ;

		axes[1][0] = 0 ;
		axes[1][1] = 1 ;
		axes[1][2] = 0 ;

		axes[2][0] = 0 ;
		axes[2][1] = 0 ;
		axes[2][2] = 1 ;

		// Triangle face
		LONG trigedge[3][3] ;
		for ( i = 0 ; i < 3 ; i ++ )
		{
			for ( j = 0 ; j < 3 ; j ++ )
			{
				trigedge[i][j] = trig[(i+1)%3][j] - trig[i][j] ;
			}
		}
		crossProduct( trigedge[0], trigedge[1], axes[3] ) ;

		// Face edges and triangle edges
		int ct = 4 ;
		for ( i = 0 ; i < 3 ; i ++ )
			for ( j = 0 ; j < 3 ; j ++ )
			{
				crossProduct( axes[i], trigedge[j], axes[ct] ) ;
				ct ++ ;
			}		

		/// Generate projections
		LONG cubeedge[3][3] ;
		for ( i = 0 ; i < 3 ; i ++ )
		{
			for ( j = 0 ; j < 3 ; j ++ )
			{
				cubeedge[i][j] = 0 ;
			}
			cubeedge[i][i] = cube[1][i] - cube[0][i] ;
		}

		for ( j = 0 ; j < 13 ; j ++ )
		{
			// Origin
			cubeProj[j][0] = dotProduct( axes[j], cube[0] ) ;

			// 3 direction vectors
			for ( i = 1 ; i < 4 ; i ++ )
			{
				cubeProj[j][i] = dotProduct( axes[j], cubeedge[i-1] ) ;
			}

			// Offsets of 2 ends of cube projection
			LONG max = 0 ;
			LONG min = 0 ;
			int maxind = 0, minind = 0 ;
			for ( i = 1 ; i < 8 ; i ++ )
			{
				LONG proj = vertmap[i][0] * cubeProj[j][1] + vertmap[i][1] * cubeProj[j][2] + vertmap[i][2] * cubeProj[j][3] ;
				if ( proj > max )
				{
					max = proj ;
					maxind = i ;
				}
				if ( proj < min )
				{
					min = proj ;
					minind = i ;
				}
			}
			cubeProj[j][4] = min ;
			cubeProj[j][5] = max ;

		}

		for ( j = 0 ; j < 13 ; j ++ )
		{
			inherit->trigProj[j][0] = dotProduct( axes[j], trig[0] ) ;
			inherit->trigProj[j][1] = inherit->trigProj[j][0] ;

			for ( i = 1 ; i < 3 ; i ++ )
			{
				LONG proj = dotProduct( axes[j], trig[i] ) ;
				if ( proj < inherit->trigProj[j][0] )
				{
					inherit->trigProj[j][0] = proj ;
				}
				if ( proj > inherit->trigProj[j][1] )
				{
					inherit->trigProj[j][1] = proj ;
				}
			}
		}

		// Get ends of projections
		/*
		for ( i = 0 ; i < 13 ; i ++ )
		{
			int maxind = 0, minind = 0 ;
			LONG max = cubeProj[i][0] ;
			LONG min = cubeProj[i][0] ;
			for ( j = 1 ; j < 8 ; j ++ )
			{
				LONG proj = cubeProj[i][0] + vertmap[j][0] * cubeProj[i][1] + vertmap[j][1] * cubeProj[i][2] + vertmap[j][2] * cubeProj[i][3] ;
				if ( proj > max )
				{
					max = proj ;
					maxind = j ;
				}
				if ( proj < min )
				{
					min = proj ;
					minind = j ;
				}
			}

			inherit->cubeEnds[i][0] = minind ;
			inherit->cubeEnds[i][1] = maxind ;
		}
		*/

		// exit(0) ;
	}

	/**
	 * Construction
	 * from a parent Projections object and the index of the children
	 */
	Projections ( Projections* parent ) 
	{
		// Copy inheritable projections
		this->inherit = parent->inherit ;

		// Shrink cube projections
		for ( int i = 0 ; i < 13 ; i ++ )
		{
			cubeProj[i][0] = parent->cubeProj[i][0] ;
			for ( int j = 1 ; j < 6 ; j ++ )
			{
				cubeProj[i][j] = parent->cubeProj[i][j] >> 1 ;
			}
		}
	};

	/**
	 * Get projections for sub-cubes (simple axes)
	 */
	void getSubProjectionsSimple( Projections* p[8] )
	{
		// Process the axes cooresponding to the triangle's normal
		int ind = 3 ;
		int len = cubeProj[ 0 ][ 1 ] >> 1 ;
		int trigproj[3] = { cubeProj[ ind ][ 1 ] >> 1, cubeProj[ ind ][ 2 ] >> 1, cubeProj[ ind ][ 3 ] >> 1 } ;

		int ct = 0 ; 
		for ( int i = 0 ; i < 2 ; i ++ )
			for ( int j = 0 ; j < 2 ; j ++ )
				for ( int k = 0 ; k < 2 ; k ++ )
				{
					p[ct] = new Projections( ) ;

					p[ct]->cubeProj[ 0 ][ 0 ] = cubeProj[ 0 ][ 0 ] + i * len ;
					p[ct]->cubeProj[ 1 ][ 0 ] = cubeProj[ 1 ][ 0 ] + j * len ;
					p[ct]->cubeProj[ 2 ][ 0 ] = cubeProj[ 2 ][ 0 ] + k * len ;
					p[ct]->cubeProj[ 0 ][ 1 ] = len ;

					for ( int m = 1 ; m < 3 ; m ++ )
					{
						p[ct]->cubeProj[ ind ][ m ] = trigproj[ m - 1 ] ;
					}
					p[ct]->cubeProj[ ind ][ 0 ] = cubeProj[ ind ][0] + i * trigproj[0] + j * trigproj[1] + k * trigproj[2] ;

					ct ++ ;
				}
	}

	/**
	 * Shifting a cube to a new origin
	 */
	void shift ( int off[3] ) 
	{
		for ( int i = 0 ; i < 13 ; i ++ )
		{
			cubeProj[i][0] += off[0] * cubeProj[i][1] + off[1] * cubeProj[i][2] + off[2] * cubeProj[i][3] ;
		}
	}

	void shiftNoPrimary ( int off[3] ) 
	{
		for ( int i = 3 ; i < 13 ; i ++ )
		{
			cubeProj[i][0] += off[0] * cubeProj[i][1] + off[1] * cubeProj[i][2] + off[2] * cubeProj[i][3] ;
		}
	}

	/**
	 * Method to test intersection of the triangle and the cube
	 */
	int isIntersecting ( ) 
	{
		for ( int i = 0 ; i < 13 ; i ++ )
		{
			/*
			LONG proj0 = cubeProj[i][0] + 
				vertmap[inherit->cubeEnds[i][0]][0] * cubeProj[i][1] + 
				vertmap[inherit->cubeEnds[i][0]][1] * cubeProj[i][2] + 
				vertmap[inherit->cubeEnds[i][0]][2] * cubeProj[i][3] ;
			LONG proj1 = cubeProj[i][0] + 
				vertmap[inherit->cubeEnds[i][1]][0] * cubeProj[i][1] + 
				vertmap[inherit->cubeEnds[i][1]][1] * cubeProj[i][2] + 
				vertmap[inherit->cubeEnds[i][1]][2] * cubeProj[i][3] ;
			*/

			LONG proj0 = cubeProj[i][0] + cubeProj[i][4] ;
			LONG proj1 = cubeProj[i][0] + cubeProj[i][5] ;

			if ( proj0 > inherit->trigProj[i][1] ||
				 proj1 < inherit->trigProj[i][0] )
			{
				return 0 ;
			}
		}
		
		return 1 ;
	};

	int isIntersectingNoPrimary ( ) 
	{
		for ( int i = 3 ; i < 13 ; i ++ )
		{
			/*
			LONG proj0 = cubeProj[i][0] + 
				vertmap[inherit->cubeEnds[i][0]][0] * cubeProj[i][1] + 
				vertmap[inherit->cubeEnds[i][0]][1] * cubeProj[i][2] + 
				vertmap[inherit->cubeEnds[i][0]][2] * cubeProj[i][3] ;
			LONG proj1 = cubeProj[i][0] + 
				vertmap[inherit->cubeEnds[i][1]][0] * cubeProj[i][1] + 
				vertmap[inherit->cubeEnds[i][1]][1] * cubeProj[i][2] + 
				vertmap[inherit->cubeEnds[i][1]][2] * cubeProj[i][3] ;
			*/

			LONG proj0 = cubeProj[i][0] + cubeProj[i][4] ;
			LONG proj1 = cubeProj[i][0] + cubeProj[i][5] ;

			if ( proj0 > inherit->trigProj[i][1] ||
				 proj1 < inherit->trigProj[i][0] )
			{
				return 0 ;
			}
		}
		
		return 1 ;
	};	
	
	/**
	 * Method to test intersection of the triangle and one edge
	 */
	int isIntersecting ( int edgeInd ) 
	{
		for ( int i = 0 ; i < 13 ; i ++ )
		{
			
			LONG proj0 = cubeProj[i][0] + 
				vertmap[edgemap[edgeInd][0]][0] * cubeProj[i][1] + 
				vertmap[edgemap[edgeInd][0]][1] * cubeProj[i][2] + 
				vertmap[edgemap[edgeInd][0]][2] * cubeProj[i][3] ;
			LONG proj1 = cubeProj[i][0] + 
				vertmap[edgemap[edgeInd][1]][0] * cubeProj[i][1] + 
				vertmap[edgemap[edgeInd][1]][1] * cubeProj[i][2] + 
				vertmap[edgemap[edgeInd][1]][2] * cubeProj[i][3] ;


			if ( proj0 < proj1 )
			{
				if ( proj0 > inherit->trigProj[i][1] ||
					 proj1 < inherit->trigProj[i][0] )
				{
					return 0 ;
				}
			}
			else
			{
				if ( proj1 > inherit->trigProj[i][1] ||
					 proj0 < inherit->trigProj[i][0] )
				{
					return 0 ;
				}
			}
		}
		
		// printf( "Intersecting: %d %d\n", edgemap[edgeInd][0], edgemap[edgeInd][1] )  ;
		return 1 ;
	};

	int isIntersectingPrimary ( int edgeInd ) 
	{
		for ( int i = 0 ; i < 13 ; i ++ )
		{
			
			LONG proj0 = cubeProj[i][0] ;
			LONG proj1 = cubeProj[i][0] + cubeProj[i][edgeInd + 1] ;

			if ( proj0 < proj1 )
			{
				if ( proj0 > inherit->trigProj[i][1] ||
					 proj1 < inherit->trigProj[i][0] )
				{
					return 0 ;
				}
			}
			else
			{
				if ( proj1 > inherit->trigProj[i][1] ||
					 proj0 < inherit->trigProj[i][0] )
				{
					return 0 ;
				}
			}

		}
		
		// printf( "Intersecting: %d %d\n", edgemap[edgeInd][0], edgemap[edgeInd][1] )  ;
		return 1 ;
	};

	float getIntersection ( int edgeInd ) 
	{
		int i = 3 ;

		LONG proj0 = cubeProj[i][0] + 
			vertmap[edgemap[edgeInd][0]][0] * cubeProj[i][1] + 
			vertmap[edgemap[edgeInd][0]][1] * cubeProj[i][2] + 
			vertmap[edgemap[edgeInd][0]][2] * cubeProj[i][3] ;
		LONG proj1 = cubeProj[i][0] + 
			vertmap[edgemap[edgeInd][1]][0] * cubeProj[i][1] + 
			vertmap[edgemap[edgeInd][1]][1] * cubeProj[i][2] + 
			vertmap[edgemap[edgeInd][1]][2] * cubeProj[i][3] ;
		LONG proj2 = inherit->trigProj[i][1] ;

		double alpha = (double)( proj2 - proj0 ) / (double)( proj1 - proj0 ) ;
		if ( alpha < 0 )
		{
			alpha = 0.5 ;
		}
		else if ( alpha > 1 )
		{
			alpha = 0.5 ;
		}

		return (float)alpha ;
	};

	float getIntersectionPrimary ( int edgeInd ) 
	{
		int i = 3 ;
		
		LONG proj0 = cubeProj[i][0] ;
		LONG proj1 = cubeProj[i][0] + cubeProj[i][edgeInd + 1] ;
		LONG proj2 = inherit->trigProj[i][1] ;

		double alpha = (double)( proj2 - proj0 ) / (double)( proj1 - proj0 ) ;
		if ( alpha < 0 )
		{
			alpha = 0.5 ;
		}
		else if ( alpha > 1 )
		{
			alpha = 0.5 ;
		}

		return (float)alpha ;
	};

	/**
	 * Method to perform cross-product
	 */
	void crossProduct ( LONG a[3], LONG b[3], LONG res[3] )
	{
		res[0] = a[1] * b[2] - a[2] * b[1] ;
		res[1] = a[2] * b[0] - a[0] * b[2] ;
		res[2] = a[0] * b[1] - a[1] * b[0] ;
	}

	/**
	 * Method to perform dot product
	 */
	LONG dotProduct ( LONG a[3], LONG b[3] )
	{
		return a[0] * b[0] + a[1] * b[1] + a[2] * b[2] ;
	}

};

/**
 * Class for projections of cube / triangle vertices on the separating axes
 */
class Projections2
{
public:
	/// Projections of the cube vertices
	LONG cubeProj[13][8] ;

	/// Projections of the triangle vertices
	LONG trigProj[13][3] ;

	/// Min max indices
	int cubeEnds[13][2] ;
	int trigEnds[13][2] ;

public:

	/**
	 * Default construction
	 */
	Projections2 (  )
	{
	};

	/** 
	 * Construction
	 * from a cube (axes aligned) and triangle
	 */
	Projections2 ( LONG cube[2][3], LONG trig[3][3] )
	{
		int i, j ;

		/// Create axes
		LONG axes[13][3] ;

		// Cube faces
		axes[0][0] = 1 ;
		axes[0][1] = 0 ;
		axes[0][2] = 0 ;

		axes[1][0] = 0 ;
		axes[1][1] = 1 ;
		axes[1][2] = 0 ;

		axes[2][0] = 0 ;
		axes[2][1] = 0 ;
		axes[2][2] = 1 ;

		// Triangle face
		LONG trigedge[3][3] ;
		for ( i = 0 ; i < 3 ; i ++ )
		{
			for ( j = 0 ; j < 3 ; j ++ )
			{
				trigedge[i][j] = trig[(i+1)%3][j] - trig[i][j] ;
			}
		}
		crossProduct( trigedge[0], trigedge[1], axes[3] ) ;

		// Face edges and triangle edges
		int ct = 4 ;
		for ( i = 0 ; i < 3 ; i ++ )
			for ( j = 0 ; j < 3 ; j ++ )
			{
				crossProduct( axes[i], trigedge[j], axes[ct] ) ;
				ct ++ ;
			}
		

		/// Generate projections
		LONG theCube[8][3] ;
		for ( i = 0 ; i < 8 ; i ++ )
			for ( j = 0 ; j < 3 ; j ++ )
			{
				theCube[i][j] = cube[vertmap[i][j]][j] ;
			}

		for ( i = 0 ; i < 8 ; i ++ )
		{
			for ( j = 0 ; j < 13 ; j ++ )
			{
				cubeProj[j][i] = dotProduct( axes[j], theCube[i] ) ;
			}
		}

		for ( i = 0 ; i < 3 ; i ++ )
			for ( j = 0 ; j < 13 ; j ++ )
			{
				trigProj[j][i] = dotProduct( axes[j], trig[i] ) ;
			}

		// Get ends of projections
		for ( i = 0 ; i < 13 ; i ++ )
		{
			int maxind = 0, minind = 0 ;
			LONG max = cubeProj[i][0] ;
			LONG min = cubeProj[i][0] ;
			for ( j = 1 ; j < 8 ; j ++ )
			{
				if ( cubeProj[i][j] > max )
				{
					max = cubeProj[i][j] ;
					maxind = j ;
				}
				if ( cubeProj[i][j] < min )
				{
					min = cubeProj[i][j] ;
					minind = j ;
				}
			}

			cubeEnds[i][0] = minind ;
			cubeEnds[i][1] = maxind ;

			maxind = 0 ;
			minind = 0 ;
			max = trigProj[i][0] ;
			min = trigProj[i][0] ;
			for ( j = 1 ; j < 3 ; j ++ )
			{
				if ( trigProj[i][j] > max )
				{
					max = trigProj[i][j] ;
					maxind = j ;
				}
				if ( trigProj[i][j] < min )
				{
					min = trigProj[i][j] ;
					minind = j ;
				}
			}

			trigEnds[i][0] = minind ;
			trigEnds[i][1] = maxind ;
		
		}
	
	};

	/**
	 * Construction
	 * from a parent Projections object and the index of the children
	 */
	Projections2 ( Projections2* parent, int childInd ) 
	{
		int i, j ;
		
		// Copy triangle projections
		for ( i = 0 ; i < 13 ; i ++ )
		{
			for ( j = 0 ; j < 3 ; j ++ )
			{
				trigProj[i][j] = parent->trigProj[i][j] ;
			}

			trigEnds[i][0] = parent->trigEnds[i][0] ;
			trigEnds[i][1] = parent->trigEnds[i][1] ;
		}
		

		// Divide cube projections
		for ( i = 0 ; i < 13 ; i ++ )
		{
			for ( j = 0 ; j < 8 ; j ++ )
			{
				const int* ind = centmap[vertmap[childInd][0] + vertmap[j][0]]
									[vertmap[childInd][1] + vertmap[j][1]]
									[vertmap[childInd][2] + vertmap[j][2]] ;
				cubeProj[i][j] = ( parent->cubeProj[i][ind[0]] + parent->cubeProj[i][ind[1]] ) >> 1 ;
			}

			cubeEnds[i][0] = parent->cubeEnds[i][0] ;
			cubeEnds[i][1] = parent->cubeEnds[i][1] ;
		}

	};

	/**
	 * Inherit projections from father
	 */
	void inherit ( Projections2* parent, int childInd )
	{
		int i, j ;

		// Copy triangle projections
		for ( i = 0 ; i < 13 ; i ++ )
		{
			for ( j = 0 ; j < 3 ; j ++ )
			{
				trigProj[i][j] = parent->trigProj[i][j] ;
			}

			trigEnds[i][0] = parent->trigEnds[i][0] ;
			trigEnds[i][1] = parent->trigEnds[i][1] ;
		}

		// Divide cube projections
		for ( i = 0 ; i < 13 ; i ++ )
		{
			for ( j = 0 ; j < 8 ; j ++ )
			{
				const int* ind = centmap[vertmap[childInd][0] + vertmap[j][0]]
									[vertmap[childInd][1] + vertmap[j][1]]
									[vertmap[childInd][2] + vertmap[j][2]] ;
				cubeProj[i][j] = ( parent->cubeProj[i][ind[0]] + parent->cubeProj[i][ind[1]] ) >> 1 ;
			}

			cubeEnds[i][0] = parent->cubeEnds[i][0] ;
			cubeEnds[i][1] = parent->cubeEnds[i][1] ;
		}

	}

	/**
	 * Destruction
	 */
	~Projections2 ( ) 
	{
			
	};

	/**
	 * Method to test intersection of the triangle and the cube
	 */
	int isIntersecting ( ) 
	{
		for ( int i = 0 ; i < 13 ; i ++ )
		{
			if ( cubeProj[i][cubeEnds[i][0]] > trigProj[i][trigEnds[i][1]] ||
				 cubeProj[i][cubeEnds[i][1]] < trigProj[i][trigEnds[i][0]] )
			{
				return 0 ;
			}
		}
		
		return 1 ;
	};

	/**
	 * Method to test intersection of the triangle and one edge
	 */
	int isIntersecting ( int edgeInd ) 
	{
		
		for ( int i = 0 ; i < 13 ; i ++ )
		{
			LONG proj0 = cubeProj[i][edgemap[edgeInd][0]] ;
			LONG proj1 = cubeProj[i][edgemap[edgeInd][1]] ;

			// Put in order
			if ( proj0 > proj1 )
			{
				LONG temp = proj0 ;
				proj0 = proj1 ;
				proj1 = temp ;
			}

			if ( proj0 > trigProj[i][trigEnds[i][1]] ||
				 proj1 < trigProj[i][trigEnds[i][0]] )
			{
				return 0 ;
			}
		}
		
		return 1 ;
	};

	/**
	 * Method to perform cross-product
	 */
	void crossProduct ( LONG a[3], LONG b[3], LONG res[3] )
	{
		res[0] = a[1] * b[2] - a[2] * b[1] ;
		res[1] = a[2] * b[0] - a[0] * b[2] ;
		res[2] = a[0] * b[1] - a[1] * b[0] ;
	}

	/**
	 * Method to perform dot product
	 */
	LONG dotProduct ( LONG a[3], LONG b[3] )
	{
		return a[0] * b[0] + a[1] * b[1] + a[2] * b[2] ;
	}
};

#endif