/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.yarn.server.nodemanager.util;

import com.google.common.annotations.VisibleForTesting;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.io.FileUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.yarn.api.records.ContainerId;
import org.apache.hadoop.yarn.api.records.Resource;
import org.apache.hadoop.yarn.server.nodemanager.LinuxContainerExecutor;
import org.apache.hadoop.yarn.server.nodemanager.util.LCEResourcesHandler;
import org.apache.hadoop.yarn.server.nodemanager.util.NodeManagerHardwareUtils;
import org.apache.hadoop.yarn.util.Clock;
import org.apache.hadoop.yarn.util.ResourceCalculatorPlugin;
import org.apache.hadoop.yarn.util.SystemClock;

public class CgroupsLCEResourcesHandler
implements LCEResourcesHandler {
    static final Log LOG = LogFactory.getLog(CgroupsLCEResourcesHandler.class);
    private Configuration conf;
    private String cgroupPrefix;
    private boolean cgroupMount;
    private String cgroupMountPath;
    private boolean cpuWeightEnabled = true;
    private boolean strictResourceUsageMode = false;
    private final String MTAB_FILE = "/proc/mounts";
    private final String CGROUPS_FSTYPE = "cgroup";
    private final String CONTROLLER_CPU = "cpu";
    private final String CPU_PERIOD_US = "cfs_period_us";
    private final String CPU_QUOTA_US = "cfs_quota_us";
    private final int CPU_DEFAULT_WEIGHT = 1024;
    private final int MAX_QUOTA_US = 1000000;
    private final int MIN_PERIOD_US = 1000;
    private final Map<String, String> controllerPaths = new HashMap<String, String>();
    private long deleteCgroupTimeout;
    Clock clock = new SystemClock();
    private float yarnProcessors;
    private static final Pattern MTAB_FILE_FORMAT = Pattern.compile("^[^\\s]+\\s([^\\s]+)\\s([^\\s]+)\\s([^\\s]+)\\s[^\\s]+\\s[^\\s]+$");

    public void setConf(Configuration conf) {
        this.conf = conf;
    }

    public Configuration getConf() {
        return this.conf;
    }

    @VisibleForTesting
    void initConfig() throws IOException {
        this.cgroupPrefix = this.conf.get("yarn.nodemanager.linux-container-executor.cgroups.hierarchy", "/hadoop-yarn");
        this.cgroupMount = this.conf.getBoolean("yarn.nodemanager.linux-container-executor.cgroups.mount", false);
        this.cgroupMountPath = this.conf.get("yarn.nodemanager.linux-container-executor.cgroups.mount-path", null);
        this.deleteCgroupTimeout = this.conf.getLong("yarn.nodemanager.linux-container-executor.cgroups.delete-timeout-ms", 1000L);
        if (this.cgroupPrefix.charAt(0) == '/') {
            this.cgroupPrefix = this.cgroupPrefix.substring(1);
        }
        this.strictResourceUsageMode = this.conf.getBoolean("yarn.nodemanager.linux-container-executor.cgroups.strict-resource-usage", false);
        int len = this.cgroupPrefix.length();
        if (this.cgroupPrefix.charAt(len - 1) == '/') {
            this.cgroupPrefix = this.cgroupPrefix.substring(0, len - 1);
        }
    }

    @Override
    public void init(LinuxContainerExecutor lce) throws IOException {
        this.init(lce, ResourceCalculatorPlugin.getResourceCalculatorPlugin(null, (Configuration)this.conf));
    }

    @VisibleForTesting
    void init(LinuxContainerExecutor lce, ResourceCalculatorPlugin plugin) throws IOException {
        this.initConfig();
        if (this.cgroupMount && this.cgroupMountPath != null) {
            ArrayList<String> cgroupKVs = new ArrayList<String>();
            cgroupKVs.add("cpu=" + this.cgroupMountPath + "/" + "cpu");
            lce.mountCgroups(cgroupKVs, this.cgroupPrefix);
        }
        this.initializeControllerPaths();
        this.yarnProcessors = NodeManagerHardwareUtils.getContainersCores(plugin, this.conf);
        int systemProcessors = plugin.getNumProcessors();
        if (systemProcessors != (int)this.yarnProcessors) {
            LOG.info((Object)("YARN containers restricted to " + this.yarnProcessors + " cores"));
            int[] limits = this.getOverallLimits(this.yarnProcessors);
            this.updateCgroup("cpu", "", "cfs_period_us", String.valueOf(limits[0]));
            this.updateCgroup("cpu", "", "cfs_quota_us", String.valueOf(limits[1]));
        } else if (this.cpuLimitsExist()) {
            LOG.info((Object)"Removing CPU constraints for YARN containers.");
            this.updateCgroup("cpu", "", "cfs_quota_us", String.valueOf(-1));
        }
    }

    boolean cpuLimitsExist() throws IOException {
        String contents;
        int quotaUS;
        String path = this.pathForCgroup("cpu", "");
        File quotaFile = new File(path, "cpu.cfs_quota_us");
        return quotaFile.exists() && (quotaUS = Integer.parseInt((contents = FileUtils.readFileToString((File)quotaFile, (String)"UTF-8")).trim())) != -1;
    }

    @VisibleForTesting
    int[] getOverallLimits(float yarnProcessors) {
        int[] ret = new int[2];
        if (yarnProcessors < 0.01f) {
            throw new IllegalArgumentException("Number of processors can't be <= 0.");
        }
        int quotaUS = 1000000;
        int periodUS = (int)(1000000.0f / yarnProcessors);
        if (yarnProcessors < 1.0f && (quotaUS = (int)((float)(periodUS = 1000000) * yarnProcessors)) < 1000) {
            LOG.warn((Object)("The quota calculated for the cgroup was too low. The minimum value is 1000, calculated value is " + quotaUS + ". Setting quota to minimum value."));
            quotaUS = 1000;
        }
        if (periodUS < 1000) {
            LOG.warn((Object)("The period calculated for the cgroup was too low. The minimum value is 1000, calculated value is " + periodUS + ". Using all available CPU."));
            periodUS = 1000000;
            quotaUS = -1;
        }
        ret[0] = periodUS;
        ret[1] = quotaUS;
        return ret;
    }

    boolean isCpuWeightEnabled() {
        return this.cpuWeightEnabled;
    }

    private String pathForCgroup(String controller, String groupName) {
        String controllerPath = this.controllerPaths.get(controller);
        return controllerPath + "/" + this.cgroupPrefix + "/" + groupName;
    }

    private void createCgroup(String controller, String groupName) throws IOException {
        String path = this.pathForCgroup(controller, groupName);
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("createCgroup: " + path));
        }
        if (!new File(path).mkdir()) {
            throw new IOException("Failed to create cgroup at " + path);
        }
    }

    private void updateCgroup(String controller, String groupName, String param, String value) throws IOException {
        OutputStreamWriter f = null;
        String path = this.pathForCgroup(controller, groupName);
        param = controller + "." + param;
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("updateCgroup: " + path + ": " + param + "=" + value));
        }
        try {
            f = new FileWriter(path + "/" + param, false);
            f.write(value);
        }
        catch (IOException e) {
            throw new IOException("Unable to set " + param + "=" + value + " for cgroup at: " + path, e);
        }
        finally {
            if (f != null) {
                try {
                    f.close();
                }
                catch (IOException e) {
                    LOG.warn((Object)("Unable to close cgroup file: " + path), (Throwable)e);
                }
            }
        }
    }

    @VisibleForTesting
    boolean deleteCgroup(String cgroupPath) {
        boolean deleted;
        if (LOG.isDebugEnabled()) {
            LOG.debug((Object)("deleteCgroup: " + cgroupPath));
        }
        long start = this.clock.getTime();
        do {
            if (deleted = new File(cgroupPath).delete()) continue;
            try {
                Thread.sleep(20L);
            }
            catch (InterruptedException ex) {
                // empty catch block
            }
        } while (!deleted && this.clock.getTime() - start < this.deleteCgroupTimeout);
        if (!deleted) {
            LOG.warn((Object)("Unable to delete cgroup at: " + cgroupPath + ", tried to delete for " + this.deleteCgroupTimeout + "ms"));
        }
        return deleted;
    }

    private void setupLimits(ContainerId containerId, Resource containerResource) throws IOException {
        String containerName = containerId.toString();
        if (this.isCpuWeightEnabled()) {
            int nodeVCores;
            int containerVCores = containerResource.getVirtualCores();
            this.createCgroup("cpu", containerName);
            int cpuShares = 1024 * containerVCores;
            this.updateCgroup("cpu", containerName, "shares", String.valueOf(cpuShares));
            if (this.strictResourceUsageMode && (nodeVCores = this.conf.getInt("yarn.nodemanager.resource.cpu-vcores", 8)) != containerVCores) {
                float containerCPU = (float)containerVCores * this.yarnProcessors / (float)nodeVCores;
                int[] limits = this.getOverallLimits(containerCPU);
                this.updateCgroup("cpu", containerName, "cfs_period_us", String.valueOf(limits[0]));
                this.updateCgroup("cpu", containerName, "cfs_quota_us", String.valueOf(limits[1]));
            }
        }
    }

    private void clearLimits(ContainerId containerId) {
        if (this.isCpuWeightEnabled()) {
            this.deleteCgroup(this.pathForCgroup("cpu", containerId.toString()));
        }
    }

    @Override
    public void preExecute(ContainerId containerId, Resource containerResource) throws IOException {
        this.setupLimits(containerId, containerResource);
    }

    @Override
    public void postExecute(ContainerId containerId) {
        this.clearLimits(containerId);
    }

    @Override
    public String getResourcesOption(ContainerId containerId) {
        String containerName = containerId.toString();
        StringBuilder sb = new StringBuilder("cgroups=");
        if (this.isCpuWeightEnabled()) {
            sb.append(this.pathForCgroup("cpu", containerName) + "/tasks");
            sb.append(",");
        }
        if (sb.charAt(sb.length() - 1) == ',') {
            sb.deleteCharAt(sb.length() - 1);
        }
        return sb.toString();
    }

    private Map<String, List<String>> parseMtab() throws IOException {
        HashMap<String, List<String>> ret = new HashMap<String, List<String>>();
        BufferedReader in = null;
        try {
            in = new BufferedReader(new FileReader(new File(this.getMtabFileName())));
            String str = in.readLine();
            while (str != null) {
                Matcher m = MTAB_FILE_FORMAT.matcher(str);
                boolean mat = m.find();
                if (mat) {
                    String path = m.group(1);
                    String type = m.group(2);
                    String options = m.group(3);
                    if (type.equals("cgroup")) {
                        List<String> value = Arrays.asList(options.split(","));
                        ret.put(path, value);
                    }
                }
                str = in.readLine();
            }
        }
        catch (IOException e) {
            throw new IOException("Error while reading " + this.getMtabFileName(), e);
        }
        finally {
            try {
                in.close();
            }
            catch (IOException e2) {
                LOG.warn((Object)("Error closing the stream: " + this.getMtabFileName()), (Throwable)e2);
            }
        }
        return ret;
    }

    private String findControllerInMtab(String controller, Map<String, List<String>> entries) {
        for (Map.Entry<String, List<String>> e : entries.entrySet()) {
            if (!e.getValue().contains(controller)) continue;
            return e.getKey();
        }
        return null;
    }

    private void initializeControllerPaths() throws IOException {
        Map<String, List<String>> parsedMtab = this.parseMtab();
        String controllerPath = this.findControllerInMtab("cpu", parsedMtab);
        if (controllerPath != null) {
            File f = new File(controllerPath + "/" + this.cgroupPrefix);
            if (!FileUtil.canWrite((File)f)) {
                throw new IOException("Not able to enforce cpu weights; cannot write to cgroup at: " + controllerPath);
            }
        } else {
            throw new IOException("Not able to enforce cpu weights; cannot find cgroup for cpu controller in " + this.getMtabFileName());
        }
        this.controllerPaths.put("cpu", controllerPath);
    }

    @VisibleForTesting
    String getMtabFileName() {
        return "/proc/mounts";
    }
}

