/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.myfaces.orchestra.lib.jsf;

import javax.faces.context.FacesContext;
import javax.faces.context.FacesContextWrapper;


/**
 * Convenient class to wrap the current FacesContext.
 * <p>
 * A class of this name is provided in JSF1.2, but not in JSF1.1.
 * <p>
 * Any methods that do not actually need to be overridden are declared final
 * in order to improve performance (helps the JVM to optimise away the call).
 * <p>
 * Note that whether a newly-created instance immediately becomes the
 * object that is returned by FacesContext.getCurrentInstance() depends
 * upon the value of the "install" parameter for the constructor method. 
 * <p>
 * This class is copied from the code in MyFaces Core Impl 1.2.x, but
 * modified to be compatible with JSF1.1.
 * <p>
 * Note that this class must be public in order to support custom
 * FacesContextFactory classes in other libraries that also wrap this
 * instance, then use reflection to invoke methods on this object. In
 * this case, an IllegalAccessException would occur if this class was
 * package-scoped. However this class is NOT intended to be part of the
 * public Orchestra API, and may change at any time.
 * 
 * @since 1.1
 * 
 * @author Manfred Geiler (latest modification by $Author: skitching $)
 * @author Anton Koinov
 * @version $Revision: 672906 $ $Date: 2008-06-30 15:45:16 -0500 (lun, 30 jun 2008) $
 */
public class _FacesContextWrapper extends FacesContextWrapper
{
    //~ Instance fields -------------------------------------------------------

    private final FacesContext _facesContext;

    //~ Constructors ----------------------------------------------------------

    /**
     * The install parameter controls whether this object will be configured as
     * the object returned from calls to FacesContext.getCurrentInstance() or not.
     * <p>
     * When only overriding the release() method, then install=false is ok as that
     * is called directly by the FacesServlet on the instance returned by the
     * FacesContextFactory. However all other methods are invoked on the object
     * that is returned from FacesContext.getCurrentInstance, so install=true is
     * needed in order for any other method overrides to have any effect.
     * <p>
     * <b>IMPORTANT</b>: install=true should not be used until MYFACES-1820 is fixed.
     */
    public _FacesContextWrapper(FacesContext facesContext, boolean install)
    {
        _facesContext = facesContext;
        
        if (install)
        {
            FacesContext.setCurrentInstance(this);
        }
    }

    //~ Non-Final Methods -----------------------------------------------------

    @Override
    public FacesContext getWrapped()
    {
        return _facesContext;
    }

    //~ Final Methods ---------------------------------------------------------

}
