/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.backends.postgres;

import com.google.common.annotations.VisibleForTesting;
import io.r2dbc.spi.Connection;
import io.r2dbc.spi.Result;
import jakarta.inject.Inject;
import java.util.List;
import org.apache.james.backends.postgres.PostgresConfiguration;
import org.apache.james.backends.postgres.PostgresDataDefinition;
import org.apache.james.backends.postgres.PostgresIndex;
import org.apache.james.backends.postgres.PostgresTable;
import org.apache.james.backends.postgres.RowLevelSecurity;
import org.apache.james.backends.postgres.utils.PostgresExecutor;
import org.apache.james.lifecycle.api.Startable;
import org.jooq.DSLContext;
import org.jooq.SelectField;
import org.jooq.exception.DataAccessException;
import org.jooq.impl.DSL;
import org.reactivestreams.Publisher;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;

public class PostgresTableManager
implements Startable {
    public static final int INITIALIZATION_PRIORITY = 1;
    private static final Logger LOGGER = LoggerFactory.getLogger(PostgresTableManager.class);
    private final PostgresExecutor postgresExecutor;
    private final PostgresDataDefinition module;
    private final RowLevelSecurity rowLevelSecurity;

    @Inject
    public PostgresTableManager(PostgresExecutor postgresExecutor, PostgresDataDefinition module, PostgresConfiguration postgresConfiguration) {
        this.postgresExecutor = postgresExecutor;
        this.module = module;
        this.rowLevelSecurity = postgresConfiguration.getRowLevelSecurity();
    }

    @VisibleForTesting
    public PostgresTableManager(PostgresExecutor postgresExecutor, PostgresDataDefinition module, RowLevelSecurity rowLevelSecurity) {
        this.postgresExecutor = postgresExecutor;
        this.module = module;
        this.rowLevelSecurity = rowLevelSecurity;
    }

    public void initPostgres() {
        this.initializePostgresExtension().then(this.initializeTables()).then(this.initializeTableIndexes()).block();
    }

    public Mono<Void> initializePostgresExtension() {
        return Mono.usingWhen(this.postgresExecutor.connectionFactory().getConnection(), connection -> Mono.just((Object)connection).flatMapMany(pgConnection -> pgConnection.createStatement("CREATE EXTENSION IF NOT EXISTS hstore").execute()).flatMap(Result::getRowsUpdated).then(), connection -> this.postgresExecutor.connectionFactory().closeConnection((Connection)connection));
    }

    public Mono<Void> initializeTables() {
        return Mono.usingWhen(this.postgresExecutor.connectionFactory().getConnection(), connection -> this.postgresExecutor.dslContext((Connection)connection).flatMapMany(dsl -> this.listExistTables().flatMapMany(existTables -> Flux.fromIterable(this.module.tables()).filter(table -> !existTables.contains(table.getName())).flatMap(table -> this.createAndAlterTable((PostgresTable)table, (DSLContext)dsl, (Connection)connection)))).then(), connection -> this.postgresExecutor.connectionFactory().closeConnection((Connection)connection));
    }

    private Mono<Void> createAndAlterTable(PostgresTable table, DSLContext dsl, Connection connection) {
        return Mono.from((Publisher)((Publisher)table.getCreateTableStepFunction().apply(dsl))).then(this.alterTableIfNeeded(table, connection)).doOnSuccess(any -> LOGGER.info("Table {} created", (Object)table.getName())).onErrorResume(exception -> this.handleTableCreationException(table, (Throwable)exception));
    }

    public Mono<List<String>> listExistTables() {
        return Mono.usingWhen(this.postgresExecutor.connectionFactory().getConnection(), connection -> this.postgresExecutor.dslContext((Connection)connection).flatMapMany(d -> Flux.from((Publisher)d.select((SelectField)DSL.field((String)"tablename")).from("pg_tables").where(DSL.field((String)"schemaname").eq((Object)DSL.currentSchema())))).map(r -> (String)r.get(0, String.class)).collectList(), connection -> this.postgresExecutor.connectionFactory().closeConnection((Connection)connection));
    }

    private Mono<Void> handleTableCreationException(PostgresTable table, Throwable e) {
        if (e instanceof DataAccessException && e.getMessage().contains(String.format("\"%s\" already exists", table.getName()))) {
            return Mono.empty();
        }
        LOGGER.error("Error while creating table: {}", (Object)table.getName(), (Object)e);
        return Mono.error((Throwable)e);
    }

    private Mono<Void> alterTableIfNeeded(PostgresTable table, Connection connection) {
        return this.executeAdditionalAlterQueries(table, connection).then(this.enableRLSIfNeeded(table, connection));
    }

    private Mono<Void> executeAdditionalAlterQueries(PostgresTable table, Connection connection) {
        return Flux.fromIterable(table.getAdditionalAlterQueries()).filter(additionalAlterQuery -> additionalAlterQuery.shouldBeApplied(this.rowLevelSecurity)).map(PostgresTable.AdditionalAlterQuery::getQuery).concatMap(alterSQLQuery -> Mono.just((Object)connection).flatMapMany(pgConnection -> pgConnection.createStatement(alterSQLQuery).execute()).flatMap(Result::getRowsUpdated).then().onErrorResume(e -> {
            if (e.getMessage().contains("already exists")) {
                return Mono.empty();
            }
            LOGGER.error("Error while executing ALTER query for table {}", (Object)table.getName(), e);
            return Mono.error((Throwable)e);
        })).then();
    }

    private Mono<Void> enableRLSIfNeeded(PostgresTable table, Connection connection) {
        if (this.rowLevelSecurity.isRowLevelSecurityEnabled() && table.supportsRowLevelSecurity()) {
            return this.alterTableEnableRLS(table, connection);
        }
        return Mono.empty();
    }

    private Mono<Void> alterTableEnableRLS(PostgresTable table, Connection connection) {
        return Mono.just((Object)connection).flatMapMany(pgConnection -> pgConnection.createStatement(this.rowLevelSecurityAlterStatement(table.getName())).execute()).flatMap(Result::getRowsUpdated).then();
    }

    private String rowLevelSecurityAlterStatement(String tableName) {
        String policyName = "domain_" + tableName + "_policy";
        return "set app.current_domain = ''; alter table " + tableName + " add column if not exists domain varchar(255) not null default current_setting('app.current_domain')::text ;do $$ \nbegin \n    if not  exists( select policyname from pg_policies where policyname = '" + policyName + "') then \n        execute 'alter table " + tableName + " enable row level security; alter table " + tableName + " force row level security; create policy " + policyName + " on " + tableName + " using (domain = current_setting(''app.current_domain'')::text)';\n    end if;\nend $$;";
    }

    public Mono<Void> truncate() {
        return Mono.usingWhen(this.postgresExecutor.connectionFactory().getConnection(), connection -> this.postgresExecutor.dslContext((Connection)connection).flatMap(dsl -> Flux.fromIterable(this.module.tables()).flatMap(table -> Mono.from((Publisher)dsl.truncateTable(table.getName())).doOnSuccess(any -> LOGGER.info("Table {} truncated", (Object)table.getName())).doOnError(e -> LOGGER.error("Error while truncating table {}", (Object)table.getName(), e))).then()), connection -> this.postgresExecutor.connectionFactory().closeConnection((Connection)connection));
    }

    public Mono<Void> initializeTableIndexes() {
        return Mono.usingWhen(this.postgresExecutor.connectionFactory().getConnection(), connection -> this.postgresExecutor.dslContext((Connection)connection).flatMapMany(dsl -> this.listExistIndexes((DSLContext)dsl).flatMapMany(existIndexes -> Flux.fromIterable(this.module.tableIndexes()).filter(index -> !existIndexes.contains(index.getName())).flatMap(index -> this.createTableIndex((PostgresIndex)index, (DSLContext)dsl)))).then(), connection -> this.postgresExecutor.connectionFactory().closeConnection((Connection)connection));
    }

    private Mono<List<String>> listExistIndexes(DSLContext dslContext) {
        return Mono.just((Object)dslContext).flatMapMany(dsl -> Flux.from((Publisher)dsl.select((SelectField)DSL.field((String)"indexname")).from("pg_indexes").where(DSL.field((String)"schemaname").eq((Object)DSL.currentSchema())))).map(r -> (String)r.get(0, String.class)).collectList();
    }

    private Mono<Void> createTableIndex(PostgresIndex index, DSLContext dsl) {
        return Mono.from((Publisher)((Publisher)index.getCreateIndexStepFunction().apply(dsl))).doOnSuccess(any -> LOGGER.info("Index {} created", (Object)index.getName())).onErrorResume(e -> this.handleIndexCreationException(index, (Throwable)e)).then();
    }

    private Mono<? extends Integer> handleIndexCreationException(PostgresIndex index, Throwable e) {
        if (e instanceof DataAccessException && e.getMessage().contains(String.format("\"%s\" already exists", index.getName()))) {
            return Mono.empty();
        }
        LOGGER.error("Error while creating index {}", (Object)index.getName(), (Object)e);
        return Mono.error((Throwable)e);
    }
}

