/*
 * Decompiled with CFR 0.152.
 */
package org.apache.brooklyn.rest.security.provider;

import com.google.common.base.CharMatcher;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Lists;
import java.util.Arrays;
import java.util.Hashtable;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.InitialDirContext;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import org.apache.brooklyn.api.mgmt.ManagementContext;
import org.apache.brooklyn.config.ConfigKey;
import org.apache.brooklyn.config.StringConfigMap;
import org.apache.brooklyn.core.config.ConfigPredicates;
import org.apache.brooklyn.rest.BrooklynWebConfig;
import org.apache.brooklyn.rest.security.provider.AbstractSecurityProvider;
import org.apache.brooklyn.rest.security.provider.SecurityProvider;
import org.apache.brooklyn.util.exceptions.Exceptions;
import org.apache.brooklyn.util.text.Strings;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LdapSecurityProvider
extends AbstractSecurityProvider
implements SecurityProvider {
    public static final Logger LOG = LoggerFactory.getLogger(LdapSecurityProvider.class);
    public static final String LDAP_CONTEXT_FACTORY = "com.sun.jndi.ldap.LdapCtxFactory";
    public static final String LDAP_USER_GROUPS_ORIGIN = LdapSecurityProvider.class.getName();
    private final String ldapUrl;
    private final String defaultLdapRealm;
    private final String organizationUnit;
    private final String userNameRegex;
    private final String domainRegex;
    private boolean logUserLoginAttempt;
    private boolean fetchUserGroups = false;
    private List<String> validGroups;
    static boolean triedLoading = false;

    public LdapSecurityProvider(ManagementContext mgmt) {
        StringConfigMap properties = mgmt.getConfig();
        this.ldapUrl = (String)properties.getConfig(BrooklynWebConfig.LDAP_URL);
        Strings.checkNonEmpty((CharSequence)this.ldapUrl, (String)("LDAP security provider configuration missing required property " + BrooklynWebConfig.LDAP_URL));
        this.fetchUserGroups = (Boolean)properties.getConfig(BrooklynWebConfig.LDAP_FETCH_USER_GROUPS);
        this.logUserLoginAttempt = (Boolean)properties.getConfig(BrooklynWebConfig.LDAP_LOGIN_INFO_LOG);
        this.domainRegex = (String)properties.getConfig(BrooklynWebConfig.LDAP_DOMAIN_REGEX);
        this.userNameRegex = (String)properties.getConfig(BrooklynWebConfig.LDAP_USERNAME_REGEX);
        List ldapGroupsPrefixes = (List)properties.getConfig(BrooklynWebConfig.GROUP_CONFIG_KEY_NAME);
        this.validGroups = this.fetchUserGroups && !ldapGroupsPrefixes.isEmpty() ? this.getConfiguredGroups(properties, ldapGroupsPrefixes) : ImmutableList.of();
        String realmConfig = (String)properties.getConfig(BrooklynWebConfig.LDAP_REALM);
        this.defaultLdapRealm = Strings.isNonBlank((CharSequence)realmConfig) ? CharMatcher.isNot((char)'\"').retainFrom((CharSequence)realmConfig) : "";
        if (Strings.isBlank((CharSequence)((CharSequence)properties.getConfig(BrooklynWebConfig.LDAP_OU)))) {
            LOG.info("Setting LDAP ou attribute to: Users");
            this.organizationUnit = "Users";
        } else {
            this.organizationUnit = CharMatcher.isNot((char)'\"').retainFrom((CharSequence)properties.getConfig(BrooklynWebConfig.LDAP_OU));
        }
    }

    public LdapSecurityProvider(String ldapUrl, String ldapRealm, String organizationUnit) {
        this.ldapUrl = ldapUrl;
        this.defaultLdapRealm = ldapRealm;
        this.organizationUnit = organizationUnit;
        this.userNameRegex = "";
        this.domainRegex = "";
    }

    @Override
    public boolean authenticate(HttpServletRequest request, Supplier<HttpSession> sessionSupplierOnSuccess, String user, String pass) throws SecurityProvider.SecurityProviderDeniedAuthentication {
        if (user == null) {
            return false;
        }
        String ldapRegex = this.getLdapRegexPattern();
        if (Strings.isNonEmpty((CharSequence)ldapRegex) && !user.matches(ldapRegex)) {
            LOG.debug("Rejecting authenticating attempt for user `{}` due to userNameRegex configuration: {}", (Object)user, (Object)ldapRegex);
            return false;
        }
        LdapSecurityProvider.checkCanLoad();
        if (Strings.isBlank((CharSequence)pass)) {
            return false;
        }
        this.addToInfoLog("Login attempt with " + user);
        try {
            Hashtable<String, String> env = new Hashtable<String, String>();
            env.put("java.naming.factory.initial", LDAP_CONTEXT_FACTORY);
            env.put("java.naming.provider.url", this.ldapUrl);
            env.put("java.naming.security.authentication", "simple");
            env.put("java.naming.security.principal", this.getSecurityPrincipal(user));
            env.put("java.naming.security.credentials", pass);
            InitialDirContext ctx = new InitialDirContext(env);
            if (this.fetchUserGroups) {
                sessionSupplierOnSuccess.get().setAttribute("brooklyn.entitlements.user.groups.origin", (Object)LDAP_USER_GROUPS_ORIGIN);
                List<String> userGroups = this.getUserGroups(user, ctx);
                if (userGroups.isEmpty()) {
                    this.addToInfoLog("Unsuccessful for " + user);
                    LOG.trace("User {} is not member of any group", (Object)user);
                    return false;
                }
                sessionSupplierOnSuccess.get().setAttribute("brooklyn.entitlements.user.groups", userGroups);
                this.addToInfoLog("Authentication successful for user " + user + ", in relevant LDAP groups " + userGroups);
            } else {
                this.addToInfoLog("Successful for " + user);
            }
            return this.allow(sessionSupplierOnSuccess.get(), user);
        }
        catch (NamingException e) {
            this.addToInfoLog("Unsuccessful for " + user);
            return false;
        }
    }

    private String getLdapRegexPattern() {
        if (Strings.isBlank((CharSequence)this.domainRegex) && Strings.isBlank((CharSequence)this.userNameRegex)) {
            return "";
        }
        if (Strings.isBlank((CharSequence)this.domainRegex)) {
            return this.userNameRegex;
        }
        return Strings.join((Object[])new String[]{this.domainRegex, Strings.isNonBlank((CharSequence)this.userNameRegex) ? this.userNameRegex : ".*"}, (String)"\\\\");
    }

    private List<String> getConfiguredGroups(StringConfigMap properties, List<String> prefixes) {
        ImmutableList.Builder configuredGroupsBuilder = ImmutableList.builder();
        for (String prefix : prefixes) {
            StringConfigMap roles = properties.submap(ConfigPredicates.nameStartsWith((String)(prefix + ".")));
            for (Map.Entry entry : roles.getAllConfigLocalRaw().entrySet()) {
                configuredGroupsBuilder.add((Object)Strings.removeFromStart((String)((ConfigKey)entry.getKey()).getName(), (String)(prefix + ".")));
            }
        }
        return configuredGroupsBuilder.build();
    }

    private void addToInfoLog(String s) {
        if (this.logUserLoginAttempt) {
            LOG.info(s);
        }
    }

    private List<String> getUserGroups(String user, DirContext ctx) throws NamingException {
        ImmutableList.Builder groupsListBuilder = ImmutableList.builder();
        SearchControls ctls = new SearchControls();
        ctls.setReturningAttributes(new String[]{"memberOf"});
        NamingEnumeration<SearchResult> answer = ctx.search(this.buildUserContainer(), "(&(objectclass=user)(sAMAccountName=" + this.getAccountName(user) + "))", ctls);
        while (answer.hasMore()) {
            SearchResult rslt = answer.next();
            Attributes attrs = rslt.getAttributes();
            Attribute memberOf = attrs.get("memberOf");
            if (memberOf == null) continue;
            NamingEnumeration<?> groups = memberOf.getAll();
            while (groups.hasMore()) {
                groupsListBuilder.add((Object)this.getGroupName(groups.next().toString()));
            }
        }
        ImmutableList ldapGroups = groupsListBuilder.build();
        if (LOG.isTraceEnabled()) {
            LOG.trace("LDAP groups for {}: {}", (Object)user, (Object)ldapGroups);
        }
        return ldapGroups.stream().filter(ldapGroup -> this.validGroups.contains(ldapGroup)).collect(Collectors.toList());
    }

    private String buildUserContainer() {
        StringBuilder userContainerBuilder = new StringBuilder();
        for (String s : this.organizationUnit.split("\\.")) {
            userContainerBuilder.append("OU=").append(s).append(",");
        }
        for (String s : this.defaultLdapRealm.split("\\.")) {
            userContainerBuilder.append("DC=").append(s).append(",");
        }
        return StringUtils.chop((String)userContainerBuilder.toString());
    }

    private String getAccountName(String user) {
        if (user.contains("\\")) {
            String[] split = user.split("\\\\");
            return split[split.length - 1];
        }
        return user;
    }

    private String getGroupName(String groupAttribute) {
        Pattern groupNamePatter = Pattern.compile("^CN=(?<groupName>[a-zA-Z0-9_-]+),*");
        Matcher m = groupNamePatter.matcher(groupAttribute);
        if (m.find()) {
            return m.group(1);
        }
        throw new IllegalStateException("Not valid group found in " + groupAttribute);
    }

    protected String getSecurityPrincipal(String user) throws NamingException {
        if (user.contains("@") || user.contains("\\")) {
            return user;
        }
        List domain = Lists.transform(Arrays.asList(this.defaultLdapRealm.split("\\.")), (Function)new Function<String, String>(){

            public String apply(String input) {
                return "dc=" + input;
            }
        });
        String dc = Joiner.on((String)",").join((Iterable)domain).toLowerCase();
        return "cn=" + user + ",ou=" + this.organizationUnit + "," + dc;
    }

    public static synchronized void checkCanLoad() {
        if (triedLoading) {
            return;
        }
        try {
            Class.forName(LDAP_CONTEXT_FACTORY);
            triedLoading = true;
        }
        catch (Throwable e) {
            throw Exceptions.propagate((Throwable)new ClassNotFoundException("Unable to load LDAP classes (com.sun.jndi.ldap.LdapCtxFactory) required for Brooklyn LDAP security provider"));
        }
    }

    @Override
    public boolean requiresUserPass() {
        return true;
    }
}

